﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       xcd ライブラリの KuinaMCU アップデートに関係する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>

namespace nn { namespace xcd {

/**
    * @brief       Kuina MCU のファームウェアバージョン情報です。
    */
struct KuinaVersionData
{
    uint8_t type;           //!< Kuina のデバイスタイプ　0x02:KOTETSU 0x01:Kuina
    uint8_t major;          //!< Kuina のメジャーバージョンです。
    uint8_t minor;          //!< Kuina のマイナーバージョンです。
    uint8_t micro;          //!< Kuina のマイクロバージョンです。
};

/**
    * @brief       充電グリップ用のKuinaFw 出荷状態
    */
const KuinaVersionData FirmwareVersionKuina_0200 = { 01, 0x01, 0x0a, 0x00 };

/**
    * @brief       Proコントローラ用KuinaDashFw 出荷状態
    */
const KuinaVersionData FirmwareVersionKuinaDash_0200 = { 02, 0x01, 0x0a, 0x00 };

/**
    * @brief       Kuina ファームウェアのバージョンを取得します。
    *
    * @details     Kuina ファームウェアのバージョンを取得します。
    *
    *
    *              正常にバージョンを取得できた後は、デバイスが切断するまでバージョンを取得できます。
    *
    * @param[out]  pOutMcuVersionData          バージョンデータ
    * @param[in]   index                       デバイスの識別子
    *
    * @retresult
    *      @handleresult{nn::ResultSuccess,                                処理に成功しました。}
    *      @handleresult{nn::xcd::ResultUsbHidDeviceDetached,              デバイスが接続されていません。}
    *      @handleresult{nn::xcd::ResultUsbHidFirmwareVersionNotAvailable, デバイスのバージョン取得が完了していません。}
    * @endresult
    *
    * @pre
    *  - XCD ライブラリは初期化済である必要があります。
    *  - @a pOutMcuVersionData != nullptr
    */
Result GetKuinaVersion(KuinaVersionData* pOutMcuVersionData, int index) NN_NOEXCEPT;

/**
    * @brief       Kuina ファームウェアのバージョン取得を要求します。
    *
    * @details     Kuina ファームウェアのバージョン取得要求を発行します。
    *              バージョンの取得には数フレームの時間が必要になります。
    *
    *              バージョンを取得するには、USB通信を無効にする必要があります。
    *              この関数を呼ぶ前に、 @ref ::nn::xcd::SetFullKeyUsbEnabled(false) を呼び出しておいてください。
    *
    * @param[in]   index                      デバイスの識別子
    *
    * @retresult
    *      @handleresult{nn::ResultSuccess,                        処理に成功しました。}
    *      @handleresult{nn::xcd::ResultUsbHidDeviceDetached,      デバイスが接続されていません。}
    * @endresult
    *
    * @pre
    *  - XCD ライブラリは初期化済である必要があります。
    */
Result RequestKuinaVersion(int index) NN_NOEXCEPT;

/**
    * @brief       Kuina を　DFU 状態に遷移させます
    *
    * @details     Kuina を　DFU 状態に遷移させます
    *              処理が成功すると、Kuina デバイスは DFU 状態のなり、USBHIDデバイスとしては認識されなくなります。
    *
    * @param[in]   index                      デバイスの識別子
    *
    * @retresult
    *      @handleresult{nn::ResultSuccess,                        処理に成功しました。}
    *      @handleresult{nn::xcd::ResultUsbHidDeviceDetached,      デバイスが接続されていません。}
    * @endresult
    *
    * @pre
    *  - XCD ライブラリは初期化済である必要があります。
*/
Result SetKuinaToFirmwareUpdateMode(int index) NN_NOEXCEPT;

}}
