﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   xcd ライブラリの モーションIRカメラに関係する型の宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_TimeSpan.h>
#include <nn/util/util_MathTypes.h>
#include "xcd_TeraFirmware.h"

namespace nn { namespace xcd {

/**
 * @brief    画像サイズの最大値
 */
const int IrImageWidthMax  = 320;
const int IrImageHeightMax = 240;
const int IrImageSizeMax   = IrImageWidthMax * IrImageHeightMax;

/**
 * @brief    Moment モードで取得できるブロック数
 */
const int IrMomentModeBlockCountWidth = 8;
const int IrMomentModeBlockCountHeight = 6;
const int IrMomentProcessorBlockCount = IrMomentModeBlockCountWidth * IrMomentModeBlockCountHeight;

/**
 * @brief    Moment モードで1度の転送で読み出し可能なブロック数の上限
 */
const int IrMomentProcessorStateCountMax = 1;

/**
 * @brief    Clustering モードで取得できるクラスタ数の最大
 */
const int IrClusteringProcessorObjectCountMax = 16;

/**
 * @brief    Clustering モードで1度の転送で読み出し可能なクラスタ数の上限
 */
const int IrClusteringProcessorStateCountMax = 3;

/**
 * @brief    Write できるレジスタの最大数
 */
const int IrWriteRegisterCountMax = 9;

/**
 * @brief    WriteEx できるレジスタの最大数
 */
const int IrWriteRegisterExCountMax = 17;

/**
 * @brief    Read できるレジスタの最大数
 */
const int IrReadRegisterCountMax = 256;

/**
 * @brief    TeraPlugin モード用の生データのサイズ
 */
const size_t IrTeraPluginProcessorRawDataSize = 300;

/**
 * @brief    TeraPlugin モードで書き込めるパラメータの最大サイズ
 */
const size_t IrTeraPluginParameterSizeMax = 16;

/**
 * @brief    Dpd モードの1フレームで取得されるクラスタ数の最大
 */
const int IrDpdProcessorObjectCountMax = 3;

/**
 * @brief    Dpd モードの1度の転送で読み出し可能なデータの最大数
 */
const int IrDpdProcessorStateCountMax = 6;

/**
 * @brief    モーションIRカメラの処理方式
 */
enum class IrProcessorType
{
    Reset = 0,             //!< リセットを実施
    Sleep,                 //!< Sleep モード
    Ready,                 //!< Ready モード
    Moment,                //!< Moment モード
    Dpd,                   //!< DPD モード
    Segmentation,          //!< Segmentation モード
    Clustering,            //!< Clustering モード
    ImageTransfer,         //!< ImageTransfer モード
    TeraPlugin,            //!< TeraPlugin モード (8~ を予約)
};

/**
 * @brief    レジスタ処理の成否結果
 */
enum class IrRegisterResult
{
    ReadOk = 0,             //!< 読み込み正常終了
    WriteOk,                //!< 書き込み正常終了
    ReadError,              //!< 読み込みエラー
    WriteError,             //!< 書き込みエラー
};

/**
 * @brief イメージ転送プロセッサで取得する画像フォーマットです。
 */
enum class IrImageTransferProcessorFormat
{
    ImageSize_320x240, //!< 8 ビットグレースケール QVGA (320x240) 画像です。
    ImageSize_160x120, //!< 8 ビットグレースケール QQVGA (160x120) 画像です。
    ImageSize_80x60,   //!< 8 ビットグレースケール QQQVGA (80x60) 画像です。
    ImageSize_40x30,   //!< 8 ビットグレースケール QQQQVGA (40x30) 画像です。
    ImageSize_20x15,   //!< 8 ビットグレースケール QQQQQVGA (20x15) 画像です。
};

/**
 * @brief イメージ転送プロセッサの転送プロトコルです。
 */
enum class IrImageTransferProtocol
{
    Normal,    //!< 通常版
    FastMode,  //!< 高速化版
};

/**
 * @brief 使用するコマンドの種類
 */
enum class IrCommandType
{
    Normal,    //!< 通常版
    Extension, //!< 高速化版
};

/**
 * @brief    モーションIRカメラが使用する Mcu のバージョン
 */
struct IrMcuVersion
{
    uint16_t major;
    uint16_t minor;

    uint32_t ToUint32() const NN_NOEXCEPT
    {
        return (static_cast<uint32_t>(major << 16) & 0xffff0000) |
               (static_cast<uint32_t>(minor) & 0x0000ffff);
    }

    /*!
     * @brief      値を比較します。
     */
    bool operator < (const nn::xcd::McuVersionData& value) const NN_NOEXCEPT
    {
        return this->ToUint32() < value.ToUint32();
    }
    /*!
     * @brief      値を比較します。
     */
    bool operator > (const nn::xcd::McuVersionData& value) const NN_NOEXCEPT
    {
        return this->ToUint32() > value.ToUint32();
    }
    /*!
     * @brief      値を比較します。
     */
    bool operator <= (const nn::xcd::McuVersionData& value) const NN_NOEXCEPT
    {
        return this->ToUint32() <= value.ToUint32();
    }
    /*!
     * @brief      値を比較します。
     */
    bool operator >= (const nn::xcd::McuVersionData& value) const NN_NOEXCEPT
    {
        return this->ToUint32() >= value.ToUint32();
    }
};

/**
 * @brief    Read するレジスタの設定
 */
struct IrReadRegisterSetting
{
    uint8_t  bankId;        //!< Bank 番号
    uint8_t  startAddress;  //!< 読み込み開始アドレス
    size_t   size;          //!< 読み込みサイズ
};

/**
 * @brief    Write するレジスタブロック
 */
struct IrWriteRegisterBlock
{
    uint8_t  bankId;        //!< Bank 番号
    uint8_t  address;       //!< 書き込みアドレス
    uint8_t  data;          //!< 書き込むデータ
};

/**
 * @brief    Write するレジスタの設定
 */
struct IrWriteRegisterSetting
{
    int                   registerCount;                          //!< 設定するレジスタ数（最大 9）
    IrWriteRegisterBlock  registerBlock[IrWriteRegisterCountMax]; //!< 書き込むレジスタデータ
};

/**
 * @brief    WriteEx するレジスタの設定
 */
struct IrWriteRegisterSettingEx
{
    bool                  fastModeFlag;                             //!< fast mode が有効かどうかのフラグ
    int                   registerCount;                            //!< 設定するレジスタ数（最大 17）
    IrWriteRegisterBlock  registerBlock[IrWriteRegisterExCountMax]; //!< 書き込むレジスタデータ
};

/**
 * @brief    ReadRegister の結果を表す構造体です。
 */
struct IrReadRegisterState
{
    IrRegisterResult      result;                           //!< RegisterRead の成否
    IrReadRegisterSetting regSetting;                       //!< Read 対象のレジスタの情報
    uint8_t               bankData[IrReadRegisterCountMax]; //!< レジスタデータ
};

/**
 * @brief    WriteRegister の結果を表す構造体です。
 */
struct IrWriteRegisterState
{
    IrRegisterResult result; //!< RegisterWrite の成否
};

/**
 * @brief    モード毎に共通に取得できるデータです。上位APIとバイナリ互換を取ります。
 */
struct IrCommonData
{
    IrProcessorType  irProcessorType;         //!< 現在の処理方式
    int              frameId;                 //!< フレーム ID
    int              averageIntensityOn;      //!< 画像輝度の平均値（LED ON）
    int              averageIntensityOff;     //!< 画像輝度の平均値（LED OFF)
    int              lightPixelOn;            //!< 閾値以上の輝度を持つピクセルの総数 (LED ON 時)
    int              lightPixelOff;           //!< 閾値以上の輝度を持つピクセルの総数 (LED OFF 時)
};

/**
 * @brief    ブロックの統計量を表す構造体です。 上位APIとバイナリ互換を取ります。
 */
struct IrMomentStatistic
{
    float averageIntensity;      //!< ブロックの平均輝度値を算出するための値です（0 以上 255 以下）。(上位でWOIサイズに合わせた補完をする必要あり)。
    ::nn::util::Float2 centroid; //!< ブロックの重心座標です（WOIで指定した領域内）。
};

/**
 * @brief    モーメントプロセッサの処理結果を表す構造体です。
 */
struct IrMomentProcessorState
{
    int samplingNumber;                                    //!< パケットの ID です。（0 以上 255 以下）。
    int diffTimeStampCount;                                //!< タイムスタンプ ID の前フレームとの差分です。
    IrMomentStatistic blocks[IrMomentProcessorBlockCount]; //!< ブロックごとの統計量です。
};

/**
 * @brief    矩形です。ピクセル単位の範囲を表す場合に使用します。上位APIとバイナリ互換を取ります。
 */
struct IrRect
{
    int16_t x;      //!< X 座標
    int16_t y;      //!< Y 座標
    int16_t width;  //!< 幅
    int16_t height; //!< 高
};

/**
 * @brief    クラスタリングプロセッサでオブジェクトを表す構造体です。上位APIとバイナリ互換を取ります。
 */
struct IrClusteringData
{
    float averageIntensity;      //!< オブジェクトの平均輝度値です（0 以上 255 以下）。
    ::nn::util::Float2 centroid; //!< オブジェクトの重心座標です（WOI で指定した領域内）。
    int32_t pixelCount;          //!< オブジェクトの面積です。
    IrRect bound;                //!< オブジェクトの外接矩形です（WOI で指定した領域内）。
};

/**
 * @brief    クラスタリングプロセッサの処理結果を表す構造体です。
 */
struct IrClusteringProcessorState
{
    int samplingNumber;                                            //!< パケットの ID です。（0 以上 255 以下）。
    int diffTimeStampCount;                                        //!< タイムスタンプ ID の前フレームとの差分です。
    int8_t objectCount;                                            //!< 検出したオブジェクトの数です（0 以上 ClusteringProcessorObjectCountMax 以下）。
    char reserved[7];
    IrClusteringData objects[IrClusteringProcessorObjectCountMax]; //!< 検出したオブジェクトのデータです。
};

/**
 * @brief    イメージ転送プロセッサの処理結果を表す構造体です。
 */
struct IrImageTransferProcessorState
{
    void* pImage;                 //!< 画像が格納されたバッファへのポインタです。
};

/**
 * @brief    TeraPlugin モード用のパラメータデータです。
 */
struct IrTeraPluginParameter
{
    bool isParameterEnabled;                        //!< パラメータが有効かどうかのフラグ
    size_t parameterSize;                           //!< パラメータのサイズ
    int8_t parameter[IrTeraPluginParameterSizeMax]; //!< パラメータのデータ
};

/**
 * @brief    TeraPlugin モード用の生データです。
 */
struct IrTeraPluginProcessorState
{
    int samplingNumber;                                  //!< パケットの ID です。（0 以上 255 以下）。
    int diffTimeStampCount;                              //!< タイムスタンプ ID の前フレームとの差分です。
    uint8_t rawData[IrTeraPluginProcessorRawDataSize]; //!< TeraPlugin モード用の生データです。
};

/**
 * @brief    Dpd プロセッサでオブジェクトを表す構造体です。上位APIとバイナリ互換を取ります。
 */
struct IrDpdData
{
    bool isDataValid;            //!< オブジェクトが有効かどうかを表すフラグです。
    char reserved[7];
    float averageIntensity;      //!< オブジェクトの平均輝度値です（0 以上 255 以下）。
    ::nn::util::Float2 centroid; //!< オブジェクトの重心座標です。
    int32_t pixelCount;          //!< オブジェクトの面積です。
    IrRect bound;                //!< オブジェクトの外接矩形です。
};

/**
 * @brief    Dpd プロセッサの処理結果を表す構造体です。
 */
struct IrDpdProcessorState
{
    int samplingNumber;                                            //!< パケットの ID です。（0 以上 255 以下）。
    int diffTimeStampCount;                                        //!< タイムスタンプ ID の前フレームとの差分です。
    IrDpdData objects[IrDpdProcessorObjectCountMax];               //!< 検出したオブジェクトのデータです。
};

}} // namespace nn::xcd

