﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       デバイスの列挙などに関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_BitTypes.h>
#include <nn/bluetooth/bluetooth_AddressTypes.h>
#include <nn/os/os_SystemEventTypes.h>

namespace nn { namespace xcd {

/**
 * @brief       デバイスへの接続処理がタイムアウトした際に通知を受け取るシステムイベントを登録します
 *
 * @details     デバイスへの接続処理を試みている場合に処理がタイムアウトで終了したことを通知します。
  *
 * @param[in]   pSystemEvent                  接続処理がタイムアウトしたことを受け取るためのシステムイベント
 *
 * @pre         - pSystemEvent != nullptr
 */
void SetConnectionTriggerTimeoutEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

/**
 * @brief      指定したアドレスのデバイスに対して Bluetooth 接続を試みます
 *
 * @details    指定されたアドレスのデバイスに対して Bluetooth 接続を試みます
 *             接続を試みるデバイスの Address を指定することで接続トリガーを送信することができます
 *             本体に登録されているデバイスの一覧は GetRegisteredDevices() で取得することができます。
 *             本 API が呼ばれた段階では接続処理が開始されません。
 *
 *             デバイスがみつからないず処理がタイムアウトした場合は、
 *             SetConnectionTriggerTimeoutEvent() で設定したイベントで通知を受け取ることができます。
 *
 * @param[in]  Address                        接続を試みる
 *
 * @return     実行結果を返します。
 * @retval     ResultSuccess                             処理に成功しました。
 * @retval     ResultBluetoothAlreadyConnected           指定されたデバイスはすでに接続済みです
 * @retval     ResultBluetoothDeviceNotRegistered        指定されたデバイスがデータベースに登録されていません
 * @retval     ResultBluetoothOtherTriggerEventOnProgreess 別の接続トリガーイベントが進行中です
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 */
Result SendConnectionTrigger(nn::bluetooth::Address address) NN_NOEXCEPT;

/**
 * @brief      Bluetooth 接続できるコントローラーの最大数を取得する
 *
 * @return     接続できるデバイスの数
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 */
int GetMaxBluetoothLinks() NN_NOEXCEPT;

}} // namespace nn::xcd
