﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       xcd ライブラリに関係する型の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/xcd/xcd_Input.h>
#include <nn/xcd/xcd_Result.h>

namespace nn { namespace xcd {

/**
 * @brief       BLE デバイスに対しての操作の種類です。
 */
enum BleDeviceOperationType : int32_t
{
    BleDeviceOperationType_PlayDeviceActivity,  //!< Activity の再生
    BleDeviceOperationType_WriteDatabaseEntry,  //!< Database Entry の書き込み
    BleDeviceOperationType_GetDatabaseIdentificationVersion,  //!< データベース識別バージョンの読み込み
    BleDeviceOperationType_SetDatabaseIdentificationVersion,  //!< データベース識別バージョンの書き込み
    BleDeviceOperationType_SetFrModeType,       //!< FrModeType の書き込み
    BleDeviceOperationType_ResetStepCount,      //!< StepCount の初期化
    BleDeviceOperationType_GetStepCount,        //!< StepCount の読み込み
    BleDeviceOperationType_EnableStep,          //!< Step 機能の設定
    BleDeviceOperationType_ReadApplicationSection,  //!< Application セクションの読み込み
    BleDeviceOperationType_WriteApplicationSection, //!< Application セクションの書き込み
    BleDeviceOperationType_ReadContentUniqueCode,       //!< UniqueCode の読み込み
    BleDeviceOperationType_SetContentUniqueCodeInvalid, //!< UniqueCode の無効化
    BleDeviceOperationType_SuspendFeature,   //!< 機能の一時停止
    BleDeviceOperationType_ReadPlayLog,      //!< Play Log の読み込み
    BleDeviceOperationType_ResetPlayLog,     //!< Play Log の初期化
};

/**
 * @brief       BLE デバイスに対しての操作の実行結果の種類です。
 */
enum BleDeviceOperationResultType : int32_t
{
    BleDeviceOperationResultType_Success,           //!< 成功
    BleDeviceOperationResultType_Busy,              //!< BLE デバイスがビジー状態
    BleDeviceOperationResultType_UniqueCodeInvalid, //!< UniqueCode が無効化されている
    BleDeviceOperationResultType_OperationNotSupported, //!< BLE デバイスが対応していない操作
    BleDeviceOperationResultType_ResponseTimeout,   //!< レスポンスのタイムアウト
};

/**
 * @brief       BLE デバイスのデータベースの種類です。
 */
enum BleDeviceDatabaseType : int32_t
{
    BleDeviceDatabaseType_Activity,   //!< Activity Database
    BleDeviceDatabaseType_Led,        //!< LED Database
    BleDeviceDatabaseType_WaveSmall,  //!< Wave Database
    BleDeviceDatabaseType_WaveMedium, //!< Wave Database
    BleDeviceDatabaseType_WaveLarge,  //!< Wave Database
};

/**
 * @brief       BLE デバイスに対しての Application Section の読み書きにおける 1回あたりの最大長です。
 */
const size_t BleDeviceApplicationSectionAccessUnitSizeMax = 256;

/**
 * @brief       BLE デバイスに格納された UniqueCode のサイズです。
 */
const size_t BleDeviceContentUniqueCodeSize = 16;

/**
 * @brief       BLE デバイスに格納された Content 識別子のサイズです。
 */
const size_t BleDeviceContentIdSize = 4;

/**
 * @brief       BLE デバイスに対しての操作の実行結果です。
 */
struct BleDeviceOperationResult
{
    BleDeviceOperationType type;         //!< 操作結果の種類
    BleDeviceOperationResultType result; //!< 実行結果
    /**
      * @brief 各操作固有の処理結果です
      */
    union IndividualInfo
    {
        struct ActivityIndex
        {
            uint16_t index;
        } activityIndex;
        struct FrModeType
        {
            uint32_t type;
        } frModeType;
        struct StepCount
        {
            uint32_t step;
        } stepCount;
        struct StepCounterState
        {
            bool isEnabled;
        } stepCounterState;
        struct InfoReadApplicationSection
        {
            int32_t address;
            int32_t size;
            Bit8 data[BleDeviceApplicationSectionAccessUnitSizeMax];
        } readApplicationSection;
        struct InfoWriteApplicationSection
        {
            int32_t address;
            int32_t size;
        } writeApplicationSection;
        struct ContentUniqueCode
        {
            bool isValid;
            Bit8 uniqueCode[BleDeviceContentUniqueCodeSize];
            Bit8 contentId[BleDeviceContentIdSize];
        } contentUniqueCode;
        struct InfoWriteDatabaseEntry
        {
            BleDeviceDatabaseType type;
            uint16_t index;
        } writeDatabaseEntry;
        struct DatabaseIdentificationVersion
        {
            int32_t version;
        } databaseIdentificationVersion;
        struct SuspendFeature
        {
            uint32_t flagSet;
        } suspendFeature;
        struct PlayLog
        {
            uint16_t index;
            int32_t size;
            Bit8 raw[8];
        } playLog;
        Bit8 raw[320];                    //!< 応答のデータ
    } individual;  //!< 各操作固有の処理結果です
};

/**
 * @brief       アクセスする対象の Database Entry の設定です。
 */
struct BleDeviceDatabaseEntryConfig
{
    BleDeviceDatabaseType type; //!< データベースの種類
    uint16_t index;             //!< Database Entry のインデックス
};

}} // namespace nn::xcd
