﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   SSIDを取り扱うクラスの宣言
 */


#pragma once

#include <cstring>
#include <nn/nn_Common.h>

namespace nn {
namespace wlan {

/*!
    @name       SSID
    @{
 */

/*!
    :category   SSID
    @brief SSID データを表すクラスです。
 */
class Ssid
{
public:
    static const size_t SsidLengthMax = 32; //!< SSID の最大サイズです。
    static const size_t SsidHexStringLengthMax = SsidLengthMax * 4 + 1; //!< SSID 文字列バッファの最大サイズです。

private:
    size_t  m_Length;
    Bit8 m_Ssid[SsidLengthMax];

public:

    /*!
       @brief     Ssid クラスのコンストラクタです。長さ 0 の SSID が生成されます。
     */
    Ssid() NN_NOEXCEPT;

    /*!
       @brief     Ssid クラスのコンストラクタです。@n
       入力したssidの長さがSsidLengthMaxを超える場合、超えた分は無視され、SsidLengthMaxまでがssidとしてセットされます。
       @param[in] pSsid    セットする SSID のバイト配列。
       @param[in] length   セットする SSID のバイト配列の長さ。
     */
    Ssid(const Bit8* pSsid, size_t length) NN_NOEXCEPT;

    /*!
       @brief     Ssid クラスのコンストラクタです。@n
       文字列をセットすることで初期化します。NULL 終端文字で読み込みを停止するため、終端文字自体をSSIDとして取り込めません。@n
       入力したssidの長さがSsidLengthMaxを超える場合、超えた分は無視され、SsidLengthMaxまでがssidとしてセットされます。
       @param[in] pSsidString    セットする SSID の文字列。
     */
    explicit Ssid(const char* pSsidString) NN_NOEXCEPT;

    /*!
       @brief     Ssid クラスのデストラクタです。
     */
    ~Ssid() NN_NOEXCEPT
    {
    }


    //-----------------------------------------------------
    // 入力
    //-----------------------------------------------------

    /*!
       @brief     バイト配列による SSID のデータセットを行います。
       @param[in] pSsid    セットする SSID のバイト配列。
       @param[in] length   セットする SSID のバイト配列の長さ。
       @return    lengthがSsidLengthMaxを超えている場合falseが返ります。正しくセットされた場合、trueが借ります。
     */
    bool Set(const Bit8* pSsid, size_t length) NN_NOEXCEPT;

    /*!
       @brief     文字列による SSID のデータセットを行います。@n
       NULL 終端文字で読み込みを停止するため、終端文字自体をSSIDとして取り込めません
       @param[in] pSsidString    セットする SSID の文字列。
       @return    文字長がSsidLengthMaxを超えている場合falseが返ります。正しくセットされた場合、trueが借ります。
     */
    bool Set(const char* pSsidString) NN_NOEXCEPT;


    //-----------------------------------------------------
    // 出力
    //-----------------------------------------------------

    /*!
       @brief     SSID の長さを取得します。
       @return    SSID の長さ。
     */
    inline size_t GetLength() const NN_NOEXCEPT
    {
        return m_Length;
    }

    /*!
       @brief     SSID データ列へのポインタを取得します。
       @return    SSID データ列へのポインタ。
     */
    const Bit8* GetSsidData() const NN_NOEXCEPT
    {
        return m_Ssid;
    }

    /*!
       @brief     指定バッファに SSID 文字列をコピーします。@n
       制御文字をASCII表現(16進の文字列)へと編集して書き込みするため、画面描画以外の用途には使用しないでください。
       @param[out] outputSsidString    SSID 文字列をコピーするバッファポインタ。SsidHexStringLengthMax Byteのバッファが必要です。
       @return    SSID 文字列へのポインタ。outputSsidString 自身の先頭アドレスが返ります。
     */
    char* GetHexString(char outputSsidString[SsidHexStringLengthMax]) const NN_NOEXCEPT;

    //-----------------------------------------------------
    // 比較
    //-----------------------------------------------------
    // 比較演算子のオーバーロード
    /*!
       @brief     SSID 同士の比較をし、一致するかどうかを判定します。
       @param[in] ssid    比較対象の SSID です。
       @return    SSID が完全一致した場合、true を返し、そうでない場合に false を返します。
     */
    bool operator == (const Ssid& ssid) const NN_NOEXCEPT
    {
        if( m_Length == ssid.m_Length )
        {
            if( std::memcmp(&m_Ssid[0], &ssid.m_Ssid[0], m_Length) == 0 )
            {
                return true;
            }
            else
            {
                return false;
            }
        }
        else
        {
            return false;
        }
    }

    /*!
       @brief     SSID 同士の比較をし、一致しないかどうかを判定します。
       @param[in] ssid    比較対象の SSID です。
       @return    SSID が完全一致した場合、false を返し、そうでない場合に true を返します。
     */
    bool operator != (const Ssid& ssid) const NN_NOEXCEPT
    {
        return !( *this == ssid );
    }
};

/*!
    @}
 */

} // end of namespace wlan
} // end of namespace nn

