﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/wlan/wlan_Common.h>
#include <nn/wlan/wlan_Types.h>
#include <nn/wlan/wlan_Ssid.h>
#include <nn/wlan/wlan_MacAddress.h>
#include <nn/wlan/wlan_BssDescriptionReader.h>

namespace nn {
namespace wlan {

/*!
    @name       スキャン結果解析
    @{
 */

/*!
    :category   スキャン結果解析
    @class ScanResultReaderBase
    @brief Scan結果 解析クラス
 */
class ScanResultReaderBase
{
public:
    /*!
       @brief     ScanResultReaderBase クラスのコンストラクタです。
       @param[in] pBuffer   解析するスキャンバッファのポインタ。
       @param[in] pNext     スキャンされたデータの次の参照位置(デフォルトは NULL )。
     */
    ScanResultReaderBase(const void* pBuffer, Bit8* pNext = NULL) NN_NOEXCEPT;

    /*!
       @brief     ScanResultReaderBase クラスのコンストラクタです。BSS 情報のない空のリストを作成します。
       @param[in] pBuffer       スキャンバッファのポインタ。
       @param[in] bufferSize    バッファのサイズ。
     */
    ScanResultReaderBase(const void* pBuffer, size_t bufferSize) NN_NOEXCEPT;

    /*!
       @brief     ScanResultReaderBase クラスのデストラクタです。
     */
    virtual ~ScanResultReaderBase() NN_NOEXCEPT {}

    /*!
       @brief  スキャンされた BSS 情報の数を取得します。
       @return スキャンされた BSS 情報の数を返します。
     */
    uint32_t GetCount() const NN_NOEXCEPT;

    /*!
       @brief  スキャンされた結果全体のサイズを取得します。
       @return スキャンされた結果全体のサイズを返します。
     */
    size_t GetTotalResultLength() const NN_NOEXCEPT;

    /*!
       @brief  スキャンバッファのサイズを取得します。
       @return スキャンバッファのサイズを返します。
     */
    size_t GetBufferLength() const NN_NOEXCEPT;

    /*!
       @brief スキャン結果を配列で取得します。
       @param[out]   ppBssDescriptions   スキャンされた BSS 情報を格納する配列。
       @param[in]    count   ppBssDescriptions 配列の要素数。
       @return       スキャン結果配列に格納された BSS 情報の数。
     */
    uint32_t GetDescriptions(const BssDescription* ppBssDescriptions[], uint32_t count) const NN_NOEXCEPT;

    /*!
       @brief スキャン結果の末尾に BSS 情報をコピーします。
       @param[in]    pBssDescriptions   コピーする BSS 情報

       @return       コピーの結果を返します。
       @retval       true   成功しました。
       @retval       false  失敗しました。バッファが足りません。
     */
    bool AddDescription(const BssDescription* pBssDescription) NN_NOEXCEPT;

protected:
    const Bit8* ProcessBssPointer() NN_NOEXCEPT;
    bool IsOverCapacity(const BssDescription* pBssDescription) NN_NOEXCEPT;
    size_t GetBssDescriptionSize(const BssDescription* pBssDescription) NN_NOEXCEPT;

    const void* m_pHead;
    Bit8*       m_pNext;
};


/*!
    :category   スキャン結果解析
    @class BeaconScanResultReader
    @brief Beacon Scan結果 解析クラス
 */
class BeaconScanResultReader : public ScanResultReaderBase
{
public:
    /*!
       @brief     BeaconScanResultReader クラスのコンストラクタです。
       @param[in] pBuffer   解析するスキャンバッファのポインタ。
     */
    explicit BeaconScanResultReader(const void* pBuffer) NN_NOEXCEPT
                    : ScanResultReaderBase(pBuffer)
    {}

    /*!
       @brief     BeaconScanResultReader クラスのデストラクタです。
     */
    virtual ~BeaconScanResultReader() NN_NOEXCEPT {}

    /*!
       @brief  先頭に格納されている BSS 情報を @ref BeaconDescriptionReader 形式で取得します。
               @ref GetNextDescription での参照位置は、先頭にリセットされます。
       @return 先頭に格納されている BSS 情報。
     */
    const BeaconDescriptionReader GetFirstDescription() NN_NOEXCEPT;

    /*!
       @brief  次に格納されている BSS 情報を @ref BeaconDescriptionReader 形式で取得します。
               コールする度に、次に格納されている BSS 情報を取得し、BSS 情報の参照位置を1つずつ進めていきます。
               もう格納されているデータがない場合は、空のオブジェクトが返ります。
       @return 次に参照する BSS 情報。
     */
    const BeaconDescriptionReader GetNextDescription() NN_NOEXCEPT;


};

/*!
    @}
 */

} // end of namespace wlan
} // end of namespace nn
