﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os/os_SystemEvent.h>
#include <nn/mbuf/mbuf_Definitions.h>

#include "wlan_Common.h"
#include "wlan_MacAddress.h"
#include "wlan_Ssid.h"
#include "wlan_StateDefinition.h"
#include "wlan_Types.h"

namespace nn {
namespace wlan {

/*!
   @name       ローカル通信 関連
   @{
 */


//--------------------------------------------------------------------------
/**
 *    @brief        WLANライブラリのローカル通信のサービスを開始します。この初期化を行っていないと、ローカル通信に関する API は使用できません。本関数は必ず成功します。@n
 *    既に本関数を呼んでいた場合、何も行わず成功が返ります。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess    初期化に成功。
 */
nn::Result InitializeLocalManager() NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
 *    @brief        WLANライブラリのローカル通信のサービスを終了します。本関数は必ず成功します。
 *    @ref InitializeLocalManger を呼んでいない状態で呼んだ場合、何もせず成功が返ります。(developビルドの場合アサートします)
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess    終了処理に成功。
 */
nn::Result FinalizeLocalManager() NN_NOEXCEPT;


namespace Local {

//--------------------------------------------------------------------------
/**
 *    @brief        ローカル通信のMASTER モードをオープンします。
 *    @details      実行可能ステート：WlanState_Ready @n
 *                               WlanState_MasterIdle (既に目的ステートにいるため何も起こりません)
 *
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         状態遷移に成功。
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。
 */
nn::Result OpenMasterMode() NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
 *    @brief        ローカル通信のMASTER モードをクローズします。
 *    @details      実行可能ステート：WlanState_Ready (既に目的ステートにいるため何も起こりません)@n
 *                               WlanState_MasterIdle
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         状態遷移に成功。
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。
 */
nn::Result CloseMasterMode() NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
 *    @brief        ローカル通信のCLIENT モードをオープンします。
 *    @details      実行可能ステート：WlanState_Ready @n
 *                               WlanState_LocalClientIdle (既に目的ステートにいるため何も起こりません)
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         状態遷移に成功。
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。
 */
nn::Result OpenClientMode() NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
 *    @brief        ローカル通信のCLIENT モードをクローズします。
 *    @details      実行可能ステート：WlanState_Ready (既に目的ステートにいるため何も起こりません) @n
 *                               WlanState_LocalClientIdle
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         状態遷移に成功。
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。
 */
nn::Result CloseClientMode() NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
 *    @brief        ローカル通信のSPECTATOR モードをオープンします。
 *    @details      実行可能ステート：WlanState_Ready @n
 *                               WlanState_LocalSpectatorIdle (既に目的ステートにいるため何も起こりません)
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         状態遷移に成功。
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。
 */
nn::Result OpenSpectatorMode() NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
 *    @brief        ローカル通信のSPECTATOR モードをクローズします。
 *    @details      実行可能ステート：WlanState_Ready (既に目的ステートにいるため何も起こりません) @n
 *                               WlanState_LocalSpectatorIdle
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         状態遷移に成功。
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。
 */
nn::Result CloseSpectatorMode() NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
 *    @brief        ローカル通信のLCS用MASTER モードをオープンします。@n
 *    LCSモードは、無線低層パラメータが通常のLocalMasterと異なるだけで、Scanや、CreateBssなどOpen/Close以外の@n
 *    機能の使い方は変わりません。@n
 *    OpenLcsMasterMode() -> CreateBss()のような順番でAPIを呼んでください。
 *    @details      実行可能ステート：WlanState_Ready @n
 *                               WlanState_LcsMasterIdle (既に目的ステートにいるため何も起こりません)
 *
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         状態遷移に成功。
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。
 */
nn::Result OpenLcsMasterMode() NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
 *    @brief        ローカル通信のLCS用MASTER モードをクローズします。@n
 *    @ref CloseMasterMode()ではクローズ出来ません。
 *    @details      実行可能ステート：WlanState_Ready (既に目的ステートにいるため何も起こりません)@n
 *                               WlanState_LcsMasterIdle
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         状態遷移に成功。
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。
 */
nn::Result CloseLcsMasterMode() NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
 *    @brief        ローカル通信のLCS用CLIENT モードをオープンします。
 *    @details      実行可能ステート：WlanState_Ready @n
 *                               WlanState_LocalLcsClientIdle (既に目的ステートにいるため何も起こりません)
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         状態遷移に成功。
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。
 */
nn::Result OpenLcsClientMode() NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
 *    @brief        ローカル通信のLCS用CLIENT モードをクローズします。@n
 *    @ref CloseClientMode()ではクローズ出来ません。
 *    @details      実行可能ステート：WlanState_Ready (既に目的ステートにいるため何も起こりません) @n
 *                               WlanState_LocalLcsClientIdle
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         状態遷移に成功。
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。
 */
nn::Result CloseLcsClientMode() NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
 *    @brief        BSSを作成します。ローカル通信のMASTERモード動作時のみ実行可能です。@n
 *    MasterBssParameters.channelに-1を指定した場合、BSSの作成に最適なチャンネルが自動的に選ばれます。@n
 *    どのチャンネルが選ばれたかは、@ref GetConnectionStatus 関数で取得したConnectionStatus.channelから確認出来ます。@n
 *    ConnectionEventはシグナルされません。本関数を抜けた時点でBSSの作成は完了していますので、そのタイミングで@ref GetConnectionStatus 関数を呼び出してください。
 *    @details      実行可能ステート：WlanState_LocalMasterIdle
 *
 *    @param[in]    bssParameters         BSS作成時に使用されるパラメータです。 @ref MasterBssParameters参照
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         BSS作成に成功。
 *    @retval       ResultInvalidArgument 不正な引数が渡されました。（developビルドの場合、アサートします）
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。
 */
nn::Result CreateBss(const MasterBssParameters& bssParameters) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
 *    @brief        BSSを破棄します。ローカル通信のMASTERモードかつBSS作成済みの場合のみ実行可能です。
 *    @details      実行可能ステート：WlanState_LocalMasterIdle (何もせず成功が返ります)@n
 *                               WlanState_LocalMaster
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         BSS破棄に成功。
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。
 */
nn::Result DestroyBss() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        無線ノードのスキャンを開始します。スキャンが完了するまでブロックされます。
 *                  pOutScanBufferに入る分だけスキャン結果をコピーしますので、十分に大きなサイズを与えることを推奨します（50KB以上）。@n
 *                  なお、入りきらなかった分は破棄されます。また、その場合でも結果はResultSuccessが返ります。@n
 *                  途中で @ref StopScan を呼んだ場合、それまでに取得したスキャン結果が入ります。@n
 *                  使用可能なチャンネル以外のチャンネルを指定した場合、ResultInvalidArgumentが返ります。@n
 *                  指定可能なチャンネルは@ref GetAllowedChannel で取得出来ます。
 *    @details      実行可能ステート：WlanState_LocalMasterIdle @n
 *                               WlanState_LocalMasterBss @n
 *                               WlanState_LocalClientIdle @n
 *                               WlanState_LocalClient @n
 *                               WlanState_LocalSpectatorIdle @n
 *                               WlanState_LocalSpectator @n
 *
 *    @param[in]    pOutScanBuffer     スキャン結果の格納先バッファです。
 *    @param[in]    size               スキャン結果格納先バッファのサイズです。@n
 *    @ref ScanResultHeader がバッファの先頭に格納されますので、必ずこれより大きなサイズのバッファを用意して下さい。@n
 *    電波環境にも依りますが、50KB以上を推奨します。
 *    @param[in]    param              スキャンパラメータです。詳細は @ref ScanParameters をご覧ください。
 *    @param[in]    pIeInfo            スキャン結果に対しIEマッチングをかけます。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         スキャンに成功。本関数が返ってきた時点でスキャンも完了しています。
 *    @retval       ResultInvalidArgument 不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。
 */
nn::Result StartScan(void* pOutScanBuffer, size_t size, const ScanParameters& param, const ScanIeMatchInfo *pIeInfo = NULL) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief   無線ノードのスキャンを中止します。@ref StartScan を呼んでいた場合は、スキャンが中止され、@ref StartScan がResultSuccessを返し、@n
 *    それまでに取得したスキャン結果は スキャンバッファ に入ります。
 *    @details      実行可能ステート：WlanState_LocalMasterIdleScan @n
 *                               WlanState_LocalMasterBssScan @n
 *                               WlanState_LocalClientIdleScan @n
 *                               WlanState_LocalClientScan @n
 *                               WlanState_LocalSpectatorIdleScan @n
 *                               WlanState_LocalSpectatorScan @n
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         Scanの中止を発行。
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。（developビルドの場合、アサートします）
 */
nn::Result StopScan() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief     Masterへの接続を開始します。Clientモード専用関数です。@n
 *    本関数では、接続試行を開始して、その行為が完了するまでを行います。行為が正常に完了したら本関数は成功を返します。@n
 *    接続が成功したことを意味するものではないので注意して下さい。@n
 *    実際に接続が成功したか失敗したかは @ref GetConnectionStatus で確認して下さい。@n
 *    行為が完了するまで本関数は抜けませんので、完了前にキャンセルをしたい場合は、別スレッドから @ref CancelConnectをコールしてください。@n
 *    本関数を抜けた時点で、@ref GetConnectionEvent で取得したイベントはシグナルされています。@n
 *    接続先チャンネルを指定している場合は、そのチャンネルに対してのみ接続試行を行いますので、行為完了までの時間が短縮されます。@n
 *    スキャン中に本関数を呼ぶと失敗(ResultInvalidState)が返ります。@n
 *    なお、内部ではActiveScanが行われます。これにより、正しいSSIDが指定されていれば、ステルスのMasterへも接続が可能です。
 *    @details      実行可能ステート：WlanState_LocalClientIdle @n
 *
 *    @param[in]    ssid               接続先アクセスポイントのSSIDです。
 *    @param[in]    bssid              接続先アクセスポイントのBSSIDです。相手先のBSSIDが分かっている場合は入力して下さい。不明の場合は、BroadcastMACアドレス(FF:FF:FF:FF:FF:FF)を入力して下さい。
 *    @param[in]    channel            接続先アクセスポイントのチャンネル番号です。@ref GetAllowedChannels で許可されているチャンネルの範囲で指定します。不明の場合は-1を入力して下さい。-1を指定した場合は、利用可能なすべてのチャンネルに対し接続を試みます。
 *    @param[in]    security           接続先アクセスポイントのセキュリティ設定です。詳細は @ref Security 構造体を参照してください。ローカル通信では、SecurityMode_OpenまたはSecurityMode_StaticAesのみ指定可能です。
 *    @param[in]    autoKeepAlive      接続先アクセスポイントへ自動的にキープアライブフレーム（NULLフレーム）を送るかどうか設定します。trueにした場合、55秒に1度NULLフレームが自動的に飛びます。(未実装です)
 *    @param[in]    indication         接続確立後のBeacon受信通知の有無です。(未実装です)※廃止予定
 *    @param[in]    beaconLostTimeout  接続先アクセスポイントからのBeaconを何秒間受け取れなかったら自動的に切断するかの時間です。単位は秒です。デフォルトは10秒に設定されます。1～30秒の範囲で指定して下さい。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         接続試行が正常に完了しました。（接続成功を意味するわけではありません）
 *    @retval       ResultInvalidArgument 不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。
 */
nn::Result Connect(const Ssid& ssid, const MacAddress& bssid, int16_t channel, const Security& security,
        bool autoKeepAlive, BeaconIndication indication, int beaconLostTimeout = 10) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief   接続試行途中の接続をキャンセルします。@ref Connectをコールして結果が返ってくるまでの間有効です。@n
 *    本関数は、無線のインターフェースを閉じることで接続試行の中止を実現します。そのため、本関数の呼び出し後はActionFrameやデータフレームなどの@n
 *    受信が出来なくなりますので、ご注意下さい。無線のインターフェースはスキャンを行ったり、もう一度接続試行を行えば開きます。@n
 *    スキャン中に本関数を呼ぶと失敗が返ります。@n
 *    @details      実行可能ステート：WlanState_LocalClientIdle @n
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         接続の中止に成功。
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。（developビルドの場合、アサートします）
 */
nn::Result CancelConnect() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief   Masterへの同期を開始します。Spectatorモードとして動作させたい場合に使用して下さい。@n
 *    本関数では、同期試行を開始して、その行為が完了するまでを行います。行為が正常に完了したら本関数は成功を返します。@n
 *    同期が成功したことを意味するものではないので注意して下さい。@n
 *    実際に同期が成功したか失敗したかは @ref GetConnectionStatus で確認して下さい。@n
 *    行為が完了するまで本関数は抜けませんので、完了前にキャンセルをしたい場合は、別スレッドから @ref CancelJoinをコールしてください。@n
 *    本関数を抜けた時点で、@ref GetConnectionEvent で取得したイベントはシグナルされています。@n
 *    同期は接続と異なり、試行時に自機の情報をMasterへ送りません。また、Broadcastデータしか受信できません。@n
 *    同期先チャンネルを指定している場合は、そのチャンネルに対してのみ同期試行を行いますので、行為完了までの時間が短縮されます。@n
 *    スキャン中に本関数を呼ぶと失敗が返ります。@n
 *    なお、内部ではPassiveScanが行われます。ステルスのMasterへ同期したい場合は、bssidとchannelを正しく指定して下さい。
 *    @details      実行可能ステート：WlanState_LocalSpectatorIdle @n
 *
 *    @param[in]    ssid               同期先アクセスポイントのSSIDです。
 *    @param[in]    bssid              同期先アクセスポイントのBSSIDです。相手先のBSSIDが分かっている場合は入力して下さい。不明の場合は、BroadcastMACアドレス(FF:FF:FF:FF:FF:FF)を入力して下さい。
 *    @param[in]    channel            同期先アクセスポイントのチャンネル番号です。@ref GetAllowedChannels で許可されているチャンネルの範囲で指定します。不明の場合は-1を入力して下さい。-1を指定した場合は、利用可能なすべてのチャンネルに対し同期を試みます。
 *    @param[in]    security           同期先アクセスポイントのセキュリティ設定です。詳細は @ref Security 構造体を参照してください。ローカル通信では、SecurityMode_OpenまたはSecurityMode_StaticAesのみ指定可能です。
 *    @param[in]    indication         接続確立後のBeacon受信通知の有無です。(未実装です)※廃止予定
 *    @param[in]    beaconLostTimeout  接続先アクセスポイントからのBeaconを何秒間受け取れなかったら自動的に切断するかの時間です。単位は秒です。デフォルトは10秒に設定されます。1～30秒の範囲で指定して下さい。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         同期試行が正常に完了しました。（同期成功を意味するわけではありません）
 *    @retval       ResultInvalidArgument 不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。
 */
nn::Result Join(const Ssid& ssid, const MacAddress& bssid, uint16_t channel, const Security& security,
        BeaconIndication indication, int beaconLostTimeout = 10) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *     @brief   同期試行をキャンセルします。@ref Joinをコールして結果が返ってくるまでの間有効です。@n
 *    スキャン中に本関数を呼ぶと失敗が返ります。
 *    @details      実行可能ステート：WlanState_LocalSpectatorIdle @n
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         同期の中止に成功。
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。（developビルドの場合、アサートします）
 */
nn::Result CancelJoin() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        Masterからの切断/同期解除を行います。また、Masterモードのときは、指定したClientを切断します。@n
 *    【Clientモードで呼んだ場合】接続状態であれば、本関数を抜けた時点で、切断状態であることが保証されます。@n
 *    また、本関数を抜けた時点で、@ref GetConnectionEvent で取得したイベントはシグナルされています。
 *    一方、既に切断済みの場合、何も行わず本関数はResultSuccessを返します。この時、イベントはシグナルされません。
 *    【Masterモードで呼んだ場合】Master が本関数を実行する場合は @ref DisconnectClient の reasonCode に@ref Dot11ReasonCode の値を設定する必要があります。@n
 *    Clientに対し切断が成功した場合、@ref GetConnectionEvent で取得したイベントがシグナルされます。@n
 *    切断したいClientがいない場合、何も行わず、またイベントもシグナルされません。
 *    スキャン中に本関数を呼ぶと失敗が返ります。@n
 *    @details      実行可能ステート：WlanState_LocalMasterIdle (何もせず成功が返ります)@n
 *                               WlanState_LocalMAsterBss @n
 *                               WlanState_LocalClientIdle (何もせず成功が返ります)@n
 *                               WlanState_LocalClient @n
 *                               WlanState_LocalSpectatorIdle (何もせず成功が返ります)@n
 *                               WlanState_LocalSpectator
 *
 *    @param[in]    mode                  Local通信のモード
 *    @param[in]    pInClient             @ref mode でLocalCommunicationMode_Masterを入力した時のみ有効な引数です。@n
 *    切断したいClientのMACアドレスを入力します。Broadcastアドレスを入力した場合、接続済みのClient全てに対し切断要求を行います。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         切断に成功。(Clientの場合、既に切断状態で呼んでもこの値が返ります。Masterの場合、1台もClientが接続していない場合はこの値が返ります。)
 *    @retval       ResultInvalidArgument 不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。
 *    @retval       ResultClientNotFound  切断相手に指定したClientは接続されていません。(そもそも1台もClientが接続されていなかった場合は、ResultSuccessの返り値が優先されます。)
 */
nn::Result Disconnect(LocalCommunicationMode mode, const DisconnectClient* pInClient = NULL) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief   接続状態の変更を通知するイベントオブジェクトを取得します。無線の接続、切断が起こると、このイベントがシグナルされます。
 *    @detail  LocalClient/LocalSpectatorの場合は、相手との接続や切断が発生した場合にシグナルされます。@n
 *    LocalBssの場合は、LocalClientとの接続が確立または切断が起きた場合にシグナルされます。@n
 *                  実行可能ステート：WlanState_Ready @n
 *                               WlanState_LocalMasterIdle @n
 *                               WlanState_LocalMasterBss @n
 *                               WlanState_LocalMasterIdleScan @n
 *                               WlanState_LocalMasterBssScan @n
 *                               WlanState_LocalClientIdle @n
 *                               WlanState_LocalClient @n
 *                               WlanState_LocalClientIdleScan @n
 *                               WlanState_LocalClientScan @n
 *                               WlanState_LocalSpectatorIdle @n
 *                               WlanState_LocalSpectator @n
 *                               WlanState_LocalSpectatorIdleScan @n
 *                               WlanState_LocalSpectatorScan @n
 *
 *    @param[in] pSystemEvent    SystemEventType オブジェクトへのポインタ
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         成功。
 *    @retval       ResultInvalidArgument 不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。
 *
 */
nn::Result GetConnectionEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief   接続状態の情報を取得します。@n
 *    @ref GetConnectionEvent で取得したイベントがシグナルされたときに ConnectionStatus の更新が行われます。@n
 *    @details      実行可能ステート：WlanState_LocalMasterIdle @n
 *                               WlanState_LocalMasterBss @n
 *                               WlanState_LocalMasterIdleScan @n
 *                               WlanState_LocalMasterBssScan @n
 *                               WlanState_LocalClientIdle @n
 *                               WlanState_LocalClient @n
 *                               WlanState_LocalClientIdleScan @n
 *                               WlanState_LocalClientScan @n
 *                               WlanState_LocalSpectatorIdle @n
 *                               WlanState_LocalSpectator @n
 *                               WlanState_LocalSpectatorIdleScan @n
 *                               WlanState_LocalSpectatorScan @n
 *
 *    @param[out]   pStatus    接続状態を示す構造体です。詳細は @ref ConnectionStatus 構造体を参照してください。
 *    LocalMasterの場合、@ref CreateBss に成功したら ConnectionStatus が更新されます。自身のSSID、BSSID、チャンネルが確認可能です。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         成功。
 *    @retval       ResultInvalidArgument 不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。
 *
 */
nn::Result GetConnectionStatus(ConnectionStatus* pOutStatus) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief   MasterがClientの接続状態の情報を取得します。
 *    @details      実行可能ステート：WlanState_LocalMasterIdle @n
 *                               WlanState_LocalMasterBss @n
 *                               WlanState_LocalMasterIdleScan @n
 *                               WlanState_LocalMasterBssScan @n
 *
 *    @param[out]   connectionStatus[]         Clientの接続状態を返します。詳細は @ref ClientStatus 構造体を参照してください。
 *    @param[out]   updatedIndexBitMap         @ref connectionStatus の要素のうち、更新のあった個所をビットマップ形式(LSB)で示します。@n
 *    例） 3番目と2番目のClientに変化があった -> 0x00000006 (b0000000000000000000110)
 *    このビットマップはwlan内部でClientの接続状態に更新があった場合に更新されており、本関数ではそのwlan内部で更新されているビットマップをコピーする形で更新箇所を示しています。@n
 *    なお、wlan内部のビットマップは本関数を呼び出した後にクリアされるので、更新情報としては、前回に本関数を呼び出してから、今回呼び出す間までに起きた変更が含まれることになります。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         成功。
 *    @retval       ResultInvalidArgument 不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。（developビルドの場合、アサートします）
 *
 */
nn::Result GetClientStatus(ClientStatus pOutStatusArray[ConnectableClientsCountMax], Bit32* pOutUpdatedIndexBitMap) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief   LocalClient/LocalSpectatorがConnect/JoinしているMasterからのBeaconフレーム受信を通知するイベントオブジェクトを取得します。(未実装です)
 *
 *    @details Beaconフレームを受信すると、このイベントがシグナルされます。Beaconフレームの詳細情報は、@ref GetBssIndicationInfo をコールすることにより取得します。
 *
 *    @param[in] pSystemEvent    SystemEventType オブジェクトへのポインタ
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         成功。
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。（developビルドの場合、アサートします）
 *
 */
nn::Result GetBssIndicationEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief   LocalClient/LocalSpectatorがConnect/JoinしているMasterからのBeaconフレームのデータを取得します。(未実装です)
 *
 *    @param[out]   bssInfo[] Beaconフレームのデータを返します。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         成功。
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。（developビルドの場合、アサートします）
 *
 */
nn::Result GetBssIndicationInfo(uint8_t (&bssInfo)[BssIndicationBufferSize]) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        無線プロセスのステートを取得します。各ステートは、@ref State を参照してください。
 *    @details      実行可能ステート： 全ステート
 *
 *    @param[out]   pOutState    現在のステート(@ref State)を返します。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultInvalidArgument 不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultSuccess         成功。
 */
nn::Result GetState(WlanState* pOutState) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        無線デバイスで使用の許可されたチャンネルリストを返します。
 *    @details      実行可能ステート： 全ステート
 *
 *    @param[out]   outChannels    無線デバイスで使用の許可されたチャンネルリストが返ります。例えば、1~11chと40chが許可チャンネルの場合、
 *                  outChannels[0] = 1;
 *                  outChannels[1] = 2;
 *                  ...
 *                  outChannels[10] = 11;
 *                  outChannels[11] = 40;
 *                  と言う値が入り、以降は0で埋められます。@n
 *                  チャンネルは必ず先頭から連続で昇順に格納されます。（0は有効なチャンネルではないので、昇順リストには含まれません。また、負数が含まれることはありません。）
 *    @param[out]   pOutCount 許可チャンネルの数が返ります。@ref channels の要素数として利用してください。上記の例ですと、12という数値が返ります。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultInvalidArgument 不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultSuccess         成功。
 */
nn::Result GetAllowedChannels(int16_t (&outChannels)[WirelessChannelsCountMax], uint32_t* pOutCount) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        指定されたManagement Frameに任意のIEを追加します。@n
 *    追加出来るIEのサイズは最大255bytesです。また、既にIEを追加済みの場合、合計で255bytes以下に収まれば追加可能です。@n
 *    複数種類のIEを追加する場合はご注意ください。@n
 *    例）Beaconに155bytesのIEを追加している。この状態でProbe.resフレームに対し、150bytesのIEを別途追加することは出来ない。IEの消費サイズが合計で255bytesを超えてしまうため。
 *    @details      実行可能ステート：WlanState_LocalMasterIdle @n
 *                               WlanState_LocalMasterBss @n
 *                               WlanState_LocalClientIdle @n
 *                               WlanState_LocalClient @n
 *                               WlanState_LocalSpectatorIdle @n
 *                               WlanState_LocalSpectator @n
 *
 *    @param[out]    pOutIeIndex 追加したIEに対し、Indexが付与され、その値が返ります。IEを取り除きたい場合は、ここで得たIndex値を指定して下さい。@n
 *    Index値は1以上の値が返ります。IEを追加する度に付与されるIndex値はインクリメントされていきます。IEのサイズ制限上、255より大きなIndex値が返ることはありません。@n
 *    また@ref DeleteIeで削除した場合、削除されたIndex値は次に@ref AddIeを行ったときに抜け番とみなされ、再利用されます。
 *    @param[in]     type どのManagement FrameにIEを追加するか指定します。@ref ManagementFrameType を参照してください。ビットで定義していますので、論理和での指定も可能です。@n
 *    例）BeaconとProbe.resに付与したい場合： type = ManagementFrameType_Beacon | ManagementFrameType_ProbeRes@n
 *    上記のように複数のManagement Frameを指定した場合、同じ内容のIEがそれぞれに付与されることになります。ただし消費するIEサイズは1つ分で済みます。@n
 *    同一内容のIEを複数種類のManagement Frameに付与したい場合は必ず上記のように論理和合成したtypeを入力するようにして下さい。
 *    @param[in]     ieBody 追加するIEの内容です。OUIフィールド以降の内容を入力して下さい。@n
 *    Tagは自動的に0xDDとなり、lengthフィールドにはisBodySizeが入ります。
 *    @param[in]     isBodySize 追加するIEのサイズです。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         成功。
 *    @retval       ResultInvalidArgument 不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。（developビルドの場合、アサートします）
 */
nn::Result AddIe(uint32_t* pOutIeIndex, ManagementFrameType type, const uint8_t* ieBody, size_t ieBodySize) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        指定したIndexに該当するIEを削除します。
 *    @details      実行可能ステート：WlanState_LocalMasterIdle @n
 *                               WlanState_LocalMasterBss @n
 *                               WlanState_LocalClientIdle @n
 *                               WlanState_LocalClient @n
 *                               WlanState_LocalSpectatorIdle @n
 *                               WlanState_LocalSpectator @n
 *
 *    @param[in]     IeIndex 削除したいIEのIndex。@ref AddIe 関数にてIE追加時にIndex値を取得出来ますので、その値を入力して下さい。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         成功。
 *    @retval       ResultInvalidArgument 指定したIEのIndex値が不正です。
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。（developビルドの場合、アサートします）
 */
nn::Result DeleteIe(uint32_t IeIndex) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        無線チップのMACアドレスを取得します。
 *    @details      実行可能ステート： 全ステート
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultInvalidArgument 不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultSuccess         成功。
 */
nn::Result GetMacAddress(MacAddress* pOutMac) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        接続中の機器との通信強度を取得します。本関数は接続中かつLocalClient/LocalSpectator時のみ有効です。@n
 *    取得に失敗した場合、pRssiには-128が入ります。(-128[dbm]は圏外レベルのRSSIを意味します)
 *    @details      実行可能ステート：WlanState_LocalClient @n
 *                               WlanState_LocalClientScan @n
 *                               WlanState_LocalSpectator @n
 *                               WlanState_LocalSpectatorScan @n
 *
 *    @param[out]   pRssi 信号強度が返ります。@n
 *    一般的には -30[dbm] ～ -90[dbm]といった負の範囲の値が入ります。電波の強さは@n
 *    (電波強い)-30 >>>>>>>>> -90 (電波弱い)@n
 *    となります。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         成功。
 *    @retval       ResultCommandFailure  RSSIの取得に失敗しました。
 *    @retval       ResultInvalidArgument 不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。
 */
nn::Result GetRssi(int32_t* pRssi) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        接続中の機器とのリンクレベル（通信品質）を取得します。本関数は接続中かつLocalClient/LocalSpectator時のみ有効です。@n
 *    取得に失敗した場合、pLinkLevelにはLinkLevel_0が入ります。
 *    @details      実行可能ステート：WlanState_LocalClient @n
 *                               WlanState_LocalClientScan @n
 *                               WlanState_LocalSpectator @n
 *                               WlanState_LocalSpectatorScan @n
 *
 *    @param[out]   pLinkLevel リンクレベル(通信品質)が返ります。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         成功。
 *    @retval       ResultCommandFailure  リンクレベルの取得に失敗しました。
 *    @retval       ResultInvalidArgument 不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。
 */
nn::Result GetLinkLevel(LinkLevel* pLinkLevel) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        入力されたRSSI値をLinkLevelに変換します。
 *    @details      実行可能ステート：全ステート
 *
 *    @param[in]    rssi RSSI値
 *
 *    @return       変換されたLenkLevelが返ります。
 */
LinkLevel ConvertRssiToLinkLevel(int32_t rssi) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        接続中の相手にデータフレームを送信します。コピー版。@n
 *    @ref AddMatchingDataToRxEntry 関数にて既に比較用のデータ列を登録しているときのみ有効です。登録を行っていない場合、データの送信は行われません。@n
 *    比較用データは送信可能かどうかを判断するためのフラグとして扱われます。
 *    @details      実行可能ステート：WlanState_LocalMasterBss @n
 *                               WlanState_LocalClient @n
 *
 *    @param[in] pInput 送信データへのポインタです。
 *    @param[in] size   送信データのサイズです。2KB未満を指定してください。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         成功。
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。
 *    @retval       ResultInvalidArgument 不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultTxQueueIsFull   送信キューが一杯なため送信を諦めました。
 *    @retval       ResultNotAuthorized   接続は完了していても、@ref AddMatchingDataToRxEntry によるデータ列の登録がされていない場合は本Result値が返ります。
 *    @retval       ResultCommandFailure  送信に失敗しました。既に通信相手との接続が切れている場合に発生します。@n
 *                  wlanプロセスが切断を検知していればResultNotAuthorizedが返りますが、検知前に送信が行われた場合に発生します。
 */
nn::Result PutFrameRaw(const uint8_t pInput[], size_t size) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        接続中の相手からデータフレームを受信します。何も受信していなかった場合、受信するか接続が切れるまでブロックします。コピー版。@n
 *    受信エントリを登録している必要があります。@ref CreateRxEntry にて受信エントリを作成しておいて下さい。@n
 *    また、@ref AddMatchingDataToRxEntry 関数にて既に比較用のデータ列を登録しているときのみ有効です。@n
 *    @ref AddMatchingDataToRxEntry 関数でデータ列の登録を行っていない場合、WLANプロセスで受信したデータは全て通常のデータパスに流れます。@n
 *    この場合、本関数はその旨を記載したエラーを返します。
 *    @details      実行可能ステート：WlanState_LocalMasterBss @n
 *                               WlanState_LocalMasterBssScan @n
 *                               WlanState_LocalClient @n
 *                               WlanState_LocalClientScan @n
 *                               WlanState_LocalSpectator @n
 *                               WlanState_LocalSpectatorScan @n
 *
 *    @param[out] pOutBuf  受信バッファへのポインタです。
 *    @param[in]  size     受信バッファのサイズです。受信データは最大でも2KBを超えることはありません。
 *    @param[out] pOutSize 実際に受信したデータのサイズが返ります。
 *    @param[in]  rxId     受信エントリのIDです。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess            成功。
 *    @retval       ResultInvalidState       不正なステート時に呼び出されました。
 *    @retval       ResultInvalidArgument    不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultRxEntryNotFound    指定されたrxIdが見つかりません。
 *    @retval       ResultNotAuthorized      接続先とのリンクが確立していないのでデータの送信が行えません。@n
 *                  接続は完了していても、@ref AddMatchingDataToRxEntry によるデータ列の登録がされていない場合は本Result値が返ります。
 *    @retval       ResultGetFrameCancelled  受信中止が指示されました。受信の中止は@ref CancelGetFrame を呼ぶか、@ref DeleteRxEntry または@n
 *                  CloseMaster/ClientMode の処理の中で自動的に呼ばれます。
 *    @retval       ResultRxEntryIsMuted     データパスが無効化されています。Close[Master/Client]Modeを呼ぶと、wlanプロセス内部でデータパスを一時的に無効化状態とします。@n
 *                  この状態で本関数を呼ぶと受信は行えずに本リザルトが返ります。Open[Master/Client]Modeを呼ぶとこの無効化状態は解除されます。
 *    @retval       ResultBufferTooShort     pOutBufのバッファサイズが小さく、受信データを格納出来ません。
 */
nn::Result GetFrameRaw(uint8_t pOutBuf[], size_t size, size_t* pOutSize, uint32_t rxId) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        @ref GetFrame または @ref GetFrameRaw で受信待ちの状態を解除します。
 *    @details      実行可能ステート：WlanState_LocalMasterIdle @n
 *                               WlanState_LocalMasterIdleScan @n
 *                               WlanState_LocalMasterBss @n
 *                               WlanState_LocalMasterBssScan @n
 *                               WlanState_LocalClientIdle @n
 *                               WlanState_LocalClientIdleScan @n
 *                               WlanState_LocalClient @n
 *                               WlanState_LocalClientScan @n
 *                               WlanState_LocalSpectatorIdle @n
 *                               WlanState_LocalSpectatorIdleScan @n
 *                               WlanState_LocalSpectator @n
 *                               WlanState_LocalSpectatorScan @n
 *
 *    @param[in] rxId         受信エントリの ID
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess            成功。
 *    @retval       ResultInvalidState       不正なステート時に呼び出されました。
 *    @retval       ResultRxEntryNotFound    指定されたrxIdが見つかりません。
 */
nn::Result CancelGetFrame(uint32_t rxId) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        受信エントリを作成します。@n
 *    受信データの取得にはあらかじめ受信エントリを作成しておく必要があります。ここでどのEthertypeのデータを受信したいか登録しておけば@n
 *    受信エントリにそのEthertypeのデータが格納されていきます。
 *    @details      実行可能ステート:WlanState_LocalMasterIdle @n
 *                               WlanState_LocalMasterBss @n
 *                               WlanState_LocalClientIdle @n
 *                               WlanState_LocalClient @n
 *                               WlanState_LocalSpectatorIdle @n
 *                               WlanState_LocalSpectator @n
 *
 *    @param[out]   pRxId        作成された受信エントリの IDが返ります。
 *    @param[in]    pEthertypes  受信エントリに登録するEthertypeの配列です。
 *    @param[in]    count        Ethertype配列の要素数です。受信エントリ1つにつき、最大@ref EthertypeCountMax個まで登録出来ます。
 *    @param[in]    capacity     受信エントリに溜めておける受信フレーム数の上限値です。最大 @ref DataFrameRxCapacityMax 個貯めておけます。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess             成功。
 *    @retval       ResultInvalidState       不正なステート時に呼び出されました。
 *    @retval       ResultInvalidArgument    不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultDuplicateRxEntry    指定した Ethertype についての受信エントリが既に存在しています。
 *    @retval       ResultRxEntryCountIsFull  作成可能な受信エントリ数が最大に達しています。
 */
nn::Result CreateRxEntry(uint32_t* pOutRxId, const uint16_t pEthertypes[], uint32_t count, uint32_t capacity) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        受信エントリを削除します。
 *    @details      実行可能ステート：WlanState_LocalMasterIdle @n
 *                               WlanState_LocalMasterBss @n
 *                               WlanState_LocalClientIdle @n
 *                               WlanState_LocalClient @n
 *                               WlanState_LocalSpectatorIdle @n
 *                               WlanState_LocalSpectator @n
 *
 *    @param[in]    rxId         受信エントリの ID
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess            成功。
 *    @retval       ResultInvalidState       不正なステート時に呼び出されました。
 *    @retval       ResultRxEntryNotFound    指定されたrxIdが見つかりません。
 */
nn::Result DeleteRxEntry(uint32_t rxId) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        登録済み受信エントリに指定したEthertypeを追加します。
 *    @details      実行可能ステート：WlanState_LocalMasterIdle @n
 *                               WlanState_LocalMasterBss @n
 *                               WlanState_LocalClientIdle @n
 *                               WlanState_LocalClient @n
 *                               WlanState_LocalSpectatorIdle @n
 *                               WlanState_LocalSpectator @n
 *
 *    @param[in]    rxId        受信エントリの ID
 *    @param[in]    ethertype   追加するEthertype
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess          成功。
 *    @retval       ResultInvalidState     不正なステート時に呼び出されました。
 *    @retval       ResultRxEntryNotFound  指定されたrxIdが見つかりません。
 *    @retval       ResultDuplicateRxEntry 受信エントリに指定したEthertype が既に登録されています。
 */
nn::Result AddEthertypeToRxEntry(uint32_t rxId, uint16_t ethertype) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        登録済み受信エントリから指定したEthertypeを削除し、もともと登録されていた受信エントリの ID を取得します。
 *    @details      実行可能ステート：WlanState_LocalMasterIdle @n
 *                               WlanState_LocalMasterBss @n
 *                               WlanState_LocalClientIdle @n
 *                               WlanState_LocalClient @n
 *                               WlanState_LocalSpectatorIdle @n
 *                               WlanState_LocalSpectator @n
 *
 *    @param[in/out]    pOutRxId    受信エントリの ID
 *    @param[in]        ethertype   削除するEthertype
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess            成功。
 *    @retval       ResultInvalidState       不正なステート時に呼び出されました。
 *    @retval       ResultInvalidArgument    不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultRxEntryNotFound    指定されたrxIdが見つかりません。。または、受信エントリに指定したEthertype が登録されていません。
 */
nn::Result DeleteEthertypeFromRxEntry(uint32_t* pOutRxId, uint16_t ethertype) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        データパケットの通信において、特定のデータ列を有するデータの受信を行うように設定します。@n
 *    どのようなデータ列とするかは@ref data で指定します。@n
 *    @ref pMatchInfo で入力されたデータ列を用いて受信データの選別が行われます。WLANプロセスにてデータの受信が行われると、@n
 *    DIX仕様のフレームフォーマットのEtherTypeフィールドに相当する箇所の次から始まる値を@ref pMatchInfo で入力された値と比較します。@n
 *    このとき、値が一致していればWLAN内でデータを確保しておき、@ref GetFrameRaw 関数で受信可能となります。@n
 *    本関数で特にデータ列を指定していなかった場合は、WLAN内で特に確保は行わず、受信データは通常のデータパスに流されます。@n
 *    複数個の比較用データ列を登録したい場合は、本関数を複数回呼んでください。最大10個まで登録が可能です。
 *    @details      実行可能ステート：WlanState_LocalMasterIdle @n
 *                               WlanState_LocalMasterBss @n
 *                               WlanState_LocalClientIdle @n
 *                               WlanState_LocalClient @n
 *                               WlanState_LocalSpectatorIdle @n
 *                               WlanState_LocalSpectator @n
 *
 *    @param[in]    rxId        受信エントリの ID
 *    @param[in]    pMatchInfo 比較対象のデータ。@ref ReceivedDataMatchLengthMaxより大きなサイズのデータは登録出来ません。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess             成功。
 *    @retval       ResultInvalidState        不正なステート時に呼び出されました。
 *    @retval       ResultInvalidArgument     不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultRxEntryNotFound     指定されたrxIdが見つかりません。
 *    @retval       ResultDataMatchInfoError  マッチングデータの登録に失敗しました。登録最大数を超過したか、または既に同じデータが登録されています。
 */
nn::Result AddMatchingDataToRxEntry(uint32_t rxId, const ReceivedDataMatchInfo& pMatchInfo) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        @ref 受信エントリに登録されている比較用データ列を削除します。@n
 *    @ref AddMatchingDataToRxEntry 関数にて既にデータ列を登録済みの場合、@ref pMatchInfo で入力されたデータ列と一致するデータ列を登録から外します。
 *    @details      実行可能ステート：WlanState_LocalMasterIdle @n
 *                               WlanState_LocalMasterBss @n
 *                               WlanState_LocalClientIdle @n
 *                               WlanState_LocalClient @n
 *                               WlanState_LocalSpectatorIdle @n
 *                               WlanState_LocalSpectator @n
 *
 *    @param[in]    rxId        受信エントリの ID
 *    @param[in]    pMatchInfo 比較対象のデータ
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess             成功。
 *    @retval       ResultInvalidState        不正なステート時に呼び出されました。
 *    @retval       ResultInvalidArgument     不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultRxEntryNotFound     指定されたrxIdが見つかりません。
 */
nn::Result RemoveMatchingDataFromRxEntry(uint32_t rxId, const ReceivedDataMatchInfo& pMatchInfo) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        Action Frameを単発で送信します。@n
 *    @ref pData にはAction frame フォーマットのCategoryフィールドから埋めてください。@n
 *    +-------------------------------------------------+ @n
 *    | Category |      OUI      | Subtype |    body    | @n
 *    |  1 byte  |    3 bytes    |  1 byte |  ? bytes   | @n
 *    +-------------------------------------------------+ @n
 *    Categoryには必ず、127(=0x7F)を入れてください。Subtypeには必ず @ref ActionFrameType 列挙体の値を入れてください。@n
 *    既に接続中かつ、送信先チャンネルが接続中のチャンネルと異なる場合、Action Frame送信の瞬間は通信が途切れる可能性があります。
 *    @details      実行可能ステート：WlanState_LocalMasterBss @n
 *                               WlanState_LocalClientIdle @n
 *                               WlanState_LocalClient @n
 *                               WlanState_LocalSpectatorIdle @n
 *                               WlanState_LocalSpectator @n
 *
 *    @param[in]    dstMac    宛先MACアドレス。ZeroMacAddressは無効です。
 *    @param[in]    data      Action Frameに付加するデータ。上記説明参照。
 *    @param[in]    size      Action Frameに付加するデータのサイズ。(Category + OUI + Subtype + body)@n
 *    Category + OUI + Subtypeは必ず必要なので、5Byte以上1640Byte以下を指定してください。
 *    @param[in]    channel   送信先チャンネル
 *    @param[in]    dwellTime 送信後、送信先チャンネルに留まっている時間。留まっている間は電波受信状態となります。@n
 *    例えば、相手からの返事を待ちたい場合はこの値を調整して下さい。単位はmsecです。最大100msec待てます。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         成功。
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。
 *    @retval       ResultInvalidArgument 不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultCommandFailure  送信に失敗しました。送信完了バッファの解放が追い付かず詰まっている可能性があります。少し間を空けてから再度試して下さい。
 */
nn::Result PutActionFrameOneShot(const MacAddress& dstMac, const uint8_t* pData, size_t size, int16_t channel, uint32_t dwellTime) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        LocalMasterBssのビーコンと共に送信するAction Frameの設定を行います。@n
 *    本関数でActionFrameの設定をしておくと、@ref CreateBss を行った際に、BSSのビーコンに沿うようにAction FrameがBroadcast送信されます。@n
 *    ビーコンと共にデータを定期送信したい場合にご使用下さい。 また、既にBSSが作成済みの場合、本関数を呼んだ時点からAction Frameの送信が開始されます。@n
 *    @ref CancelActionFrameWithBeacon か @ref DestroyBss を行うとAction Frameの送信は停止され、内容もクリアされます。@n
 *    再度@ref CreateBssを行っても前回セットしたActionFrameは送信されませんので、BSS作成の度に、本関数を呼んで、Action Frameのセットを行ってください。@n
 *    @ref pData へのデータの入れ方は @ref PutActionFrameOneShot と同様です。ただし、Subtypeには必ず、ActionFrameType_Beacon を指定して下さい。@n
 *    送信中に再度本関数を呼ぶと、内容が更新されたものが送信されるようになります。実際にAction Frameに反映されるのは、セット後の次の送信タイミングとなります。
 *    @details      実行可能ステート：WlanState_LocalMasterIdle @n
 *                               WlanState_LocalMasterBss @n
 *
 *    @param[in]    data Action Frameに付加するデータ。
 *    @param[in]    size Action Frameに付加するデータのサイズ。5Byte以上1450Byte以下を指定してください。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess             成功。
 *    @retval       ResultInvalidState        不正なステート時に呼び出されました。
 *    @retval       ResultInvalidArgument     不正な引数値です。（developビルドの場合、アサートします）
 */
nn::Result SetActionFrameWithBeacon(const uint8_t* pData, size_t size) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        Action Frameの周期的送信を停止します。
 *    @details      実行可能ステート：WlanState_LocalMasterIdle @n
 *                               WlanState_LocalMasterBss @n
 *
 *    @details      @ref SetActionFrameWithBeacon 関数で送信中の周期的Action Frameの送信を停止します。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess             成功。
 *    @retval       ResultInvalidState        不正なステート時に呼び出されました。
 *
 */
nn::Result CancelActionFrameWithBeacon() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        Action Frameの受信を行います。指定したタイプのAction Frameが返ります。@n
 *    受信にはあらかじめ@ref CreateRxEntryForActionFrameにてAction Frame用の受信エントリを作成しておく必要があります。
 *    @details      実行可能ステート：WlanState_LocalMasterIdle @n
 *                               WlanState_LocalMasterIdleScan @n
 *                               WlanState_LocalMasterBss @n
 *                               WlanState_LocalMasterBssScan @n
 *                               WlanState_LocalClientIdle @n
 *                               WlanState_LocalClientIdleScan @n
 *                               WlanState_LocalClient @n
 *                               WlanState_LocalClientScan @n
 *                               WlanState_LocalSpectatorIdle @n
 *                               WlanState_LocalSpectatorIdleScan @n
 *                               WlanState_LocalSpectator @n
 *                               WlanState_LocalSpectatorScan @n
 *
 *    @param[out]   pOutSrcMac 送信元MACアドレスが返ります。
 *    @param[out]   pOutBuf    Action Frameが返ります。ActionFrame formatのCategoryフィールド以降が入っています。
 *    @param[in]    size       受信バッファのサイズです。
 *    @param[out]   pOutSize   実際に受信したデータのサイズが返ります。
 *    @param[in]    rxId       受信エントリID
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess            成功。
 *    @retval       ResultInvalidState       不正なステート時に呼び出されました。
 *    @retval       ResultInvalidArgument    不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultRxEntryNotFound    指定されたrxIdが見つかりません。
 *    @retval       ResultGetFrameCancelled  受信中止が指示されました。受信の中止は@ref CancelGetActionFrame を呼ぶか、@ref DeleteRxEntry または@n
 *                  CloseMaster/ClientMode の処理の中で自動的に呼ばれます。
 *    @retval       ResultRxEntryIsMuted     データパスが無効化されています。Close[Master/Client]Modeを呼ぶと、wlanプロセス内部でデータパスを一時的に無効化状態とします。@n
 *                  この状態で本関数を呼ぶと受信は行えずに本リザルトが返ります。Open[Master/Client]Modeを呼ぶとこの無効化状態は解除されます。
 *    @retval       ResultBufferTooShort     pOutBufのバッファサイズが小さく、受信データを格納出来ません。
 */
nn::Result GetActionFrame(MacAddress* pOutSrcMac, uint8_t pOutBuf[], size_t size, size_t* pOutSize, uint32_t rxId) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        Action Frameの受信を行います。指定したタイプのAction Frameが返ります。@n
 *    受信にはあらかじめ@ref CreateRxEntryForActionFrameにてAction Frame用の受信エントリを作成しておく必要があります。@n
 *    送信元チャンネルとRSSI値も取得出来ます。@n
 *    @details      実行可能ステート：WlanState_LocalMasterIdle @n
 *                               WlanState_LocalMasterIdleScan @n
 *                               WlanState_LocalMasterBss @n
 *                               WlanState_LocalMasterBssScan @n
 *                               WlanState_LocalClientIdle @n
 *                               WlanState_LocalClientIdleScan @n
 *                               WlanState_LocalClient @n
 *                               WlanState_LocalClientScan @n
 *                               WlanState_LocalSpectatorIdle @n
 *                               WlanState_LocalSpectatorIdleScan @n
 *                               WlanState_LocalSpectator @n
 *                               WlanState_LocalSpectatorScan @n
 *
 *    @param[out]   pOutSrcMac   送信元MACアドレスが返ります。
 *    @param[out]   pOutBuf      Action Frameが返ります。ActionFrame formatのCategoryフィールド以降が入っています。
 *    @param[in]    size         受信バッファのサイズです。
 *    @param[out]   pOutSize     実際に受信したデータのサイズが返ります。
 *    @param[in]    rxId         受信エントリID
 *    @param[out]   pOutChannel  送信元チャンネル
 *    @param[out]   pOutRssi     RSSI値
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess            成功。
 *    @retval       ResultInvalidState       不正なステート時に呼び出されました。
 *    @retval       ResultInvalidArgument    不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultRxEntryNotFound    指定されたrxIdが見つかりません。
 *    @retval       ResultGetFrameCancelled  受信中止が指示されました。受信の中止は@ref CancelGetActionFrame を呼ぶか、@ref DeleteRxEntry または@n
 *                  CloseMaster/ClientMode の処理の中で自動的に呼ばれます。
 *    @retval       ResultRxEntryIsMuted     データパスが無効化されています。Close[Master/Client]Modeを呼ぶと、wlanプロセス内部でデータパスを一時的に無効化状態とします。@n
 *                  この状態で本関数を呼ぶと受信は行えずに本リザルトが返ります。Open[Master/Client]Modeを呼ぶとこの無効化状態は解除されます。
 *    @retval       ResultBufferTooShort     pOutBufのバッファサイズが小さく、受信データを格納出来ません。
 *    @retval       ResultNotImplemented     上記wlanプロセスを使用していない場合、これが返ります。
 */
nn::Result GetActionFrameEx(MacAddress* pOutSrcMac, uint8_t pOutBuf[], size_t size,
        size_t* pOutSize, uint32_t rxId, uint16_t* pOutChannel, int16_t* pOutRssi) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        @ref GetActionFrame で受信待ちの状態を解除します。
 *    @details      実行可能ステート：WlanState_LocalMasterIdle @n
 *                               WlanState_LocalMasterIdleScan @n
 *                               WlanState_LocalMasterBss @n
 *                               WlanState_LocalMasterBssScan @n
 *                               WlanState_LocalClientIdle @n
 *                               WlanState_LocalClientIdleScan @n
 *                               WlanState_LocalClient @n
 *                               WlanState_LocalClientScan @n
 *                               WlanState_LocalSpectatorIdle @n
 *                               WlanState_LocalSpectatorIdleScan @n
 *                               WlanState_LocalSpectator @n
 *                               WlanState_LocalSpectatorScan @n
 *
 *    @param[in] rxId         受信エントリの ID
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess            成功。
 *    @retval       ResultInvalidState       不正なステート時に呼び出されました。
 *    @retval       ResultRxEntryNotFound    指定されたrxIdが見つかりません。
 */
nn::Result CancelGetActionFrame(uint32_t rxId) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        Action Frame用の受信エントリを作成します。
 *    Action Frameデータの取得にはあらかじめ受信エントリを作成しておく必要があります。ここでどのActionFrameTypeのデータを受信したいか登録しておけば@n
 *    受信エントリにそのActionFrameTypeのデータが格納されていきます。
 *    @details      実行可能ステート:WlanState_LocalMasterIdle @n
 *                               WlanState_LocalMasterBss @n
 *                               WlanState_LocalClientIdle @n
 *                               WlanState_LocalClient @n
 *                               WlanState_LocalSpectatorIdle @n
 *                               WlanState_LocalSpectator @n
 *
 *    @param[out]   pRxId        作成された受信エントリの IDが返ります。
 *    @param[in]    pAfTypes     受信エントリに登録するActionFrameTypeの配列です。
 *    @param[in]    count        ActionFrameType配列の要素数です。最大 @ref ActionFrameTypeCountMax個登録出来ます。
 *    @param[in]    capacity     受信エントリに溜めておける受信フレーム数の上限値です。最大 @ref ActionFrameRxCapacityMax 個溜められます。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess             成功。
 *    @retval       ResultInvalidState        不正なステート時に呼び出されました。
 *    @retval       ResultInvalidArgument     不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultDuplicateRxEntry    指定した ActionFrameType についての受信エントリが既に存在しています。
 *    @retval       ResultRxEntryCountIsFull  作成可能な受信エントリ数が最大に達しています。
 */
nn::Result CreateRxEntryForActionFrame(uint32_t* pOutRxId, const uint16_t pAfTypes[], uint32_t count, uint32_t capacity) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        Action Frame用受信エントリを削除します。
 *    @details      実行可能ステート:WlanState_LocalMasterIdle @n
 *                               WlanState_LocalMasterBss @n
 *                               WlanState_LocalClientIdle @n
 *                               WlanState_LocalClient @n
 *                               WlanState_LocalSpectatorIdle @n
 *                               WlanState_LocalSpectator @n
 *
 *    @param[in]    rxId         受信エントリの ID
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess            成功。
 *    @retval       ResultInvalidState       不正なステート時に呼び出されました。
 *    @retval       ResultRxEntryNotFound    指定されたrxIdが見つかりません。
 */
nn::Result DeleteRxEntryForActionFrame(uint32_t rxId) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        登録済みのAction Frame用受信エントリに指定したActionFrameTypeを追加します。
 *    @details      実行可能ステート:WlanState_LocalMasterIdle @n
 *                               WlanState_LocalMasterBss @n
 *                               WlanState_LocalClientIdle @n
 *                               WlanState_LocalClient @n
 *                               WlanState_LocalSpectatorIdle @n
 *                               WlanState_LocalSpectator @n
 *
 *    @param[in]    rxId        受信エントリの ID
 *    @param[in]    type        追加するActionFrameType
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess          成功。
 *    @retval       ResultInvalidState     不正なステート時に呼び出されました。
 *    @retval       ResultInvalidArgument  不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultRxEntryNotFound  指定されたrxIdが見つかりません。
 *    @retval       ResultDuplicateRxEntry 受信エントリに指定したActionFrameType が既に登録されています。
 */
nn::Result AddSubtypeToRxEntryForActionFrame(uint32_t rxId, ActionFrameType type) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        登録済みのAction Frame用受信エントリから指定したActionFrameTypeを削除し、@n
 *    もともと登録されていた受信エントリの ID を取得します。
 *    @details      実行可能ステート:WlanState_LocalMasterIdle @n
 *                               WlanState_LocalMasterBss @n
 *                               WlanState_LocalClientIdle @n
 *                               WlanState_LocalClient @n
 *                               WlanState_LocalSpectatorIdle @n
 *                               WlanState_LocalSpectator @n
 *
 *    @param[in/out]   pOutRxId    受信エントリの ID
 *    @param[in]       type        削除するActionFrameType
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess           成功。
 *    @retval       ResultInvalidState      不正なステート時に呼び出されました。
 *    @retval       ResultInvalidArgument   不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultRxEntryNotFound   指定されたrxIdが見つからないか、指定されたActionFrameTypeが登録されていません。
 */
nn::Result DeleteSubtypeFromRxEntryForActionFrame(uint32_t* pOutRxId, ActionFrameType type) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
 *    @brief        MASTER側で、CLIENTの接続台数に制限を設けます。システム上の最大接続台数は@n
 *    @ref ConnectableClientsCountMaxとなっており、これを超える値はセットできません。@n
 *    また、0を指定することも出来ません。1以上の値をセットして下さい。@n
 *    接続台数制限を超えて接続試行したCLIENTは接続に必ず失敗し、@ref ConnectionStatus.statusReasonCode には@n
 *    Dot11StatusCode_AssocDeniedNoResourceが入るようになっています。@n
 *    CLIENTは statusReasonCodeを確認することで最大接続台数超過による失敗かどうかを見分けることが出来ます。
 *    @detail       実行可能ステート   WlanState_LocalMasterIdle
 *
 *    @param[in]    maxNum         最大接続許容台数。(1 ~ ConnectableClientsCountMax)
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         成功。
 *    @retval       ResultInvalidArgument 不正な引数が渡されました。（developビルドの場合、アサートします）
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。
 */
nn::Result SetMaxAssociationNumber(uint32_t maxNum) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
 *    @brief        チャンネルごとの電波混雑度を取得します。(デバッグ専用)
 *    内部で一度スキャンを行い、混雑度を測定するため、スキャンパラメータを指定する必要があります。
 *    @details      実行可能ステート:WlanState_LocalMasterIdle @n
 *                               WlanState_LocalClientIdle @n
 *                               WlanState_LocalClient @n
 *                               WlanState_LocalSpectatorIdle @n
 *                               WlanState_LocalSpectator @n
 *
 *    @param[out]   outStats    チャンネル毎の混雑度を示すパラメータが返ります。無線ドライバーの都合上、要素数はWirelessChannelsCountMax + 1となっています。@n
 *                  @ref ScanParameters で測定するチャンネルを限定することが出来ます。@n
 *                  結果は必ず先頭から@ref GetAllowedChannels で得られたチャンネル順に格納されます。(下記参照)。@n
 *                  outStats[0].channel = 1 @n
 *                  outStats[1].channel = 2 @n
 *                  outStats[2].channel = 3 @n
 *                  ...
 *                  outStats[12].channel = 13 @n
 *                  outStats[13].channel = 36 @n
 *                  ScanParameters.channelList でスキャンチャンネルを限定しても、結果が格納される位置は変わりません。@n
 *                  例えば、スキャンチャンネルを6chと11chのみと限定しても、@n
 *                  outStats[0].channel = 6 @n
 *                  outStats[1].channel = 11 @n
 *                  とはならず、
 *                  outStats[5].channel = 6 @n
 *                  outStats[10].channel = 11 @n
 *                  となります。@n
 *                  スキャンされなかったチャンネルの結果は前回測定したときの値が入ります。@n
 *                  本関数を呼んでいなくても、スキャンが実行された時点で測定値は更新されますので、他のプロセスやアプリがスキャンをしたことがあれば@n
 *                  その時に測定された値が入ってくることになります。@n
 *                  また、配列の余った要素は0で埋められます。@n
 *    @param[out]   pOutCount   有効なチャンネル数が返ります。@ref GetAllowedChannels　で得られる許可チャンネル数と同じ値となります。@n
 *                  スキャンしたチャンネル数ではないことに注意して下さい。
 *    @param[in]    param       スキャンパラメータです。詳細は @ref ScanParameters をご覧ください。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         状態遷移に成功。
 *    @retval       ResultInvalidArgument 不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。
 */
nn::Result GetChannelStats(ChannelStats (&outStats)[WirelessChannelsCountMax + 1], uint32_t* pOutCount, const ScanParameters& param) NN_NOEXCEPT;
}
}
}

