﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os/os_SystemEvent.h>
#include "wlan_Common.h"
#include "wlan_MacAddress.h"
#include "wlan_Ssid.h"
#include "wlan_StateDefinition.h"
#include "wlan_Types.h"

namespace nn {
namespace wlan {


/*!
   @name       インフラ通信 関連
   @{
 */


//--------------------------------------------------------------------------
/**
 *    @brief        WLANライブラリのインフラ通信のサービスを開始します。この初期化を行っていないと、インフラ通信に関する API は使用できません。
 *    既に本関数を呼んでいた場合、何も行わず成功が返ります。(developビルドの場合アサートします)
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess    初期化に成功。
 */
nn::Result InitializeInfraManager() NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
 *    @brief        WLANライブラリのインフラ通信のサービスを終了します。
 *    @ref InitializeLocalManger を呼んでいない状態で呼んだ場合、何もせず成功が返ります。(developビルドの場合アサートします)
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess    終了処理に成功。
 */
nn::Result FinalizeInfraManager() NN_NOEXCEPT;


namespace Infra {


//--------------------------------------------------------------------------
/**
 *    @brief        インフラストラクチャ モードをオープンします。
 *    @details      実行可能ステート：WlanState_Ready @n
 *                               WlanState_InfraIdle (既に目的ステートにいるため何も起こりません)
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         状態遷移に成功。
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。実行可能ステート以外で呼び出した場合このエラーが返ります。
 */
nn::Result OpenMode() NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
 *    @brief        インフラストラクチャ モードをクローズします。
 *    @details      実行可能ステート：WlanState_Ready (既に目的ステートにいるため何も起こりません) @n
 *                               WlanState_InfraIdle @n
 *                               WlanState_Sleep @n
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         状態遷移に成功。
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。実行可能ステート以外で呼び出した場合このエラーが返ります。
 */
nn::Result CloseMode() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        無線ノードのスキャンを開始します。スキャンが完了するまでブロックされます。
 *                  pOutScanBufferに入る分だけスキャン結果をコピーしますので、十分に大きなサイズを与えることを推奨します（50KB以上）。@n
 *                  なお、入りきらなかった分は破棄されます。また、その場合でも結果はResultSuccessが返ります。@n
 *                  途中で @ref StopScan を呼んだ場合、それまでに取得したスキャン結果が入ります。@n
 *                  使用可能なチャンネル以外のチャンネルを指定した場合、ResultInvalidArgumentが返ります。@n
 *                  指定可能なチャンネルは@ref GetAllowedChannel で取得出来ます。
 *    @details      実行可能ステート：WlanState_InfraIdle @n
 *                               WlanState_InfraSta
 *
 *    @param[in]    pOutScanBuffer     スキャン結果の格納先バッファです。
 *    @param[in]    size               スキャン結果格納先バッファのサイズです。@n
 *    @ref ScanResultHeader がバッファの先頭に格納されますので、必ずこれより大きなサイズのバッファを用意して下さい。@n
 *    電波環境にも依りますが、50KB以上を推奨します。
 *    @param[in]    param              スキャンパラメータです。詳細は @ref ScanParameters をご覧ください。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         スキャンに成功。本関数が返ってきた時点でスキャンも完了しています。
 *    @retval       ResultInvalidArgument 不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。
 */
nn::Result StartScan(void* pOutScanBuffer, size_t size, const ScanParameters& param) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        無線ノードのスキャンを停止します。
 *    @details      実行可能ステート：WlanState_InfraIdle (何もせず成功が返ります)@n
 *                               WlanState_InfraSta (何もせず成功が返ります)@n
 *                               WlanState_InfraIdleScan @n
 *                               WlanState_InfraStaScan @n
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         Scanの中止に成功。
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。
 */
nn::Result StopScan() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief     アクセスポイントへの接続を開始します。@n
 *    本関数では、接続試行を開始して、その行為が完了するまでを行います。行為が正常に完了したら本関数は成功を返します。@n
 *    接続が成功したことを意味するものではないので注意して下さい。@n
 *    実際に接続が成功したか失敗したかは @ref GetConnectionStatus で確認して下さい。@n
 *    行為が完了するまで本関数は抜けませんので、完了前にキャンセルをしたい場合は、別スレッドから @ref CancelConnectをコールしてください。@n
 *    @ref CancelConnectをコール直後に本関数を実行しないで下さい。少なくとも10msは空けて下さい。関数から返ってこなくなる場合があります。@n
 *    対象のAPにも寄りますが、WPAで暗号鍵を間違えていた場合、本関数を抜けるまでに15秒ほど掛かる場合があります。
 *    本関数を抜けた時点で、@ref GetConnectionEvent で取得したイベントはシグナルされています。@n
 *    接続先チャンネルを指定している場合は、そのチャンネルに対してのみ接続試行を行いますので、行為完了までの時間が短縮されます。@n
 *    スキャン中に本関数を呼ぶと失敗(ResultInvalidState)が返ります。@n
 *
 *    接続候補が複数見つかった場合、もっとも電波強度（RSSI値）が高いものに接続を試みます。@n
 *    channelを指定していた場合、そのchannelの中でもっとも電波強度の高い接続候補に接続を試みます。この時、他チャンネルに、より電波強度の強い接続候補がいても@n
 *    接続には行きません。
 *    @details      実行可能ステート：WlanState_InfraIdle @n
 *
 *    @param[in]    ssid               接続先アクセスポイントのSSIDです。必ず指定して下さい。空のSSIDを入力した場合は不正と判断されます。
 *    @param[in]    bssid              接続先アクセスポイントのBSSIDです。不明な場合はBroadcastアドレスを入力して下さい。ZeroMacAddressは不正と判断されます。
 *    @param[in]    channel            接続先アクセスポイントのチャンネル番号です。@ref GetAllowedChannels で許可されているチャンネルの範囲で指定します。@n
 *    不明の場合は-1を指定して下さい。-1を指定した場合は、利用可能なすべてのチャンネルに対し接続を試みます。
 *    @param[in]    security           接続先アクセスポイントのセキュリティ設定です。詳細は @ref Security 構造体を参照してください。@n
 *    Infraモードの場合、privacyModeにSecurityMode_StaticAesは指定できません。不正と判断されます。
 *    @param[in]    autoKeepAlive      接続先アクセスポイントへ自動的にキープアライブフレーム（NULLフレーム）を送るかどうか設定します。@n
 *    trueにした場合、55秒に1度NULLフレームが自動的に飛びます。(未実装)
 *    @param[in]    beaconLostTimeout  接続先アクセスポイントからのBeaconを何秒間受け取れなかったら自動的に切断するかの時間です。@n
 *    単位は秒です。デフォルトは10秒に設定されます。1～30秒の範囲で指定して下さい。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         接続試行が正常に完了しました。（接続成功を意味するわけではありません）
 *    @retval       ResultInvalidArgument 不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。
 */
nn::Result Connect(const Ssid& ssid, const MacAddress& bssid, int16_t channel, const Security& security,
        bool autoKeepAlive, int beaconLostTimeout = 10) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        無線アクセスポイントからの切断を行います。@n
 *    本関数を抜けた時点で、切断状態であることが保証されます。@n
 *    また、本関数を抜けた時点で、@ref GetConnectionEvent で取得したイベントはシグナルされています。@n
 *    ただし、既に切断済みの場合、何も行わず本関数はResultSuccessを返し、イベントもシグナルはされません。@n
 *    スキャン中に本関数を呼ぶと失敗が返ります。@n
 *    @details      実行可能ステート：WlanState_InfraIdle (何もせず成功が返ります)@n
 *                               WlanState_InfraSta @n
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。
 */
nn::Result Disconnect() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief   接続試行途中の接続をキャンセルします。@ref Connectをコールして結果が返ってくるまでの間有効です。@n
 *    本関数は、無線のインターフェースを閉じることで接続試行の中止を実現します。@n
 *    スキャン中に本関数を呼ぶと失敗が返ります。@n
 *    本関数を実行した直後に@ref Connect コールしたい場合、最低でも10msは空けて下さい。@ref Connect が返ってこなくなる可能性があります。@n
 *    @details      実行可能ステート：WlanState_InfraIdle @n
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。
 */
nn::Result CancelConnect() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief   接続状態の変更を通知するイベントオブジェクトを取得します。無線の接続、切断が起こると、このイベントがシグナルされます。
 *    @details      実行可能ステート：WlanState_Ready @n
 *                               WlanState_InfraIdle @n
 *                               WlanState_InfraSta @n
 *                               WlanState_InfraIdleScan @n
 *                               WlanState_InfraStaScan @n
 *
 *    @param[in] pSystemEvent    SystemEventType オブジェクトへのポインタ
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         成功。
 *    @retval       ResultInvalidArgument 不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。
 *
 */
nn::Result GetConnectionEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief   接続状態の情報を取得します。@n
 *    @ref GetConnectionEvent で取得したイベントがシグナルされたときに ConnectionStatus の更新が行われます。@n
 *
 *    @details      実行可能ステート：WlanState_Stop @n
 *                               WlanState_Ready @n
 *                               WlanState_InfraIdle @n
 *                               WlanState_InfraSta @n
 *                               WlanState_InfraIdleScan @n
 *                               WlanState_InfraStaScan @n
 *                               WlanState_Sleep @n
 *
 *    @param[out]   pStatus    接続状態を示す構造体です。詳細は @ref ConnectionStatus 構造体を参照してください。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         成功。
 *    @retval       ResultInvalidArgument 不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。
 *
 */
nn::Result GetConnectionStatus(ConnectionStatus* pOutStatus) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        無線プロセスのステートを取得します。各ステートは、@ref State を参照してください。
 *    @details      実行可能ステート： 全ステート
 *
 *    @param[out]   pOutState    現在のステート(@ref State)を返します。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultInvalidArgument 不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultSuccess         成功。
 */
nn::Result GetState(WlanState* pOutState) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        無線デバイスで使用の許可されたチャンネルリストを返します。
 *    @details      実行可能ステート： 全ステート
 *
 *    @param[out]   outChannels    無線デバイスで使用の許可されたチャンネルリストが返ります。例えば、1~11chと40chが許可チャンネルの場合、
 *                  outChannels[0] = 1;
 *                  outChannels[1] = 2;
 *                  ...
 *                  outChannels[10] = 11;
 *                  outChannels[11] = 40;
 *                  と言う値が入り、以降は0で埋められます。@n
 *                  チャンネルは必ず先頭から連続で昇順に格納されます。（0は有効なチャンネルではないので、昇順リストには含まれません。また、負数が含まれることはありません。）
 *    @param[out]   pOutCount 許可チャンネルの数が返ります。@ref channels の要素数として利用してください。上記の例ですと、12という数値が返ります。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultInvalidArgument 不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultSuccess         成功。
 */
nn::Result GetAllowedChannels(int16_t (&outChannels)[WirelessChannelsCountMax], uint32_t* pOutCount) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        無線チップのMACアドレスを取得します。
 *    @details      実行可能ステート： 全ステート
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultInvalidArgument 不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultSuccess         成功。
 */
nn::Result GetMacAddress(MacAddress* pOutMac) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        接続中の機器との通信強度を取得します。本関数は接続中のみ有効です。@n
 *    取得に失敗した場合、pRssiには-128が入ります。(-128[dbm]は圏外レベルのRSSIを意味します)
 *    @details      実行可能ステート：WlanState_InfraSta @n
 *                               WlanState_InfraStaScan @n
 *
 *    @param[out]   pRssi 信号強度が返ります。@n
 *    一般的には -30[dbm] ～ -90[dbm]といった負の範囲の値が入ります。電波の強さは@n
 *    (電波強い)-30 >>>>>>>>> -90 (電波弱い)@n
 *    となります。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         成功。
 *    @retval       ResultCommandFailure  RSSIの取得に失敗しました。
 *    @retval       ResultInvalidArgument 不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。
 */
nn::Result GetRssi(int32_t* pRssi) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        接続中の機器とのリンクレベル（通信品質）を取得します。本関数は接続中のみ有効です。@n
 *    取得に失敗した場合、pLinkLevelにはLinkLevel_0が入ります。
 *    @details      実行可能ステート：WlanState_InfraSta @n
 *                               WlanState_InfraStaScan @n
 *
 *    @param[out]   pLinkLevel リンクレベル(通信品質)が返ります。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         成功。
 *    @retval       ResultCommandFailure  リンクレベルの取得に失敗しました。pLinkLevelにはLinkLevel_0が入ります。
 *    @retval       ResultInvalidArgument 不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。
 */
nn::Result GetLinkLevel(LinkLevel* pLinkLevel) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        入力されたRSSI値をLinkLevelに変換します。@n
 *    現在接続しているAPの利用チャンネル（2.4GHz帯か5GHz帯か）に依ってリンクレベルの導出方法が異なっており、同じRSSI値を渡しても@n
 *    同じリンクレベルが返ってくるとは限りません。@n
 *    例） 入力RSSI値：-74@n
 *    2.4GHzチャンネルに接続している場合 -> LinkLevel_1@n
 *    5GHzチャンネルに接続している場合 -> LinkLevel_2@n
 *    未接続状態の場合は、2.4GHz用の導出方法が使用されます。
 *    @details      実行可能ステート：全ステート
 *
 *    @param[in]    rssi RSSI値
 *
 *    @return       変換されたLenkLevelが返ります。
 */
LinkLevel ConvertRssiToLinkLevel(int32_t rssi) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        受信アンテナの切り替えを行います。(開発者専用です。一般アプリではご使用にならないで下さい。)@n
 *    @ref Connect() が呼ばれると、wlan内部でアンテナ設定がwlan独自のものに切り替わります。@n
 *    Connect()後もアンテナ設定を変更したい場合は、再度本関数を呼ぶ必要があります。@n
 *    また、@ref CloseMode() を呼ぶと、本関数で指定したアンテナ設定は消えるようになっています。
 *    @details      実行可能ステート：WlanState_InfraIdle @n
 *                               WlanState_InfraSta @n
 *
 *    @param[in]    rxAntenna 受信アンテナの切り替えパターン。RxAntennaPattern列挙体より選択して下さい。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         アンテナの切り替えに成功しました。
 *    @retval       ResultInvalidArgument 引数の値が不正です。（developビルドの場合、アサートします）
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。（developビルドの場合、アサートします）
 */
nn::Result ChangeRxAntenna(RxAntennaPattern rxAntenna) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        無線FWのバージョンを取得します。
 *    @details      実行可能ステート：WlanState_StopおよびWlanState_Exitを除くすべてのステート
 *
 *    @param[out]   pOutStr 無線FWのバージョンの格納先バッファです。文字列で格納されます。256byte以上のバッファを用意して下さい。
 *    @param[in]    size    pOutStrのサイズです。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         無線FWのバージョンの取得に成功しました。
 *    @retval       ResultInvalidArgument 引数の値が不正です。（developビルドの場合、アサートします）
 */
nn::Result GetFwVersion(char* pOutStr, size_t size) NN_NOEXCEPT;


/**
 *    @brief        wlanプロセスに対してスリープモードに入るように要求します。@n
 *    スリープモードへの遷移が成功すると、ResultSuccessが返ります。 @n
 *    APとの接続が行われていない状態で要求を発行した場合、無線デバイスは停止し、@n
 *    起床用APIが呼ばれない限りスリープから復帰することはありません。 @n
 *    一方、APとの接続が維持されている状態で要求を発行した場合、無線デバイスはAPとの接続を保ちながら省電力状態に入ります。@n
 *    この状態にあるときは、以下の要因で起床用のGPIOが叩かれます。@n
 *        1. 特定のパケットを受信したとき
 *        2. APとの接続が切れたとき
 *    これにより本体SoCが起床します。ただし、wlanプロセスはまだスリープ状態にあるので、wlanプロセスを動作させたい場合は、@n
 *    起床用APIを呼ぶ必要があります。 @n
 *    また、APIの処理中にAPからの接続切れが生じた場合は、本APIは失敗（ResultSleepFailure）を返し、スリープモードへの遷移は行いません。
 *
 *    @details      実行可能ステート：WlanState_Ready @n
 *                               WlanState_InfraIdle @n
 *                               WlanState_InfraSta (未対応です)@n
 *                               スキャン中や接続試行中、切断中はResultInvalidStateが返ります。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         スリープモードへの遷移に成功。wlanプロセスはWlanState_Sleepに入ります。
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。（developビルドの場合、アサートします）
 *    @retval       ResultSleepFailure    スリープモードへの遷移に失敗。スリープ処理中にAPからの切断が生じた場合、これが返ります。
 *    @retval       ResultNotImplemented  現在はWoWLによるスリープモードは未対応のため、WlanState_InfraSta時に呼ぶとこれが返ります。
 */
nn::Result RequestSleep() NN_NOEXCEPT;


/**
 *    @brief        wlanプロセスに対してスリープモードから復帰するように要求します。@n
 *    スリープモードからの復帰が成功すると、ResultSuccessが返ります。 @n
 *    復帰時、wlanが取る行動は以下です。@n
 *        1. 無線ドライバーに復帰処理を要求し、無線デバイスと本体SoCが通信可能な状態にします。@n
 *        2. wlanのステートはスリープ前のステートに遷移しておきます。@n
 *        3. ResultSuccessを呼び出し元に返します。@n
 *    なお、起床要因がマジックパケットによる起床の場合、上記1. の処理が完了したのち、@n
 *    マジックパケットが受信バッファに格納されます（要確認）。@n
 *    起床要因が切断によるものだった場合、上記3. のあと、無線デバイスから切断イベントが発生するので（要確認）@n
 *    wlanは切断状態に遷移し、ConnectionEventをシグナルします。
 *
 *    @details      実行可能ステート：WlanState_Sleep @n
 *                               WlanState_Ready (ResultAlreadyAwakeが返ります) @n
 *                               WlanState_InfraIdle (ResultAlreadyAwakeが返ります) @n
 *                               WlanState_InfraSta (ResultAlreadyAwakeが返ります)@n
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         スリープモードからの復帰に成功。
 *    @retval       ResultAlreadyAwake    既に起床状態です。
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。（developビルドの場合、アサートします）
 */
nn::Result RequestWakeUp() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief     アクセスポイントへの接続をWPSを使用して行います。@n
 *    PIN方式では @ref GeneratePin で生成したPINコードを使用します。あらかじめ@ref GeneratePin でPINコードの生成を行っておいてください。@n
 *    本関数では、接続試行を開始して、その行為が完了するまでを行います。行為が正常に完了したら本関数は成功を返します。@n
 *    接続が成功したことを意味するものではないので注意して下さい。@n
 *    実際に接続が成功したか失敗したかは @ref GetConnectionStatus で確認して下さい。@n
 *    行為が完了するまで本関数は抜けませんので、完了前にキャンセルをしたい場合は、別スレッドから @ref CancelConnectをコールしてください。@n
 *    WPS接続試行にはあらかじめタイムアウトが設けられており、その時間は120秒となっています。最大でも120秒で本関数は返ることになっています。@n
 *    本関数を抜けた時点で、@ref GetConnectionEvent で取得したイベントはシグナルされています。
 *
 *    @details      実行可能ステート： WlanState_InfraIdle @n
 *
 *    @param[in]    method                WPSの方式です。
 *    @param[in]    pPin                   PINコードです。@ref methodにWpsMethod_Pinを選んだ場合にのみ使用されます。@n
 *                                         @ref GenerateWpsPin 関数で生成されたPINコードを入力するようにして下さい。@n
 *    @param[in]    size                  上記で与えた文字列バッファのサイズです。必ず@ref WpsPinLength + 1となるように指定して下さい。
 *    @param[in]    beaconLostTimeout     接続先アクセスポイントからのBeaconを何秒間受け取れなかったら自動的に切断するかの時間です。単位は秒です。デフォルトは10秒に設定されます。1～30秒の範囲で指定して下さい。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         WPS試行が正常に完了しました。（接続成功を意味するわけではありません）
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。（developビルドの場合、アサートします）
 *    @retval       ResultInvalidArgument 引数の値が不正です。（developビルドの場合、アサートします）
 */
nn::Result ConnectWithWps(WpsMethod method, const char* pPin, size_t size, int beaconLostTimeout = 10) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief     WPSのPIN方式で使用する8桁のPINコードを生成します。
 *
 *    @details      実行可能ステート：全ステート

 *    @param[out]   pin                   生成されたPINコードが返ります。PINコード自体は@ref WpsPinLength桁の数字となりますが、文字列として返すため、NULL文字分のバッファが必要です。
 *    @param[in]    size                  上記で与えた文字列バッファのサイズです。必ず@ref WpsPinLength + 1となるように指定して下さい。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         接続に成功。
 *    @retval       ResultInvalidArgument 引数の値が不正です。（developビルドの場合、アサートします）
 */
nn::Result GenerateWpsPin(char* pin, size_t size) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief     WoWL状態で起床が発生した場合に、どのような起床要因であったかを取得します。
 *
 *    @details      実行可能ステート：WlanState_Ready @n
 *                               WlanState_InfraSta @n
 *                               WlanState_InfraStaScan @n

 *    @param[out]   pOutReason                   起床要因が格納されます。ビット列です。（暫定）
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         成功。
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。（developビルドの場合、アサートします）
 *    @retval       ResultInvalidArgument 引数の値が不正です。（developビルドの場合、アサートします）
 */
nn::Result GetWakeupReason(WowlWakeReason* pOutReason) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        WoWLスリープ中にサーバーとのセッションを維持するために必要となる情報を無線チップにセットします。@n
 *    実際にセットされるタイミングはWoWLスリープに入る直前です。@n
 *    wlanプロセス内で一時的に情報を保管しておき、WoWLスリープに入る時にwlanプロセスが自動的にセットします。@n
 *    既に情報を登録済みでも、上書きすることが出来ます。@n
 *    セットを解除する場合は、@ref RemoveTcpSessionInformation を使って下さい。@n
 *    また、スリープから復帰した時に保管した情報および無線チップ側の情報はwlanにて自動的にクリアされます。@n
 *    WoWLスリープに入る度にセットする必要があるのでご注意下さい。
 *
 *    @details      実行可能ステート：全ステート
 *
 *    @param[in]    dstMac  宛先のMACアドレスです。@n
 *    @param[in]    srcIp   自分のIPアドレスです。@n
 *    @param[in]    dstIp   相手のIPアドレスです。@n
 *    @param[in]    srcPort 自分のポート番号です。@n
 *    @param[in]    dstPort 相手のポート番号です。@n
 *    @param[in]    ackNum  TCPパケットのAck番号です。@n
 *    @param[in]    windowSize TCPのウィンドウサイズです。@n
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess              成功。
 *    @retval       ResultInvalidArgument      不正な引数値です。（developビルドの場合、アサートします）
 */
nn::Result SetTcpSessionInformation(MacAddress dstMac, WlanIpv4Address srcIp, WlanIpv4Address dstIp,
        uint16_t srcPort, uint16_t dstPort, uint32_t ackNum, uint16_t windowSize) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
 *    @brief        @ref SetTcpSessionInformation にてセットした情報を削除します。@n
 *    実際にはwlan内部で保管されている情報が削除されます。何も登録されていない状態で呼んだ場合、何も行なわず成功を返します。@n
 *    無線チップ側にセットされているものはスリープからの復帰時点でクリアされます。またwlanで保管しているものもクリアされます。@n
 *    (wowlスリープかどうかに関わらずスリープからの復帰時点でクリアされることにご注意下さい)@n
 *
 *    @details      実行可能ステート：全ステート
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess            成功。既に削除済みの番号が指定された場合もこれが返ります。
 *    @retval       ResultInvalidArgument    不正な引数値です。（developビルドの場合、アサートします）
 */
nn::Result RemoveTcpSessionInformation() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief     WoWL状態で起床が発生した場合に、どのような起床要因であったかを取得します。（デバッグ用）@n
 *    無線ドライバーから取得した値をそのまま取得します。
 *
 *    @details      実行可能ステート：WlanState_Ready @n
 *                               WlanState_InfraSta @n
 *                               WlanState_InfraStaScan @n
 *
 *    @param[out]   pOutReason                   起床要因が格納されます。ビット列です。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         成功。
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。（developビルドの場合、アサートします）
 *    @retval       ResultInvalidArgument 引数の値が不正です。（developビルドの場合、アサートします）
 */
nn::Result GetWakeupReasonRaw(uint32_t* pOutReason) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief     WoWLモードのスリープ状態からどの要因で起床するかを設定します。（デバッグ用）@n
 *    特にセットしなければ自動的にデフォルトの設定が使用されます。
 *
 *    @details      実行可能ステート：WlanState_Ready @n
 *                               WlanState_InfraSta @n
 *                               WlanState_InfraStaScan @n
 *
 *    @param[out]   reason                   起床要因をセットします。ビット列です。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         成功。
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。（developビルドの場合、アサートします）
 */
nn::Result SetWakeupReasonRaw(uint32_t reason) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief     WoWLの機能を個別に有効・無効化します。（デバッグ用）@n
 *    特にセットしなければ自動的にデフォルトの設定が使用されます。
 *
 *    @details      実行可能ステート：WlanState_Ready @n
 *                               WlanState_InfraSta @n
 *                               WlanState_InfraStaScan @n
 *
 *    @param[out]   features               WoWL機能です。ビット列です。 @ref WowlFeatures を使ってセットします。@n
 *    ここで指定しなかった機能は無効化されることに注意して下さい。
 *    例）@n
 *    uint32_t features;@n
 *    features = (WowlFeatures_ArpOffload | WowlFeatures_TcpKeepAlive);@n
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         成功。
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。（developビルドの場合、アサートします）
 */
nn::Result EnableWowlFeatures(uint32_t features) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief     スリープからの要因別起床回数と、WoWLスリープ中に無線FWがどういう行動を行なったかの統計データを取得します。（デバッグ用）@n
 *    @ref ClearWowlStats でクリアしない限りデータは累計値を返します。
 *
 *    @details      実行可能ステート：全ステート
 *
 *    @param[out]   pCounts                要因別起床回数データです。
 *    @param[out]   pSstats                 WoWLスリープ中の無線FWに関する統計データです。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         成功。
 *    @retval       ResultInvalidArgument 引数の値が不正です。（developビルドの場合、アサートします）
 */
nn::Result GetWowlStats(WowlWakeCount* pCounts, WowlSleepStats* pStats) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief     スリープからの要因別起床回数と、WoWLスリープ中に無線FWがどういう行動を行なったかの統計データを0クリアします。（デバッグ用）@n
 *
 *    @details      実行可能ステート：全ステート
 *
 *    @param[in]    counts                要因別起床回数データをクリアするか指定します。
 *    @param[in]    stats                 WoWLスリープ中の無線FWに関する統計データをクリアするか指定します。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         成功。
 */
nn::Result ClearWowlStats(bool counts, bool stats) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        無線機能の初期化を行います。(検査ツール専用機能)
 *    @details      実行可能ステート：WlanState_Stop (wlanのskip_bootフラグがtrueになっている時のみ)
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         状態遷移に成功。
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。実行可能ステート以外で呼び出した場合このエラーが返ります。
 */
nn::Result InitializeWlanOnSkipBoot() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        無線ドライバーの初期化に失敗したという状況を擬似的に作り出します。(デバッグ専用)
 *    @details      実行可能ステート：全ステート
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         状態遷移に成功。
 */
nn::Result EmulateDriverInitFail() NN_NOEXCEPT;
}
}
}

