﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief Cruiser返り値が共通で利用する API (非公開)
 */

#pragma once

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Result.h>
#include <nn/applet/applet.h>
#include <nn/applet/applet_Types.h>
#include <nn/la/la_Api.h>
#include <nn/web/common/web_CommonTypes.h>

namespace nn { namespace web { namespace common {

class CommonReturnData
{
public:
    /**
     * @brief 情報を格納するために使用するデータのサイズ（バイト）です。
     */
    static const size_t DataSize_Max = nn::applet::StartupParamSizeMax;

    /**
     * @brief ReturnKind
     */
    enum class ReturnKind
    {
        None,                                     //!<
        ExitReason,                               //!<
        LastUrl,                                  //!<
        LastUrlSize,                              //!<
        PostResult,                               //!<
        PostServiceName,                          //!<
        PostServiceNameSize,                      //!<
        PostId,                                   //!<
        PostIdSize,                               //!<
        Max                                       //!< 列挙子の総数です。
    };

    /**
     * @brief セット時のエラータイプ
     */
    enum class ErrorType
    {
        None,         //!< エラー無し
        Invalid,      //!< 無効なデータをセットしようとしました。
        InvalidArg,   //!< 不正な引数が指定されています。
        Conflict,     //!< 同じ種類の異なる格納サイズを持つデータをセットしようとしました。
        OverCapacity, //!< 格納できる最大値を超えています。
        Max           //!< 列挙子の総数です。
    };

    /**
     * @brief データ全体のヘッダ構造体です。
     *
     * @details
     *   8byte使用
     */
    struct Header
    {
        uint16_t count;     //!< 総格納数
        uint16_t _padding;  //!< Padding
        uint32_t kind;      //!< 種類
    };

    /**
     * @brief データのヘッダ構造体です。
     *
     * @details
     *   8byte使用
     */
    struct DataHeader
    {
        uint16_t kind;      //!< 引数の種類
        uint16_t fixedSize; //!< 格納サイズ
        uint32_t padding;   //!< パディング領域

        inline static DataHeader Create()
        {
            DataHeader header;
            header.kind = uint16_t(ReturnKind::None);
            header.fixedSize = 0;
            header.padding = 0;
            return header;
        }
    };

    /**
     * @brief データの構造体です。
     */
    struct Data
    {
        DataHeader header;          //!< ヘッダ
        const uint8_t* pBuffer;     //!< バッファ
    };

    //------------------------------------------------------------------------
    /**
     * @brief データを初期化します。
     *
     * @param[out]  pDst        初期化対象のデータを指定してください。
     * @param[in]   shimKind    起動対象となるShimの種類を指定ください。
     */
    inline static void InitializeData(uint8_t* pDst, const ShimKind shimKind) NN_NOEXCEPT;

    //------------------------------------------------------------------------
    /**
     * @brief 対象バッファにデータをセットします。
     *        同じReturnKindを指定した場合対象領域が上書きされます。
     *
     * @param[out]  pDst        セット対象となるデータを指定してください。
     * @param[in]   returnKind  セットするパラメータの種類を指定してください。
     * @param[in]   fixedSize   セットするパラメータを格納する際の固定サイズを指定してください。
     * @param[in]   pValue      セットするパラメータの値を指定してください。
     *
     * @details セットに失敗した場合アサートします。
     */
    inline static void SetData(uint8_t* pDst, ReturnKind returnKind, size_t fixedSize, const void* pValue) NN_NOEXCEPT;

    //------------------------------------------------------------------------
    /**
     * @brief 対象バッファに文字列をセットします。
     *        同じReturnKindを指定した場合対象領域が上書きされます。
     *
     * @param[out]  pDst        セット対象となるデータを指定してください。
     * @param[in]   returnKind  セットするパラメータの種類を指定してください。
     * @param[in]   fixedSize   セットするパラメータを格納する際の固定サイズを指定してください。
     * @param[in]   pValue      セットするパラメータの値を指定してください。
     *
     * @details セットに失敗した場合アサートします。
     */
    inline static void SetData(uint8_t* pDst, ReturnKind returnKind, size_t fixedSize, const char* pString) NN_NOEXCEPT;

    //------------------------------------------------------------------------
    /**
     * @brief 対象バッファからヘッダを取得します。
     *
     * @param[in]   pSrc    取得対象となるバッファを指定してください。
     *
     * @return 取得に成功した場合ヘッダを返します。
     */
    inline static const Header& GetHeader(const uint8_t* pSrc) NN_NOEXCEPT;

    //------------------------------------------------------------------------
    /**
     * @brief 対象バッファから指定したデータの先頭アドレスを取得します。
     *
     * @param[in]   pSrc    取得対象となるバッファを指定してください。
     * @param[in]   returnKind 取得するデータの種類を指定してください。
     *
     * @return 取得に成功した場合指定したデータの先頭アドレスを返します。
     */
    inline static const uint8_t* GetDataPos(const uint8_t* pSrc, ReturnKind returnKind) NN_NOEXCEPT;

    //------------------------------------------------------------------------
    /**
     * @brief getDataPosで取得したデータの値を取得します。
     *
     * @param[in]   pSrc    取得対象となるデータの先頭アドレスを指定してください。
     *
     * @return 取得に成功した場合データの値へのアドレスを返します。
     */
    inline static const uint8_t* GetDataBuffer(const uint8_t* pSrc) NN_NOEXCEPT;

    //------------------------------------------------------------------------
    /**
     * @brief 対象バッファから指定したデータを取得します。
     *
     * @param[in]   pSrc    取得対象となるバッファを指定してください。
     * @param[in]   returnKind 取得するデータの種類を指定してください。
     *
     * @return 取得に成功した場合指定したデータを返します。
     */
    inline static const Data GetData(const uint8_t* pSrc, ReturnKind returnKind) NN_NOEXCEPT;

    //------------------------------------------------------------------------
    /*
     * @brief 共通パラメータのセット用Utility
     */
    inline static void SetExitReason(uint8_t* pSrc, const ExitReason exitReason) NN_NOEXCEPT;
    inline static void SetLastUrlSize(uint8_t* pSrc, const uint64_t size) NN_NOEXCEPT;
    inline static void SetPostResult(uint8_t* pSrc, const PostResult result) NN_NOEXCEPT;
    inline static void SetPostServiceNameSize(uint8_t* pSrc, const uint64_t size) NN_NOEXCEPT;
    inline static void SetPostIdSize(uint8_t* pSrc, const uint64_t size) NN_NOEXCEPT;

    //------------------------------------------------------------------------
    /*
     * @brief 共通パラメータのゲット用Utility
     */
    inline static const ExitReason* GetExitReason(const uint8_t* pSrc) NN_NOEXCEPT;
    inline static const uint64_t* GetLastUrlSize(const uint8_t* pSrc) NN_NOEXCEPT;
    inline static const PostResult* GetPostResult(const uint8_t* pSrc) NN_NOEXCEPT;
    inline static const uint64_t* GetPostServiceNameSize(const uint8_t* pSrc) NN_NOEXCEPT;
    inline static const uint64_t* GetPostIdSize(const uint8_t* pSrc) NN_NOEXCEPT;

private:
    //------------------------------------------------------------------------
    /**
     * @brief 対象バッファにデータをセットします。
     *        同じReturnKindを指定した場合対象領域が上書きされます。
     *
     * @param[out]  pDst        セット対象となるデータを指定してください。
     * @param[in]   returnKind     セットするパラメータの種類を指定してください。
     * @param[in]   size        セットするパラメータのサイズを指定してください。
     * @param[in]   fixedSize   セットするパラメータを格納する際の固定サイズを指定してください。
     * @param[in]   pValue      セットするパラメータの値を指定してください。
     *
     * @return セットに成功した場合ErrorType::Noneが返ります。
     */
    inline static ErrorType SetData(uint8_t* pDst, ReturnKind returnKind,
        size_t size, size_t fixedSize, const void* pValue) NN_NOEXCEPT;

    //------------------------------------------------------------------------
    /**
     * @brief getDataPosで取得したデータのヘッダを取得します。
     *
     * @param[in]   pSrc    取得対象となるデータの先頭アドレスを指定してください。
     *
     * @return 取得に成功した場合データのヘッダを返します。
     */
    inline static const DataHeader GetDataHeader(const uint8_t* pSrc) NN_NOEXCEPT;

};

//------------------------------------------------------------------------
void CommonReturnData::InitializeData(uint8_t* pDst, const ShimKind shimKind) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
    NN_SDK_ASSERT_NOT_NULL(pDst);
    NN_SDK_ASSERT(shimKind > ShimKind::None && shimKind < ShimKind::Max);

    std::memset(pDst, 0, DataSize_Max);
    auto header = reinterpret_cast<Header*>(pDst);
    header->kind = uint32_t(shimKind);
#else
    NN_UNUSED(pDst);
    NN_UNUSED(shimKind);
#endif
}

//------------------------------------------------------------------------
CommonReturnData::ErrorType CommonReturnData::SetData(uint8_t* pDst, ReturnKind returnKind, size_t size,
size_t fixedSize, const void* pValue) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
    NN_SDK_ASSERT_NOT_NULL(pDst);
    NN_SDK_ASSERT(returnKind > ReturnKind::None && returnKind < ReturnKind::Max);
    NN_SDK_ASSERT(size > 0 && size <= fixedSize);
    NN_SDK_ASSERT_NOT_NULL(pValue);

    // 不正な引数が渡されている場合はセットしない。
    if (!pDst ||
        (returnKind <= ReturnKind::None || returnKind >= ReturnKind::Max) ||
        (size <= 0 | size > fixedSize) ||
        !pValue)
    {
        return ErrorType::InvalidArg;
    }

    // 格納数取得
    auto header = reinterpret_cast<Header*>(pDst);

    size_t offset = sizeof(Header);
    // 既に同じkindのデータがあれば上書きする
    // @note データが固定長である事を前提としているので可変長になったらデータの並び替え等の処理が必要です。
    for (int i = 0; i < header->count; ++i)
    {
        uint8_t* p = pDst + offset;
        auto dataHeader = *(reinterpret_cast<DataHeader*>(p));
        if (dataHeader.kind == uint16_t(returnKind))
        {
            NN_SDK_ASSERT(dataHeader.fixedSize == fixedSize);
            if (dataHeader.fixedSize != fixedSize)
            {
                return ErrorType::Conflict;
            }
            uint8_t* pBuf = p + sizeof(DataHeader);
            std::memset(pBuf, 0, fixedSize);
            std::memcpy(pBuf, pValue, size);
            return ErrorType::None;
        }
        offset += sizeof(DataHeader) + dataHeader.fixedSize;
    }

    // 書き込めるか確認
    if ((offset + fixedSize) >= DataSize_Max)
    {
        return ErrorType::OverCapacity;
    }

    // データ追加
    {
        // 書き込む先頭アドレス取得
        uint8_t* p = pDst + offset;

        auto dataHeader = DataHeader::Create();
        dataHeader.kind = static_cast<uint16_t>(returnKind);
        dataHeader.fixedSize = static_cast<uint16_t>(fixedSize);

        // データ書き込み
        std::memcpy(p, &dataHeader, sizeof(DataHeader));
        p += sizeof(DataHeader);
        std::memcpy(p, pValue, size);
    }
    // 総格納数をインクリメント
    header->count++;

    return ErrorType::None;
#else
    NN_UNUSED(pDst);
    NN_UNUSED(returnKind);
    NN_UNUSED(size);
    NN_UNUSED(fixedSize);
    NN_UNUSED(pValue);
    return ErrorType::Invalid;
#endif
}

//------------------------------------------------------------------------
void CommonReturnData::SetData(uint8_t* pDst, ReturnKind returnKind, size_t fixedSize, const void* pValue) NN_NOEXCEPT
{
    auto errorType = SetData(pDst, returnKind, fixedSize, fixedSize, pValue);
    switch (errorType)
    {
    case ErrorType::None:break;
    case ErrorType::Invalid:
        NN_SDK_ASSERT(0, "Invalid returnKind or invalid pValue");
        break;
    case ErrorType::InvalidArg:
        NN_SDK_ASSERT(0, "Invalid Argument");
        break;
    case ErrorType::Conflict:
        NN_SDK_ASSERT(0, "Arg conflict. Different API set is used");
        break;
    case ErrorType::OverCapacity:
        NN_SDK_ASSERT(0, "OverCapacity");
        break;
    default:
        break;
    }
}

//------------------------------------------------------------------------
void CommonReturnData::SetData(uint8_t* pDst, ReturnKind returnKind, size_t fixedSize, const char* pString) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pString);
    size_t size = std::strlen(pString) + 1;
    if (size <= fixedSize)
    {
        NN_SDK_ASSERT(pString[size - 1] == '\0', "Non null terminated string");

        auto errorType = SetData(pDst, returnKind, size, fixedSize, pString);

        switch (errorType)
        {
        case ErrorType::None:break;
        case ErrorType::Invalid:
            NN_SDK_ASSERT(0, "Invalid returnKind or invalid pValue");
            break;
        case ErrorType::InvalidArg:
            NN_SDK_ASSERT(0, "Invalid Argument");
            break;
        case ErrorType::Conflict:
            NN_SDK_ASSERT(0, "Arg conflict. Different API set is used");
            break;
        case ErrorType::OverCapacity:
            NN_SDK_ASSERT(0, "OverCapacity");
            break;
        default:
            break;
        }
    }
    else
    {
        NN_SDK_ASSERT(0, "Size over");
    }
}

//------------------------------------------------------------------------
const CommonReturnData::Header& CommonReturnData::GetHeader(const uint8_t* pSrc) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
    NN_SDK_ASSERT_NOT_NULL(pSrc);

    return *(reinterpret_cast<const Header*>(pSrc));
#else
    NN_UNUSED(pSrc);
    return Header();
#endif
}

//------------------------------------------------------------------------
const uint8_t* CommonReturnData::GetDataPos(const uint8_t* pSrc, ReturnKind returnKind) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
    NN_SDK_ASSERT_NOT_NULL(pSrc);
    NN_SDK_ASSERT(returnKind > ReturnKind::None && returnKind < ReturnKind::Max);

    auto header = reinterpret_cast<const Header*>(pSrc);

    size_t offset = sizeof(Header);

    for (int i = 0; i < header->count; ++i)
    {
        const uint8_t* p = pSrc + offset;
        auto dataHeader = *(reinterpret_cast<const DataHeader*>(p));
        if (dataHeader.kind == uint16_t(returnKind))
        {
            return p;
        }
        offset += sizeof(DataHeader) + dataHeader.fixedSize;
    }
    return nullptr;
#else
    NN_UNUSED(pSrc);
    NN_UNUSED(returnKind);
    return nullptr;
#endif
}

//------------------------------------------------------------------------
const CommonReturnData::DataHeader CommonReturnData::GetDataHeader(const uint8_t* pSrc) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
    NN_SDK_ASSERT_NOT_NULL(pSrc);

    return *(reinterpret_cast<const DataHeader*>(pSrc));
#else
    NN_UNUSED(pSrc);
    return DataHeader();
#endif
}

//------------------------------------------------------------------------
const uint8_t* CommonReturnData::GetDataBuffer(const uint8_t* pSrc) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
    NN_SDK_ASSERT_NOT_NULL(pSrc);

    return pSrc + sizeof(DataHeader);
#else
    NN_UNUSED(pSrc);
    return nullptr;
#endif
}

//------------------------------------------------------------------------
const CommonReturnData::Data CommonReturnData::GetData(const uint8_t* pSrc, ReturnKind returnKind) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
    NN_SDK_ASSERT_NOT_NULL(pSrc);
    NN_SDK_ASSERT(returnKind > ReturnKind::None && returnKind < ReturnKind::Max);

    const uint8_t* dataPos = GetDataPos(pSrc, returnKind);
    if (dataPos != nullptr)
    {
        Data data;

        data.header = GetDataHeader(dataPos);
        data.pBuffer = GetDataBuffer(dataPos);
        return data;
    }

    return Data();
#else
    NN_UNUSED(pSrc);
    NN_UNUSED(returnKind);
    return Data();
#endif
}

//------------------------------------------------------------------------
void CommonReturnData::SetExitReason(uint8_t* pSrc, const ExitReason exitReason) NN_NOEXCEPT
{
    SetData(pSrc, ReturnKind::ExitReason, sizeof(exitReason), &exitReason);
}

//------------------------------------------------------------------------
void CommonReturnData::SetLastUrlSize(uint8_t* pSrc, const uint64_t size) NN_NOEXCEPT
{
    SetData(pSrc, ReturnKind::LastUrlSize, sizeof(size), &size);
}

//------------------------------------------------------------------------
void CommonReturnData::SetPostResult(uint8_t* pSrc, const PostResult result) NN_NOEXCEPT
{
    SetData(pSrc, ReturnKind::PostResult, sizeof(result), &result);
}

//------------------------------------------------------------------------
void CommonReturnData::SetPostServiceNameSize(uint8_t* pSrc, const uint64_t size) NN_NOEXCEPT
{
    SetData(pSrc, ReturnKind::PostServiceNameSize, sizeof(size), &size);
}

//------------------------------------------------------------------------
void CommonReturnData::SetPostIdSize(uint8_t* pSrc, const uint64_t size) NN_NOEXCEPT
{
    SetData(pSrc, ReturnKind::PostIdSize, sizeof(size), &size);
}

//------------------------------------------------------------------------
const ExitReason* CommonReturnData::GetExitReason(const uint8_t* pSrc) NN_NOEXCEPT
{
    return reinterpret_cast<const ExitReason*>(
        GetData(pSrc, ReturnKind::ExitReason).pBuffer);
}

//------------------------------------------------------------------------
const uint64_t* CommonReturnData::GetLastUrlSize(const uint8_t* pSrc) NN_NOEXCEPT
{
    return reinterpret_cast<const uint64_t*>(
        GetData(pSrc, ReturnKind::LastUrlSize).pBuffer);
}

//------------------------------------------------------------------------
const PostResult* CommonReturnData::GetPostResult(const uint8_t* pSrc) NN_NOEXCEPT
{
    return reinterpret_cast<const PostResult*>(
        GetData(pSrc, ReturnKind::PostResult).pBuffer);
}

//------------------------------------------------------------------------
const uint64_t* CommonReturnData::GetPostServiceNameSize(const uint8_t* pSrc) NN_NOEXCEPT
{
    return reinterpret_cast<const uint64_t*>(
        GetData(pSrc, ReturnKind::PostServiceNameSize).pBuffer);
}

//------------------------------------------------------------------------
const uint64_t* CommonReturnData::GetPostIdSize(const uint8_t* pSrc) NN_NOEXCEPT
{
    return reinterpret_cast<const uint64_t*>(
        GetData(pSrc, ReturnKind::PostIdSize).pBuffer);
}

}}} // namespace nn::web::common
