﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   Layer manipulation functions for the VI library.
 * @details The functions described in this file are for internal developers only
 *          and not to be released to third parties.
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/vi/vi_Types.h>
#include <nn/vi/vi_LayerSettings.h>
#include <nn/vi/vi_PixelFormat.h>

namespace nn { namespace vi {

    /**
     * @brief Creates a new layer on a particular display.
     *
     * @param[out] pOutLayer Layer handle.
     * @param[in]  pDisplay  Display handle.
     * @param[in]  settings  Persistent settings to use for the layer.
     *
     * @return                          Whether the layer was created.
     * @retval ResultSuccess            Layer created successfully.
     * @retval ResultDenied             The process has exceeded the maximum number of layers allowed
     *                                  on @a pDisplay.
     * @retval ResultOperationFailed    An unknown error occurred.
     *
     * @pre  VI library is initialized. @n
     *       @a pOutLayer != nullptr @n
     *       @a pDisplay != nullptr and @a pDisplay is a valid handle.
     * @post A valid handle is written to @a pOutLayer.
     *
     * @details The layer is visible by default.  Until a frame is submitted by the process, the layer will
     *          be blank.
     *
     *          The textures associated with this layer have restrictions due to a limitation with downscaling: @n
     *          @a width  <= 1920 @n
     *          @a height <= 1080 @n
     *          This restriction may be lifted in the future.
     *
     *          A particular display may only support a limited number of layers system-wide.  At minimum, a
     *          process can use a single layer to display content.  To query how many layers can be created by
     *          the process, use ListDisplays().
     */
    Result CreateLayer(Layer** pOutLayer, Display* pDisplay, LayerSettings settings) NN_NOEXCEPT;

    /**
     * @brief Changes the location of the layer in the display's coordinate system.
     *
     * @param[in] pLayer Layer handle.
     * @param[in] x      The requested position's x-coordinate.
     * @param[in] y      The requested position's y-coordinate.
     *
     * @return                       Whether the position of @a pLayer was changed.
     * @retval ResultSuccess         The layer has been repositioned.
     * @retval ResultInvalidRange    The requested position is out of range for this resolution.
     * @retval ResultOperationFailed An unknown error occurred.
     *
     * @pre  VI library is initialized. @n
     *       @a pLayer != nullptr and @a pLayer is a valid handle.
     * @post The layer is now positioned at (@a x, @a y).
     *
     * @details The position provided in (@a x, @a y) denotes the top-left corner of the layer.
     *          The units are pixels.
     *
     *          The layer may only be positioned where it is entirely visible within the display's
     *          current resolution.
     *
     *          The coordinate system used may not match the display's resolution.  Use GetDisplayLogicalResolution
     *          to retrieve the coordinate system used for positioning.
     */
    Result SetLayerPosition(Layer* pLayer, float x, float y) NN_NOEXCEPT;

    /**
     * @brief Changes the size of the layer.
     *
     * @param[in] pLayer Layer handle.
     * @param[in] width  The requested width.
     * @param[in] height The requested height.
     *
     * @return                         Whether @a pLayer's size is @a width by @a height.
     * @retval ResultSuccess           The layer has been resized.
     * @retval ResultInvalidDimensions The display cannot support a layer of this size.
     * @retval ResultInvalidRange      The requested size is out of range for its current position
     *                                 and the display's resolution.
     * @retval ResultOperationFailed   An unknown error occurred.
     *
     * @pre  VI library is initialized. @n
     *       @a pLayer != nullptr and @a pLayer is a valid handle.
     * @post The layer is now @a width by @a height.
     *
     * @details This will only resize the layer displaying content.  If the dimensions of
     *          the layer do not match that of the framebuffer, then the content is subject
     *          to the current scaling mode.
     *
     *          The layer must be entirely visible within the display's current resolution.
     *          Account for the layer's size and position when calling this function.
     *
     *          The coordinate system used may not match the display's resolution.  Use GetDisplayLogicalResolution
     *          to retrieve the coordinate system used for sizing.
     */
    Result SetLayerSize(Layer* pLayer, int width, int height) NN_NOEXCEPT;

    /**
     * @brief Sets the layer's z-ordering for overlapping content.
     *
     * @param[in] pLayer Layer handle.
     * @param[in] z      The requested z-order.
     *
     * @return                       Whether the z-ordering of @a pLayer was changed.
     * @retval ResultSuccess         The new z-order has been applied.
     * @retval ResultInvalidRange    The process cannot use the requested z-order due to system policy.
     * @retval ResultOperationFailed An unknown error occurred.
     *
     * @pre  VI library is initialized. @n
     *       @a pLayer != nullptr and @a pLayer is a valid handle.
     * @post The layer now has the requested z-order.
     *
     * @details The layer will overlap other layers with a z-ordering of (@a z - 1).
     *
     * @details Each process has a specified range of values for @a z.  The set is partitioned
     *          so that all content from both applications and system overlays display properly.
     *          To determine the valid range, use GetZOrderCountMin() and GetZOrderCountMax().
     */
    Result SetLayerZ(Layer* pLayer, int z) NN_NOEXCEPT;

    /**
     * @brief Retrieves the layer's z-ordering.
     *
     * @param[out] pOutZ  The z-order of the layer.
     * @param[in]  pLayer Layer handle.
     *
     * @return                       Whether the z-ordering of @a pLayer was retrieved successfully.
     * @retval ResultSuccess         Successfully retrieved the layer's z-ordering.
     * @retval ResultOperationFailed An unknown error occurred.
     *
     * @pre  VI library is initialized. @n
     *       @a pLayer != nullptr and @a pLayer is a valid handle. @n
     *       @a pOutZ != nullptr
     * @post The layer's z-ordering is written to @a pOutZ.
     */
    Result GetLayerZ(int* pOutZ, const Layer* pLayer) NN_NOEXCEPT;

    /**
     * @brief Sets the alpha value of the entire layer.
     *
     * @param[in] pLayer Layer handle.
     * @param[in] alpha  Requested alpha value.
     *
     * @return                       Whether @a pLayer's alpha matches that of @a alpha.
     * @retval ResultSuccess         Layer's alpha is now the requested value.
     * @retval ResultInvalidRange    The valid range is 0.f <= @a alpha <= 1.f
     * @retval ResultOperationFailed An unknown error occurred.
     *
     * @pre  VI library is initialized. @n
     *       @a pLayer != nullptr and @a pLayer is a valid handle.
     * @post The layer's alpha matches that of @a alpha.
     */
    Result SetLayerAlpha(Layer* pLayer, float alpha) NN_NOEXCEPT;

}}
