﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/vi/sf/vi_DisplayService.sfdl.h>
#include <nn/vi/fbshare/vi_SharedBufferConfig.h>
#include <nn/vi/fbshare/vi_SharedBufferHandle.h>
#include <nn/vi/fbshare/vi_SharedLayerHandle.h>
#include <nn/vi/fbshare/vi_SharedTextureMemoryPool.h>
#include <nn/vi/fbshare/vi_SharedTexturePool.h>
#include <nn/vi/fbshare/vi_SharedNativeWindow.h>

#include <nvn/nvn.h>

namespace nn{ namespace vi{ namespace fbshare{

    // ※アプレットが AcquireTexture を呼び出してから PresentTexture または CancelTexture を呼び出すまでの間、
    //   AM は SharedLayer からバッファを（安全に）デタッチできない。
    //   アプレットは他のメッセージのやり取りとは無関係に少なくとも (Present|Cancel)Texture まで処理をすすめなければならない。
    //
    // ※アプレットが (Present|Cancel)Texture を呼ばなかった場合でも AM が強制的にバッファをデタッチすることができる。
    //   強制デタッチした場合でもアプレットが積んだ GPU コマンドは実行されるため、画面に表示される内容は不定。
    //
    // ※ SharedLayer からバッファがデタッチされた場合、アプレットの処理は AcquireTexture でブロックされる。
    //   このため、アプレットは明示的に SharedLayer のアタッチ／デタッチをハンドルする必要はない。
    //
    // ※アプレットは AcquireTexture が長時間返ってこない場合があることを前提に実装される必要がある。
    // ※もしくはアプレットは PreAcquireTexture を用いてタイムアウト付きで Acquire することができる。
    //   PreAcquireTexutre が成功した場合、次の AcquireTexture の呼出は即座に成功することが保証される。

    class SharedLayerWindowImpl
    {
    public:
        static const int BufferCountMin = NativeWindowTextureCountMin;
        static const int BufferCountMax = NativeWindowTextureCountMax;
        static const SharedTextureMemoryPoolOption MemoryPoolOption = SharedTextureMemoryPoolOption_None;

        typedef nn::sf::SharedPointer<nn::visrv::sf::ISystemDisplayService> ServicePointer;

    public:
        SharedLayerWindowImpl() NN_NOEXCEPT;

        // @pre true
        bool IsInitialized() const NN_NOEXCEPT;

        // @pre IsInitialized
        bool IsTexturePreAcquired() const NN_NOEXCEPT;

        // @pre IsInitialized
        bool IsTextureAcquired() const NN_NOEXCEPT;

        // @pre !IsInitialized
        // @pre bufferCount >= BufferCountMin && bufferCount <= BufferCountMax
        nn::Result Initialize(
            ServicePointer pService,
            NVNdevice* pDevice,
            nn::vi::fbshare::SharedBufferHandle hBuffer,
            nn::vi::fbshare::SharedLayerHandle hLayer,
            int bufferCount,
            NVNformat format
        ) NN_NOEXCEPT;

        // @pre IsInitialized
        void Finalize() NN_NOEXCEPT;

        // 事前に AcquireTexture を試す。
        // この関数が成功した場合、次の AcquireTexture はブロックせずに成功することが保証される。
        // @retval nn::vi::ResultNotReady 失敗した。
        // @details
        // タイムアウト時間中に成功しなかった場合、 ResultNotReady により失敗します。
        // また AM によるデタッチ開始の瞬間にこの関数が呼び出された場合タイムアウトを待たずに ResultNotReady が返る場合があります。
        //
        // 既に PreAcquireTexture に成功していた場合、何もせずに成功を返します。
        //
        // この関数が成功した場合、アプレットは以下のいずれかの処理を行う必要があります。
        // - CancelPreAcquiredTexture()
        // - AcquireTexture() -> PresentTexture()
        // - AcquireTexture() -> CancelTexture()
        nn::Result PreAcquireTexture(nn::TimeSpan timeout) NN_NOEXCEPT;

        // PreAcquireTexture の結果を取り消す。
        // @details
        // PreAcquireTexture が成功してから AcquireTexture を呼び出すまでの間にこの関数を呼び出すことで PreAcquireTexture を呼び出す前の状態に戻します。
        // それ以外のタイミングで呼び出した場合、何もせずに返ります。
        void CancelPreAcquiredTexture() NN_NOEXCEPT;

        // PreAcquireTexture で予約したテクスチャを取得する。
        // @param[out] pOutTextureAvailbaleSync テクスチャへの書込みが可能になったことを通知する Sync オブジェクトを受け取る変数のポインタ
        // @param[out] pOutTextureIndex テクスチャのインデックスを受け取る変数のポインタ。
        //                              内部的なインデックスやテクスチャの有無に関係なく WindowAcquireTexture が呼ばれるたびに 0, 1, 0, 1 ... と取得される
        // @param[out] pOutTexture 書込み先のテクスチャのポインタ。 PreAcquireTexture が成功していなかった場合 nullptr が返る。
        // @details
        // PreAcquireTexture に成功していた場合、獲得済のテクスチャを取得する。
        // それ以外の場合、テクスチャとして nullptr を取得する。
        void AcquirePreAcquiredTexture(NVNsync* pOutTextureAvailableSync, int* pOutTextureIndex, NVNtexture** pOutTexture) NN_NOEXCEPT;

        // nvnQueuePresentTexture 相当の機能を行う
        // @param[in] pQueue テクスチャを提出するコマンドを積むキュー。
        // @param[in] textureIndex AcquireTexture で取得されたテクスチャインデックス。
        // @details
        // テクスチャを SharedLayer に提出するコマンドをキューに積む。
        // SharedLayer からバッファがデタッチされた場合でもこの関数はブロックしない。
        //
        // SharedLayer が AM によって破棄されたとしても、この関数ではエラーを通知しない。
        // 次のフレームの WindowAcquireTexture でエラーが捕捉される。
        void PresentTexture(NVNqueue* pQueue, int textureIndex) NN_NOEXCEPT;

        // Acquire したテクスチャをディスプレイに出力せずに返却する。
        // @param[in] textureIndex AcquireTexture で取得されたテクスチャインデックス。
        // @details
        // AcquireTexture で取得したテクスチャを返却します。
        // この関数は GPU との同期を行いません。返却したテクスチャに GPU の描画が実行された場合、描画が崩れる場合があります。
        // このため、アプレットはこの関数を呼び出す前に返却するテクスチャに対する描画コマンドが発行されていないか、発行済の描画コマンドがすべて完了していることを保証する必要があります。
        //
        // Acquire していない textureIndex に対して呼び出した場合、何もせずに返ります。
        void CancelTexture(int textureIndex) NN_NOEXCEPT;

        // 共有しているテクスチャを取得する。
        // @param[out] pOutTexture テクスチャを受け取る変数へのポインタ
        // @param[in]  sharedTextureIndex 取得するテクスチャのインデックス。
        //             SharedBuffer 中のインデックスで指定する。
        //             AcquireTexture() で取得されるインデックスではない。
        // この関数はテクスチャの所有権に関わらずテクスチャを取得する。
        // テクスチャの内容にアクセスする際は別途同期を取ること。
        void GetSharedTexture(NVNtexture** pOutTexture, int sharedTextureIndex) NN_NOEXCEPT;

        void SetSwapInterval(int value) NN_NOEXCEPT;
        void SetCrop(int x, int y, int width, int height) NN_NOEXCEPT;
        int GetTextureWidth(NVNtexture* pTexture) const NN_NOEXCEPT;
        int GetTextureHeight(NVNtexture* pTexture) const NN_NOEXCEPT;

        // 内部実装用。テクスチャの内部インデックスを取得します。
        nn::Result GetInternalTextureIndex(int* pOutValue, const NVNtexture* pTexture) const NN_NOEXCEPT;

    private:
        void ReconstructInternalWindowImpl(const SharedLayerTextureIndexList& indexList) NN_NOEXCEPT;

    private:
        ServicePointer m_pService;
        NVNdevice* m_pDevice;
        void (*m_pDeviceFunction0)(); // 初期化時に設定される。

        nn::vi::fbshare::SharedBufferHandle m_hBuffer;
        nn::vi::fbshare::SharedLayerHandle  m_hLayer;

        int m_BufferCount;

        // 内部の NVNwindow から取得した textureIndex。
        // 同時に 1 枚しか Acquire できないので 1 つ持っていれば十分。
        // 無効値は -1。
        // Acquire に成功した場合有効値を設定して Present で無効値に戻す。
        int m_InternalTextureIndex;

        // textureIndex として Acquire されたインデックス。
        // Acquire で有効値を設定して Present で無効値に戻す。
        //
        // 有効値は FrameCount % BufferCount で計算される。
        int m_AcquiredTextureIndex;

        // FrameCount の累積値。
        // Present 完了時にカウントアップ。
        // Cancel ではカウントアップしない。
        int64_t m_FrameCount;

        bool m_IsPreAcquired;
        bool m_IsAttachedBufferChanged;
        bool m_IsInternalWindowReconstructionRequired;
        SharedLayerTextureIndexList m_PreAcquiredAttachedTextureIndexList;
        SharedLayerTextureIndexList m_AttachedTextureIndexList;
        NVNtexture* m_AttachedTextureList[BufferCountMax];

        SharedTextureMemoryPool m_MemoryPool;
        SharedTexturePool       m_TexturePool;
        SharedNativeWindow      m_NativeWindow;

        // AttachedTextureCount > 0 のとき初期化済
        NVNwindow               m_InternalWindow;

    };

}}}
