﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file    usb_Pm.h
 * @brief   USB Port Manager Public Interface
 */

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>

#include <nn/sf/sf_HipcClient.h>
#include <nn/sf/sf_ExpHeapAllocator.h>

#include <nn/usb/usb_Result.h>
#include <nn/usb/usb_PmTypes.h>
#include <nn/usb/usb_DiagTypes.h>
#include <nn/usb/pm/sfdl/usb_IPmService.sfdl.h>

namespace nn {
namespace usb {

class PmClient
{
public:
    /**
     * @brief Initialize the USB Port Manager client
     *
     * @retresult{Always success}
     * @endretresult
     *
     * @pre
     *   The client is not initialized
     *
     * @post
     *   The client is initialized
     *
     * @details
     *   This call establishes a connection between the client and the server.
     */
    Result Initialize() NN_NOEXCEPT;

    /**
     * @brief Finalize the USB Port Manager client
     *
     * @retresult{Always success}
     * @endretresult
     *
     * @pre
     *   The client is initialized
     *
     * @post
     *   The client is not initialized
     *
     * @details
     *   Disconnect the client from the server.
     */
    Result Finalize() NN_NOEXCEPT;

    /**
     * @brief Get the system event for Power State Change Event
     *
     * @return A pointer to the system event
     *
     * @pre
     *   The client is initialized
     *
     * @details
     *   The returned pointer points to a system event object managed by this
     *   object. Whenever the power role is changed or a new power contract is
     *   established, this system event will be triggered.
     *
     *   The returned system event should be cleared manually.
     *
     *   Note that the pointer returned will become invalid if this PmClient
     *   object is finalized or destroied.
     */
    nn::os::SystemEventType* GetPowerEvent() NN_NOEXCEPT;

    /**
     * @brief Query the current Power State
     *
     * @param[out] pOutState
     *   Current power state
     *
     * @retresult{Always success}
     * @endretresult
     *
     * @pre
     *   The client is initialized
     *
     * @details
     *   Get the current Power State information, which includes power role,
     *   PDO and RDO. The Power State Change Event is always cleared by this
     *   call.
     */
    Result GetPowerState(UsbPowerState *pOutState) NN_NOEXCEPT;

    /**
     * @brief Get the system event for Data Role Change Event
     *
     * @return A pointer to the system event
     *
     * @pre
     *   The client is initialized
     *
     * @details
     *   The returned pointer points to a system event object managed by this
     *   object. Whenever the data role of USB subsystem is changed, this
     *   system event will be triggered.
     *
     *   The returned system event should be cleared manually.
     *
     *   Note that the pointer returned will become invalid if this PmClient
     *   object is finalized or destroied.
     */
    nn::os::SystemEventType* GetDataEvent() NN_NOEXCEPT;

    /**
     * @brief Query the current Data Role
     *
     * @param[out] pOutRole
     *   Current data role
     *
     * @pre
     *   The client is initialized
     *
     * @details
     *   Get the current Data Role of the USB subsystem. The Data Role Change
     *   Event is always cleared by this call.
     */
    Result GetDataRole(UsbDataRole *pOutRole) NN_NOEXCEPT;

    /**
     * @brief Diagnostic API, internal developer use only
     */
    Result   SetDiagData(DiagData data, uint32_t value) NN_NOEXCEPT;
    uint32_t GetDiagData(DiagData data) NN_NOEXCEPT;

private:
    typedef nn::sf::ExpHeapAllocator MyAllocator;

    MyAllocator                               m_Allocator;
    std::aligned_storage<32 * 1024>::type     m_HeapBuffer;
    nn::lmem::HeapHandle                      m_HeapHandle;

    nn::sf::SharedPointer<pm::IPmService>     m_Handle;

    nn::os::SystemEventType                   m_DataEvent;
    nn::os::SystemEventType                   m_PowerEvent;

private:
    void DumpPowerState(UsbPowerState *pState) NN_NOEXCEPT;
    void DumpPdoFixedSource(nn::usb::pd::Pdo &pdo) NN_NOEXCEPT;
    void DumpPdoVariableSource(nn::usb::pd::Pdo &pdo) NN_NOEXCEPT;
    void DumpPdoBatterySource(nn::usb::pd::Pdo &pdo) NN_NOEXCEPT;
    void DumpPdoFixedSink(nn::usb::pd::Pdo &pdo) NN_NOEXCEPT;
    void DumpPdoVariableSink(nn::usb::pd::Pdo &pdo) NN_NOEXCEPT;
    void DumpPdoBatterySink(nn::usb::pd::Pdo &pdo) NN_NOEXCEPT;
    void DumpRdoFixedOrVariable(nn::usb::pd::Rdo &rdo) NN_NOEXCEPT;
    void DumpRdoBattery(nn::usb::pd::Rdo &rdo) NN_NOEXCEPT;
};

} // end of namespace usb
} // end of namespace nn
