﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file    usb_HostTypes.h
 * @brief   USB Host Stack Public Data Types
 */

#pragma once

#include <nn/usb/usb_Limits.h>
#include <nn/usb/usb_Types.h>

namespace nn { namespace usb {

typedef uint32_t InterfaceHandle;
typedef int32_t  EndpointNumber;
typedef uint32_t DeviceUid;
typedef uint32_t FrameNumber;
typedef int16_t  HubPortNumber;
typedef int32_t  ClientId;

struct XferReport
{
    uint32_t   id;
    Result     result;
    uint32_t   requestedSize;
    uint32_t   transferredSize;
    uint64_t   context;
};

enum SchedulePolicy
{
    SchedulePolicy_Absolute,
    SchedulePolicy_Relative,
};

struct InterfaceProfile
{
    InterfaceHandle         handle;
    DeviceUid               deviceUid;
    int32_t                 altSettingCount;
    UsbInterfaceDescriptor  ifDesc;
    UsbEndpointDescriptor   epInDesc[UsbLimitMaxEndpointPairCount];
    UsbEndpointDescriptor   epOutDesc[UsbLimitMaxEndpointPairCount];
    UsbEndpointCompanionDescriptor epInCompDesc[UsbLimitMaxEndpointPairCount];
    UsbEndpointCompanionDescriptor epOutCompDesc[UsbLimitMaxEndpointPairCount];
};

struct DeviceProfile
{
    char                deviceDebugName[HsLimitMaxDebugNameSize];
    UsbDeviceSpeed      deviceSpeed;
    DeviceUid           deviceUid;
    UsbDeviceDescriptor deviceDesc;
    UsbConfigDescriptor cfgDesc;
    nn::TimeSpan        enumerationTimestamp;
};

/**
 * @brief The data structure / information obtained by interface query
 */
struct InterfaceQueryOutput
{
    InterfaceProfile ifProfile;
    DeviceProfile    deviceProfile;
};

/**
 * @brief The bitmask used to specify which fileds in DeviceFilter are used
 */
enum DeviceFilterMatchFlags
{
    DeviceFilterMatchFlags_Wildcard           = 0x0000,
    DeviceFilterMatchFlags_Vendor             = 0x0001,
    DeviceFilterMatchFlags_Product            = 0x0002,
    DeviceFilterMatchFlags_DeviceLo           = 0x0004,
    DeviceFilterMatchFlags_DeviceHi           = 0x0008,
    DeviceFilterMatchFlags_DeviceClass        = 0x0010,
    DeviceFilterMatchFlags_DeviceSubClass     = 0x0020,
    DeviceFilterMatchFlags_DeviceProtocol     = 0x0040,
    DeviceFilterMatchFlags_InterfaceClass     = 0x0080,
    DeviceFilterMatchFlags_InterfaceSubClass  = 0x0100,
    DeviceFilterMatchFlags_InterfaceProtocol  = 0x0200,
};

/**
 * @brief The filter used to select a subset of available devices
 */
struct DeviceFilter
{
    /* which fields to match against, as defined by DeviceFilterMatchFlags */
    uint16_t   matchFlags;

    /* Used for product specific matches; range is inclusive */
    uint16_t   idVendor;
    uint16_t   idProduct;
    uint16_t   bcdDeviceLo;
    uint16_t   bcdDeviceHi;

    /* Used for device class matches */
    uint8_t    bDeviceClass;
    uint8_t    bDeviceSubClass;
    uint8_t    bDeviceProtocol;

    /* Used for interface class matches */
    uint8_t    bInterfaceClass;
    uint8_t    bInterfaceSubClass;
    uint8_t    bInterfaceProtocol;
};

/**
 * @brief Invalid device filter has gratuitous behavior (passes all)
 */
const DeviceFilter InvalidDeviceFilter = { DeviceFilterMatchFlags_Wildcard,
                                           0,0,0,0, 0,0,0, 0,0,0};

enum TestMode
{
    TestMode_Invalid,
    TestMode_J,
    TestMode_K,
    TestMode_Se0Nak,
    TestMode_Packet,
};

} // end of namespace usb
} // end of namespace nn
