﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/util/util_BitPack.h>

namespace nn {
namespace usb {
namespace pd {

const uint16_t MilliVoltOnConsole = 15000;  // 15V
const uint16_t MilliAmpereOnConsole = 2600; // 2.6A
const uint16_t MilliWattOnConsole = 39000;  // 39W

const uint16_t MilliVoltOnNonPd = 5000;     // 5V

const uint16_t MilliVoltMax = 17000;        // 17V
const uint16_t MilliVoltMin = 5000;         // 5V
const uint16_t MilliAmpereMax = 3000;       // 3.0A
const uint16_t MilliAmpereMin = 0;          // 0A

// PDO / RDO Unit
const uint8_t PdoMilliVoltUnit = 50;
const uint8_t PdoMilliAmpereUnit = 10;
const uint8_t PdoMilliWattUnit = 250;

const uint8_t RdoMilliVoltUnit = PdoMilliVoltUnit;
const uint8_t RdoMilliAmpereUnit = PdoMilliAmpereUnit;
const uint8_t RdoMilliWattUnit = PdoMilliWattUnit;

/**
 * @brief 電源供給タイプ
 */
enum PdoSupplyType
{
    PdoSupplyType_Fixed                                         = 0,
    PdoSupplyType_Battery                                       = 1,
    PdoSupplyType_Variable                                      = 2     // non-battery
};

/**
 * @brief 電力指定
 */
enum RdoObjectPosition
{
    RdoObjectPosition_5VFixedSupplyPdo                          = 1,
    RdoObjectPosition_ReferToNextPdo                            = 2
};

/**
 * @brief PDO (Power Data Object) のビット定義
 */
struct Pdo : public nn::util::BitPack32
{
    // Common
    typedef nn::util::BitPack32::Field< 30,  2, Bit32 > SupplyType;

    // Fixed supply PDO - Source
    typedef nn::util::BitPack32::Field< 28,  1, bool  > UsbSuspendSupported;

    // Fixed supply PDO - Sink
    typedef nn::util::BitPack32::Field< 28,  1, bool  > HigherCapability;

    // Fixed supply PDO - Source/Sink
    typedef nn::util::BitPack32::Field< 29,  1, bool  > DualRolePower;
    typedef nn::util::BitPack32::Field< 27,  1, bool  > UnconstrainedPower;                 // PD2.0 v1.3
    typedef nn::util::BitPack32::Field< 27,  1, bool  > ExternallyPowered;                  // PD2.0 v1.1
    typedef nn::util::BitPack32::Field< 26,  1, bool  > USBCommunicationsCapable;
    typedef nn::util::BitPack32::Field< 25,  1, bool  > DualRoleData;                       // PD2.0 v1.3
    typedef nn::util::BitPack32::Field< 25,  1, bool  > DataRoleSwap;                       // PD2.0 v1.1
    typedef nn::util::BitPack32::Field< 20,  2, Bit32 > PeakCurrent;
    typedef nn::util::BitPack32::Field< 10, 10, Bit32 > Voltage;

    // Fixed/Variable supply PDO - Source/Sink
    typedef nn::util::BitPack32::Field<  0, 10, Bit32 > MaximumCurrent;

    // Variable/Battery Supply PDO - Source/Sink
    typedef nn::util::BitPack32::Field< 20, 10, Bit32 > MaximumVoltage;
    typedef nn::util::BitPack32::Field< 10, 10, Bit32 > MinimumVoltage;

    // Battery Supply PDO - Source/Sink
    typedef nn::util::BitPack32::Field<  0, 10, Bit32 > MaximumAllowablePower;

    // 単位は mV / mA
    int GetVoltage() const NN_NOEXCEPT { return this->Get<Pdo::Voltage>() * PdoMilliVoltUnit; }
    int GetMaximumCurrent() const NN_NOEXCEPT { return this->Get<Pdo::MaximumCurrent>() * PdoMilliAmpereUnit; }
};

/**
 * @brief RDO (Request Data Object) のビット定義
 */
struct Rdo : public nn::util::BitPack32
{
    // Common
    typedef nn::util::BitPack32::Field< 31,  1, bool  > Reserved31;
    typedef nn::util::BitPack32::Field< 28,  3, Bit32 > ObjectPosition;                     // 1: 5V Fixed Supply PDO
                                                                                            // 2: refers to the next PDO
    typedef nn::util::BitPack32::Field< 27,  1, bool  > GiveBackFlag;                       // should be 0 for BM92Txx
    typedef nn::util::BitPack32::Field< 26,  1, bool  > CapabilityMismatch;
    typedef nn::util::BitPack32::Field< 25,  1, bool  > UsbCommunicationsCapable;
    typedef nn::util::BitPack32::Field< 24,  1, bool  > NoUsbSuspend;

    // Fixed and Variable Request Data Object
    typedef nn::util::BitPack32::Field< 10, 10, Bit32 > OperatingCurrent;
    typedef nn::util::BitPack32::Field<  0, 10, Bit32 > MaximumOperatingCurrent;
    typedef nn::util::BitPack32::Field<  0, 10, Bit32 > MinimumOperatingCurrent;            // when GiveBackFlag = 1

    // Battery Request Data Object
    typedef nn::util::BitPack32::Field< 10, 10, Bit32 > OperatingPower;
    typedef nn::util::BitPack32::Field<  0, 10, Bit32 > MaximumOperatingPower;
    typedef nn::util::BitPack32::Field<  0, 10, Bit32 > MinimumOperatingPower;              // when GiveBackFlag = 1

    // 単位は mA
    int GetOperatingCurrent() const NN_NOEXCEPT { return this->Get<Rdo::OperatingCurrent>() * RdoMilliAmpereUnit; }
    int GetMaximumOperatingCurrent() const NN_NOEXCEPT { return this->Get<Rdo::MaximumOperatingCurrent>() * RdoMilliAmpereUnit; }
};

} // pd
} // usb
} // nn
