﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/usb/pd/usb_PdVdm.h>

namespace nn {
namespace usb {
namespace pd {

const uint8_t VdmBrightnessMin = 0;
const uint8_t VdmBrightnessMax = 255;

const uint8_t VdmOnTimeMilliSecondUnit = 10;
const uint8_t VdmOffTimeMilliSecondUnit = 10;
const uint8_t VdmDelayTimeMilliSecondUnit = 10;
const uint8_t VdmPressTimeMilliSecondUnit = 1;

/**
 * @brief PID
 */
const uint16_t VdmCradleId = 0x2003;
const uint16_t VdmAcAdaptorMtmId = 0x2004;
const uint16_t VdmAcAdaptorTbtId = 0x2005;
const uint16_t VdmAcAdaptorDltId = 0x200B;
const uint16_t VdmAcAdaptorFthId = 0x200D;
const uint16_t VdmTableDockId = 0x2014;

/**
 * @brief Firmware Revision
 */
const uint16_t CradlePdcHFwRevisionVa1 = 0x15D0;
const uint16_t CradlePdcHFwRevisionDp1 = 0x0C9A;
const uint16_t CradlePdcHFwRevisionEp2_2 = 0x0B5B;
const uint16_t CradlePdcHFwRevisionEp2 = 0x08D0;
const uint16_t CradlePdcHFwRevisionEp1 = 0x067D;
const uint16_t CradlePdcHFwRevisionFp2 = 0x04E8;

const uint32_t CradleMcuFwRevisionFinalDp1 = 0x01D60210; // 1606020935
const uint32_t CradleMcuFwRevisionDp1 = 0x01B50910; // 1605090900
const uint32_t CradleMcuFwRevisionEp2_2 = 0x01130E10; // 1603140905
const uint32_t CradleMcuFwRevisionEp2 = 0x00D21910; // 1602250915
const uint32_t CradleMcuFwRevisionEp1 = 0x002A1C0F; // 1510281718

/**
 * @brief VDM (Vendor Defined Messages) コマンド
 */
enum VdmCommand
{
    VdmCommand_None                                         = 0,
    VdmCommand_Led                                          = 1,
    VdmCommand_LedReply                                     = 2,
    VdmCommand_Dp2HdmiFwVersion                             = 6,
    VdmCommand_Dp2HdmiFwVersionReply                        = 7,
    VdmCommand_PdcHFwVersion                                = 11,
    VdmCommand_PdcHFwVersionReply                           = 12,
    VdmCommand_PdcAFwVersion                                = 16,
    VdmCommand_PdcAFwVersionReply                           = 17,
    VdmCommand_DeviceErrorNotice                            = 21,
    VdmCommand_DeviceState                                  = 22,
    VdmCommand_DeviceStateReply                             = 23,
    VdmCommand_McuFwVersion                                 = 24,
    VdmCommand_McuFwVersionReply                            = 25,
    VdmCommand_McuFwUpdate                                  = 26,
    VdmCommand_McuFwUpdateReply                             = 27,
    VdmCommand_McuSleepEnable                               = 28,
    VdmCommand_McuSleepEnableReply                          = 29,
    VdmCommand_UsbHubReset                                  = 30,
    VdmCommand_UsbHubResetReply                             = 31,
    VdmCommand_UsbHubVBusDetect                             = 32,
    VdmCommand_UsbHubVBusDetectReply                        = 33,

    VdmCommand_Max                                          = VdmCommand_UsbHubVBusDetectReply
};

enum VdmCommandLength
{
    VdmCommandLength_None                                   = 0,
    VdmCommandLength_Led                                    = 3,
    VdmCommandLength_LedReply                               = 3,
    VdmCommandLength_Dp2HdmiFwVersion                       = 2,
    VdmCommandLength_Dp2HdmiFwVersionReply                  = 3,
    VdmCommandLength_PdcHFwVersion                          = 2,
    VdmCommandLength_PdcHFwVersionReply                     = 3,
    VdmCommandLength_PdcAFwVersion                          = 2,
    VdmCommandLength_PdcAFwVersionReply                     = 3,
    VdmCommandLength_DeviceErrorNotice                      = 3,
    VdmCommandLength_DeviceState                            = 2,
    VdmCommandLength_DeviceStateReply                       = 3,
    VdmCommandLength_McuFwVersion                           = 2,
    VdmCommandLength_McuFwVersionReply                      = 3,
    VdmCommandLength_McuFwUpdate                            = 2,
    VdmCommandLength_McuFwUpdateReply                       = 2,
    VdmCommandLength_McuSleepEnable                         = 2,
    VdmCommandLength_McuSleepEnableReply                    = 2,
    VdmCommandLength_UsbHubReset                            = 2,
    VdmCommandLength_UsbHubResetReply                       = 2,
    VdmCommandLength_UsbHubVBusDetect                       = 2,
    VdmCommandLength_UsbHubVBusDetectReply                  = 2,

    VdmCommandLength_Max                                    = 4 // for DiscoverIdentity
};

/**
 * @brief VDM コマンドタイプ
 */
enum VdmCommandType
{
    VdmCommandType_Initiator                                = 0,
    VdmCommandType_Ack                                      = 1,
    VdmCommandType_Nak                                      = 2,
    VdmCommandType_Busy                                     = 3,

    VdmCommandType_None                                     = -1,
    VdmCommandType_Led                                      = VdmCommandType_Initiator,
    VdmCommandType_LedReply                                 = VdmCommandType_Ack,
    VdmCommandType_Dp2HdmiFwVersion                         = VdmCommandType_Initiator,
    VdmCommandType_Dp2HdmiFwVersionReply                    = VdmCommandType_Ack,
    VdmCommandType_PdcHFwVersion                            = VdmCommandType_Initiator,
    VdmCommandType_PdcHFwVersionReply                       = VdmCommandType_Ack,
    VdmCommandType_PdcAFwVersion                            = VdmCommandType_Initiator,
    VdmCommandType_PdcAFwVersionReply                       = VdmCommandType_Ack,
    VdmCommandType_DeviceErrorNotice                        = VdmCommandType_Initiator,
    VdmCommandType_DeviceState                              = VdmCommandType_Initiator,
    VdmCommandType_DeviceStateReply                         = VdmCommandType_Ack,
    VdmCommandType_McuFwVersion                             = VdmCommandType_Initiator,
    VdmCommandType_McuFwVersionReply                        = VdmCommandType_Ack,
    VdmCommandType_McuFwUpdate                              = VdmCommandType_Initiator,
    VdmCommandType_McuFwUpdateReply                         = VdmCommandType_Ack,
    VdmCommandType_McuSleepEnable                           = VdmCommandType_Initiator,
    VdmCommandType_McuSleepEnableReply                      = VdmCommandType_Ack,
    VdmCommandType_UsbHubReset                              = VdmCommandType_Initiator,
    VdmCommandType_UsbHubResetReply                         = VdmCommandType_Ack,
    VdmCommandType_UsbHubVBusDetect                         = VdmCommandType_Initiator,
    VdmCommandType_UsbHubVBusDetectReply                    = VdmCommandType_Ack
};

/**
 * @brief VDM 通知タイプ
 */
enum VdmNoticeType
{
    VdmNoticeType_None                                      = -1,
    VdmNoticeType_DeviceError                               = 0,

    VdmNoticeType_Num
};


/**
 * @brief VDM リザルト
 */
enum VdmResult
{
    VdmResult_Success                                       = 0,
    VdmResult_Failure                                       = 1
};

/**
 * @brief VDM ディレクション
 */
enum VdmDirection
{
    VdmDirection_FromSwitch                                 = 1,
    VdmDirection_FromCradle                                 = 2
};

/**
 * @brief VDM セット
 */
enum VdmSet
{
    VdmSet_NoSubRequest                                     = 0,
    VdmSet_ReadRequest                                      = 0,
    // Led コマンド用
    VdmSet_WriteRequest                                     = 1,
    // SleepEnable コマンド用
    VdmSet_AwakeRequest                                     = 0,
    VdmSet_SleepRequest                                     = 1,
    // UsbHubVBusDetect コマンド用
    VdmSet_UsbHubDisableRequest                             = 0,
    VdmSet_UsbHubEnableRequest                              = 1
};

/**
 * @brief ファームウェアアップデート状態
 */
enum VdmFwUpdateState
{
    VdmFwUpdateState_Success                                = 0,
    VdmFwUpdateState_Progress                               = 1,
    VdmFwUpdateState_Suspend                                = 2,
    VdmFwUpdateState_SuspendAndTerminate                    = 3
};

/**
 * @brief デバイスタイプ
 */
enum VdmDeviceType
{
    VdmDeviceType_Cradle                                    = 0,
    VdmDeviceType_RelayBox                                  = 1
};

/**
 * @brief 電力供給警告
 */
enum VdmPowerDeriveryWarning
{
    VdmPowerDeriveryWarning_None                            = 0,
    VdmPowerDeriveryWarning_PowerShortage                   = 1,
    VdmPowerDeriveryWarning_AnotherVendor                   = 2
};

/**
 * @brief リレーボックス限定 USB 電力
 */
enum VdmRelayBoxUsbPower
{
    VdmRelayBoxUsbPower_None                                = 0,
    VdmRelayBoxUsbPower_Pc                                  = 1,
    VdmRelayBoxUsbPower_Device                              = 2,
    VdmRelayBoxUsbPower_ErrorA                              = 3,
    VdmRelayBoxUsbPower_ErrorB                              = 4
};

/**
 * @brief FATAL エラー
 */
enum VdmFatalError
{
    VdmFatalError_None                                      = 0,
    VdmFatalError_FatalError                                = 1
};

/**
 * @brief ピンアサイン
 */
enum VdmPinAssignment
{
    VdmPinAssignment_None                                   = 0x00,
    VdmPinAssignment_A                                      = 0x01,
    VdmPinAssignment_B                                      = 0x02,
    VdmPinAssignment_C                                      = 0x04,
    VdmPinAssignment_D                                      = 0x08,
    VdmPinAssignment_E                                      = 0x10,
    VdmPinAssignment_F                                      = 0x20
};

/**
 * @brief 転送シグナル
 */
enum VdmSignalingForTransport
{
    VdmSignalingForTransport_Dp13                           = 0x1,
    VdmSignalingForTransport_UsbGen2                        = 0x2
};

/**
 * @brief ポートケイパビリティ
 */
enum VdmPortCapability
{
    VdmPortCapability_UfpD                                  = 1,
    VdmPortCapability_DfpD                                  = 2,
    VdmPortCapability_Both                                  = 3
};


/**
 * @brief 共通 VDO1 のビット定義
 */
struct VdmCommonVdo1 : public Vdo
{
    typedef nn::util::BitPack32::Field< 16, 16, Bit32 > Command;
    typedef nn::util::BitPack32::Field<  8,  8, Bit32 > Direction;
    // Led / SleepEnable / UsbHubVBusDetect コマンド限定
    typedef nn::util::BitPack32::Field<  0,  1, bool  > Set;
    // UsbHubVBusDetect コマンド限定
    typedef nn::util::BitPack32::Field<  0,  1, bool  > VBusDetectError;
};

/**
 * @brief LED 制御のビット定義
 */
struct VdmLed : public Vdo
{
    typedef nn::util::BitPack32::Field< 24,  8, Bit32 > Brightness;
    typedef nn::util::BitPack32::Field< 16,  8, Bit32 > OnTime;
    typedef nn::util::BitPack32::Field<  8,  8, Bit32 > OffTime;
    typedef nn::util::BitPack32::Field<  0,  8, Bit32 > DelayTime;
};

/**
 * @brief VdmDeviceState / VdmDeviceErrorNotice のビット定義
 */
struct VdmDeviceState : public Vdo
{
    typedef nn::util::BitPack32::Field< 17,  1, Bit32 > DeviceType;
    typedef nn::util::BitPack32::Field< 15,  2, Bit32 > PowerDeriveryWarning;
    typedef nn::util::BitPack32::Field< 12,  3, Bit32 > RelayBoxUsbPower;
    typedef nn::util::BitPack32::Field< 10,  2, Bit32 > FatalError;
};

/**
 * @brief VdmDp2HdmiFwVersion のビット定義
 */
typedef Bit32 VdmDp2HdmiFwVersion;

/**
 * @brief VdmPdcHFwVersion のビット定義
 */
typedef Bit32 VdmPdcHFwVersion;

/**
 * @brief VdmPdcAFwVersion のビット定義
 */
typedef Bit32 VdmPdcAFwVersion;

/**
 * @brief VdmMcuFwVersion のビット定義
 */
typedef Bit32 VdmMcuFwVersion;

/**
 * @brief VDM ディスプレイポート設定のビット定義
 */
struct VdmDisplayPortCapabilities : public Vdo
{
    typedef nn::util::BitPack32::Field< 20,  1, bool  > UfpdPinAssignmentSupportedE;
    typedef nn::util::BitPack32::Field< 19,  1, bool  > UfpdPinAssignmentSupportedD;
    typedef nn::util::BitPack32::Field< 18,  1, bool  > UfpdPinAssignmentSupportedC;
    typedef nn::util::BitPack32::Field< 17,  1, bool  > UfpdPinAssignmentSupportedB;
    typedef nn::util::BitPack32::Field< 16,  1, bool  > UfpdPinAssignmentSupportedA;
    typedef nn::util::BitPack32::Field< 16,  8, Bit32 > UfpdPinAssignmentsSupported;
    typedef nn::util::BitPack32::Field< 13,  1, bool  > DfpdPinAssignmentSupportedF;
    typedef nn::util::BitPack32::Field< 12,  1, bool  > DfpdPinAssignmentSupportedE;
    typedef nn::util::BitPack32::Field< 11,  1, bool  > DfpdPinAssignmentSupportedD;
    typedef nn::util::BitPack32::Field< 10,  1, bool  > DfpdPinAssignmentSupportedC;
    typedef nn::util::BitPack32::Field<  9,  1, bool  > DfpdPinAssignmentSupportedB;
    typedef nn::util::BitPack32::Field<  8,  1, bool  > DfpdPinAssignmentSupportedA;
    typedef nn::util::BitPack32::Field<  8,  8, Bit32 > DfpdPinAssignmentsSupported;
    typedef nn::util::BitPack32::Field<  7,  1, bool  > Usb20SignalingNotUsed;              // 0 = USB r2.0 signaling may be required on A6 – A7 or B6 – B7 while in DisplayPort Configuration.
                                                                                            // 1 = USB r2.0 signaling is not required on A6 – A7 or B6 – B7 while in DisplayPort Configuration.
    typedef nn::util::BitPack32::Field<  6,  1, bool  > ReceptacleIndication;               // 0 = DisplayPort interface is presented on a USB Type-C Plug.
                                                                                            // 1 = DisplayPort interface is presented on a USB Type-C Receptacle.
    typedef nn::util::BitPack32::Field<  2,  4, Bit32 > SignalingForTransport;              // xxx1 = Supports DP v1.3 signaling rates and electrical specification (shall always be set apart from diagnostic purposes).
                                                                                            // xx1x = Supports USB Gen 2 signaling rate and electrical specification.
    typedef nn::util::BitPack32::Field<  0,  2, Bit32 > PortCapability;                     // 00 = RESERVED.
                                                                                            // 01 = UFP_D-capable (including Branch device).
                                                                                            // 10 = DFP_D-capable (including Branch device).
                                                                                            // 11 = Both DFP_D and UFP_D-capable

};

/**
 * @brief Cert Stat VDO のビット定義
 */
struct CertStatVdo : Vdo
{
    typedef nn::util::BitPack32::Field<  0, 20, Bit32 > TestId;
};

/**
 * @brief Product VDO のビット定義
 */
struct ProductVdo : Vdo
{
    typedef nn::util::BitPack32::Field< 16, 16, Bit16 > ProductId;
    typedef nn::util::BitPack32::Field<  0, 16, Bit16 > BcdDevice;
};

/**
 * @brief Discover Identity VDM のビット定義
 */
struct DiscoverIdentityVdm
{
    VdmHeader           m_Header;
    VdmIdHeader         m_IdHeader;
    CertStatVdo         m_CertStatVdo;
    ProductVdo          m_ProductVdo;
};

/**
 * @brief Enter Mode VDM のビット定義
 */
struct EnterModeVdm
{
    VdmHeader           m_Header;
    VdmIdHeader         m_IdHeader;
};

} // pd
} // usb
} // nn
