﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   ソフトウェアキーボードのインラインモードに関する API の宣言
 */

#pragma once

#include <nn/nn_Result.h>

#include <nn/applet/applet_LibraryApplet.h>

#include <nn/swkbd/swkbd_PrivateTypes.h>

#include <nn/swkbd/swkbd_InlineTypes.h>
#include <nn/swkbd/swkbd_InlinePrivateTypes.h>
#include <nn/swkbd/swkbd_InlineKeyboardCustomS.h>
#include <nn/swkbd/swkbd_InlineKeyboardApiForLibraryApplet.h>


/**
 * @namespace nn::swkbd
 * @brief ソフトウェアキーボードアプレット呼び出しの名前空間です。
 */

namespace nn { namespace swkbd {

namespace detail {

    enum InputModeFadeType : uint8_t
    {
        InputModeFadeType_None,   //!< @brief フェードしない
        InputModeFadeType_FadeAndInvisible,   //!< @brief フェードしてからインビジブル
        InputModeFadeType_InvisibleAndFade,   //!< @brief インビジブルにしてからフェード
        InputModeFadeType_Max
    };

    enum InvalidButton : uint32_t
    {
        InvalidButton_AnalogStickL      = nn::swkbd::InvalidButton_AnalogStickL,
        InvalidButton_AnalogStickR      = nn::swkbd::InvalidButton_AnalogStickR,
        InvalidButton_ZL                = nn::swkbd::InvalidButton_ZL,
        InvalidButton_ZR                = nn::swkbd::InvalidButton_ZR,
        // 5～27 :未定義
        InvalidButton_DirectionalButton = 1 << 28,
        InvalidButton_Y                 = nn::swkbd::InvalidButtonS_Y,
        InvalidButton_Minus             = nn::swkbd::InvalidButtonS_Minus,
    };

    /**
     * @brief ソフトウェアをインラインモードで起動するためのクラスです。
     */
    class InlineKeyboardImpl
    {

    public:

        //! @name インラインモードの設定
        //! @{

        InlineKeyboardImpl() NN_NOEXCEPT;
        ~InlineKeyboardImpl() NN_NOEXCEPT;

        //------------------------------------------------------------------------
        /**
         * @brief ソフトウェアキーボードをインラインモードで起動します。
         *
         * @param[in]  arg    表示時に必要な設定パラメータです。
         *
         * @return
         *  ソフトウェアキーボード起動結果を返します。@n
         *  特に問題なければ true を返します。@n
         *  起動に失敗したり、二重に起動しようとした場合は false を返します。
         *
         * @details
         *  ソフトウェアキーボードを起動します。
         */
        bool Initialize( const InitializeArg& arg ) NN_NOEXCEPT;


        /**
         * @brief インラインモードで起動したソフトウェアキーボードを終了します。
         */
        nn::Result Finalize() NN_NOEXCEPT;


        /**
         * @brief ソフトウェアキーボードをインラインモードで起動します。
         * @return
         *  ソフトウェアキーボード起動結果を返します。@n
         *  特に問題なければ true を返します。@n
         *  起動に失敗したり、二重に起動しようとした場合は false を返します。
         */
        bool Launch() NN_NOEXCEPT;

        /**
         * @brief ソフトウェアキーボード全体の音量を設定します。
         */
        void SetVolume( float volume ) NN_NOEXCEPT;

        /**
         * @brief インラインモードで起動したソフトウェアキーボードを表示します。
         */
        void Appear( const AppearArg& arg ) NN_NOEXCEPT;

        /**
         * @brief インラインモードで起動したソフトウェアキーボードを非表示にします。
         */
        void Disappear() NN_NOEXCEPT;

        void NormalizeParams() NN_NOEXCEPT;

        /**
         * @brief 入力テキストとして入力されている文字列情報をソフトウェアキーボードに渡します。
         */
        void SetInputText( const InputText& info ) NN_NOEXCEPT;

        /**
         * @brief テキストカーソル位置をソフトウェアキーボードに渡します。
         */
        void SetCursorPos( int32_t cursorPos ) NN_NOEXCEPT;

        /**
         * @brief ユーザー辞書をソフトウェアキーボードに渡します。
         */
        bool SetUserWordInfo( const UserWordInfo& info ) NN_NOEXCEPT;

        /**
         * @brief ユーザー辞書の設定を解除します。
         */
        bool UnsetUserWordInfo() NN_NOEXCEPT;

        /**
         * @brief ソフトウェアキーボードが返す文字列を UTF-8 にするかどうかを指定します。
         */
        void SetUtf8Mode( bool isUseUtf8 ) NN_NOEXCEPT;

        /**
         * @brief カスタマイズ辞書をソフトウェアキーボードに渡します。
         */
        bool SetCustomizeDic( const void* pBuffer, size_t size, const CustomizeDicInfo& info ) NN_NOEXCEPT;

        /**
         * @brief キートップ背景のアルファ値を指定します。 0.0 ～ 1.0 にクランプします。
         */
        void SetKeytopBgAlpha( float alpha ) NN_NOEXCEPT;

        /**
         * @brief キートップ背景のアルファ値を取得します。
         */
        float GetKeytopBgAlpha() const NN_NOEXCEPT
        {
            return m_KeyboardStorageInfo.keytopBgAlpha;
        }

        /**
         * @brief フッター背景のアルファ値を指定します。 0.0 ～ 1.0 にクランプします。
         */
        void SetFooterBgAlpha( float alpha ) NN_NOEXCEPT;

        /**
         * @brief フッター背景のアルファ値を取得します。
         */
        float GetFooterBgAlpha() const NN_NOEXCEPT
        {
            return m_KeyboardStorageInfo.footerBgAlpha;
        }

        /**
         * @brief キートップのスケールを指定します。
         */
        void SetKeytopScale( float scaleX, float scaleY ) NN_NOEXCEPT;

        /**
         * @brief キートップの Translate を指定します。
         */
        void SetKeytopTranslate( float transX, float transY ) NN_NOEXCEPT;

        /**
         * @brief キートップをフローティング設定するか。(下部にマージンが付きます)
         */
        void SetKeytopAsFloating( bool isFloating ) NN_NOEXCEPT;

        /**
         * @brief キートップをフローティング設定しているかを取得します。
         */
        bool IsKeytopAsFloating() const NN_NOEXCEPT
        {
            return m_KeyboardStorageInfo.isFloatingKeytop;
        }

        /**
         * @brief カスタマイズ辞書の設定を解除します。
         */
        bool UnsetCustomizeDic() NN_NOEXCEPT;

        /**
         * @brief フッターのアルファ値を設定します。
         */
        void SetFooterAlpha( float alpha ) NN_NOEXCEPT;

        /**
         * @brief バルーンのスケールを設定します。
         */
        void SetBalloonScale( float scale ) NN_NOEXCEPT;

        /**
         * @brief フッターのスケールを設定します。
         */
        void SetFooterScalable( bool footerScalable ) NN_NOEXCEPT;

        /**
         * @brief フッターのスケール設定を取得します。
         */
        bool IsFooterScalable() const NN_NOEXCEPT
        {
            return m_KeyboardStorageInfo.isScaleWithFooter;
        }

        /**
         * @brief インプットモードでもアルファを有効にします。
         */
        void SetAlphaEnabledInInputMode( bool enabled ) NN_NOEXCEPT;

        /**
         * @brief InputMode 時のフェード方法の指定
         */
        void SetInputModeFadeType( uint8_t type ) NN_NOEXCEPT;

        /**
         * @brief タッチを無効化します。
         */
        void SetDisableTouch( bool disable ) NN_NOEXCEPT;

        /**
         * @brief USBキーボードからの入力を無効化します。
         */
        void SetDisableUSBKeyboard( bool disable ) NN_NOEXCEPT;

        /**
         * @brief ソフトウェアキーボードのメイン処理です。ゲームフレーム毎に呼んでください。
         */
        State Calc() NN_NOEXCEPT;

        /**
         * @brief 現時点で設定した起動パラメータを元に、ソフトウェアキーボードが使用する画面領域の最大の高さ（ピクセル）を返します。
         */
        int GetMaxHeight() const NN_NOEXCEPT;

        /**
         * @brief 指定した座標がキーボードに使用されているかどうかを判定します。
         * @param[in] x タッチの x 座標です。
         * @param[in] y タッチの y 座標です。
         * @return 指定した座標がキーボードに使用されているかどうかを返します。
         */
        bool IsUsedTouchPointByKeyboard( int32_t x, int32_t y ) NN_NOEXCEPT;

        /**
        * @brief ソフトウェアキーボードが方向ボタンを使用するかどうかを指定します。
        * @param[in] 方向ボタンをソフトウェアキーボード側が使用する場合 true、使用しない場合は false。
        */
        void SetDirectionalButtonAssignEnabled( bool enabled ) NN_NOEXCEPT;

        /**
        * @brief 指定のSEを有効化/無効化します。
        */
        void EnableSe( SeGroup group, bool enable ) NN_NOEXCEPT;

        /**
        * @brief バックスペースを有効化/無効化します。
        */
        void EnableBackspace( bool enable ) NN_NOEXCEPT;

        //! @}

        //! @name インラインモード コールバック設定
        //! @{

        /**
         * @brief 初期化終了情報取得コールバックを設定します。
         */
        void SetFinishedInitializeCallback( FinishedInitializeCallback pCallback ) NN_NOEXCEPT;


        /**
        * @brief ソフトウェアキーボード終了時コールバックを設定します。
        */
        void SetFinishedKeyboardCallback( FinishedKeyboardCallback pCallback ) NN_NOEXCEPT;


        /**
         * @brief 文字列情報取得コールバックを設定します。UTF-16版。
         */
        void SetChangedStringCallback( ChangedStringCallback pCallback ) NN_NOEXCEPT;


        /**
         * @brief 文字列情報取得コールバックを設定します。UTF-8版。
         */
        void SetChangedStringCallbackUtf8( ChangedStringCallbackUtf8 pCallback ) NN_NOEXCEPT;


        /**
         * @brief カーソル位置情報取得コールバックを設定します。UTF-16版。
         */
        void SetMovedCursorCallback( MovedCursorCallback pCallback ) NN_NOEXCEPT;


        /**
         * @brief カーソル位置情報取得コールバックを設定します。UTF-8版。
         */
        void SetMovedCursorCallbackUtf8( MovedCursorCallbackUtf8 pCallback ) NN_NOEXCEPT;


        /**
         * @brief
         */
        void SetMovedTabCallback( MovedTabCallback pCallback ) NN_NOEXCEPT;


        /**
         * @brief 確定閉じ情報取得コールバックを設定します。UTF-16版。
         */
        void SetDecidedEnterCallback( DecidedEnterCallback pCallback ) NN_NOEXCEPT;


        /**
        * @brief 確定閉じ情報取得コールバックを設定します。UTF-8版。
        */
        void SetDecidedEnterCallbackUtf8( DecidedEnterCallbackUtf8 pCallback ) NN_NOEXCEPT;


        /**
         * @brief キャンセル閉じ情報取得コールバックを設定します。
         */
        void SetDecidedCancelCallback( DecidedCancelCallback pCallback ) NN_NOEXCEPT;

        /**
         * @brief ユーザー辞書開放情報取得コールバックを設定します。
         */
        void SetReleasedUserWordInfoCallback( ReleasedUserWordInfoCallback pCallback ) NN_NOEXCEPT;

        /**
         * @brief SWKBD がタッチトリガで使用する矩形を取得します。
         *
         * @param[out] primary   1つめの矩形を取得する出力アドレスです。
         * @param[out] secondary 2つめの矩形を取得する出力アドレスです。 FloatingMode の時にだけ使用します。
         *
         * @return 矩形の数を返します。
         */
        int GetTouchRectangles( Rect* primary, Rect* secondary ) const NN_NOEXCEPT;

        //! @}

        //! @name インラインモード 描画結果の取得
        //! @{

        /**
         * @brief ソフトウェアキーボードのレンダリング結果の画像が要求する画面サイズを取得します。
         */
        static void GetWindowSize( int* pWidth, int* pHeight ) NN_NOEXCEPT;

        /**
         * @brief ソフトウェアキーボードのレンダリング結果の画像を取得するために必要なバッファの大きさやアライメントの制約を取得します。
         */
        static void GetImageMemoryRequirement( size_t* pOutRequiredSize,
            size_t* pOutRequiredAlignment ) NN_NOEXCEPT;

        /**
         * @brief ソフトウェアキーボードのレンダリング結果の画像を取得します。
         */
        bool GetImage( void* pBuffer, size_t bufferSize ) NN_NOEXCEPT;

        //! @}

    private:

        nn::Result SendRequest_( RequestCommand command, const void* pRequestCommandBuffer, size_t requestCommandSize ) NN_NOEXCEPT;

        void CalcFromReplyType_( nn::swkbd::ReplyType replyType,
            nn::applet::StorageHandle handle,
            size_t offset ) NN_NOEXCEPT;
        void CalcFinishedInitialize_( nn::applet::StorageHandle handle, size_t offset ) NN_NOEXCEPT;
        void CalcChangedState_( nn::applet::StorageHandle handle, size_t offset ) NN_NOEXCEPT;
        void CalcChangedString_( nn::applet::StorageHandle handle, size_t offset ) NN_NOEXCEPT;
        void CalcChangedStringUtf8_( nn::applet::StorageHandle handle, size_t offset ) NN_NOEXCEPT;
        void CalcMovedCursor_( nn::applet::StorageHandle handle, size_t offset ) NN_NOEXCEPT;
        void CalcMovedCursorUtf8_( nn::applet::StorageHandle handle, size_t offset ) NN_NOEXCEPT;
        void CalcMovedTab_( nn::applet::StorageHandle handle, size_t offset ) NN_NOEXCEPT;
        void CalcDecidedEnter_( nn::applet::StorageHandle handle, size_t offset ) NN_NOEXCEPT;
        void CalcDecidedEnterUtf8_( nn::applet::StorageHandle handle, size_t offset ) NN_NOEXCEPT;
        void CalcDecidedCancel_( nn::applet::StorageHandle handle, size_t offset ) NN_NOEXCEPT;
        void CalcUnsetCustomizeDic_( nn::applet::StorageHandle handle, size_t offset ) NN_NOEXCEPT;
        void CalcUnsetUserWordInfo_( nn::applet::StorageHandle handle, size_t offset ) NN_NOEXCEPT;

        void WaitForUnsetCustomizeDic_() NN_NOEXCEPT;

        /**
         * @brief アプレットの終了待機・終了処理を行います
         */
        void FinishApplet_() NN_NOEXCEPT;

    private:
        State m_State;
        nn::applet::LibraryAppletHandle m_Handle;
        FinishedInitializeCallback m_FinishedInitializeCallback;
        FinishedKeyboardCallback m_FinishedKeyboardCallback;
        ChangedStringCallback m_ChangedStringCallback;
        ChangedStringCallbackUtf8 m_ChangedStringCallbackUtf8;
        MovedCursorCallback m_MovedCursorCallback;
        MovedCursorCallbackUtf8 m_MovedCursorCallbackUtf8;
        MovedTabCallback m_MovedTabCallback;
        DecidedEnterCallback m_DecidedEnterCallback;
        DecidedEnterCallbackUtf8 m_DecidedEnterCallbackUtf8;
        DecidedCancelCallback m_DecidedCancelCallback;
        ReleasedUserWordInfoCallback m_ReleasedUserWordInfoCallback;

        // アプリから受け取ったもののキャッシュ ( LargeStorage / TransferStorage 以外 )
        KeyboardStorageInfo m_KeyboardStorageInfo;

        // ユーザ辞書情報
        UserWordInfo m_UserWordInfo;

        // カスタマイズ辞書のバッファ
        void* m_CustomizeDicBuffer;
        // カスタマイズ辞書のバッファサイズ
        size_t m_CustomizeDicBufferSize;
        // カスタマイズ辞書のデータ情報
        CustomizeDicInfo m_CustomizeDicInfo;

        // TransferStorage 管理用
        bool m_IsUseTransferStorage;

        // 十字キーの有効状態
        bool m_IsInvalidDirectionalButtons;
    };

} // namespace detail
} } // namespace nn::swkbd

