﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/srepo/detail/service/srepo_Common.h>
#include <nn/srepo/detail/service/core/srepo_DurationManager.h>

namespace nn { namespace srepo { namespace detail { namespace service { namespace core {

/*!
    @brief      継続時間プレイレポートデータの生成器です。
*/
class DurationReportGenerator
{
public:
    static const int KeyValueCount = 64; //!< キー数. TODO 要調整
    static const size_t ReportBufferSize = nn::srepo::ReportBufferSizeMin + (nn::srepo::KeyValueSizeMax * KeyValueCount); //!< バッファサイズ

    /*!
        @brief      コンストラクタ
    */
    DurationReportGenerator() NN_NOEXCEPT
        : m_BootId(GenerateRamdom<uint64_t>())
    {
    }

    /*!
        @brief      初期化
        @param[in] current 現在時刻
    */
    void Initialize(const nn::TimeSpan& current) NN_NOEXCEPT
    {
        m_SystemStateChangeManager.Initialize(current);
        m_PreviousReportedTime = current;
    }

    /*!
        @brief  ユーザーを登録します。
        @param[in] uid      ユーザー
        @param[in] current  現在時刻
    */
    void RegisterUser(const nn::account::Uid& uid, const nn::TimeSpan& current) NN_NOEXCEPT
    {
        m_MultiUserDurationManager.Register(uid, current);
    }

    /*!
        @brief  ユーザーを登録解除します。
        @param[in] uid      ユーザー
    */
    void UnregisterUser(const nn::account::Uid& uid) NN_NOEXCEPT
    {
        m_MultiUserDurationManager.Unregister(uid);
    }

    /*!
        @brief  すべてのユーザーを登録解除します。
    */
    void UnregisterAllUsers() NN_NOEXCEPT
    {
        nn::account::Uid uidList[nn::account::UserCountMax];
        const int UserNum = m_MultiUserDurationManager.GetRegisteredUserList(uidList);
        for(const auto& uid : nn::util::MakeSpan(uidList, UserNum))
        {
            m_MultiUserDurationManager.Unregister(uid);
        }
    }

    /*!
        @brief  レポートバッファの出力データ
        @details
            主にテスト向けに、バッファ内容の確認を行うために利用します。
    */
    struct ReportOutput
    {
        //!< レポートバッファ1つ分
        struct Data
        {
            nn::account::Uid uid; //!< ユーザー
            nn::Bit8 buffer[ReportBufferSize]; //!< レポートバッファ
            size_t actualBufferSize; //!< レポートバッファサイズ

            void Set(const nn::account::Uid& setUid, nn::Bit8* pBuffer, size_t size) NN_NOEXCEPT
            {
                this->uid = setUid;
                std::memcpy(this->buffer, pBuffer, size);
                this->actualBufferSize = size;
            }
        };

        int count; //!< レポートバッファ数
        Data values[nn::account::UserCountMax]; //!< レポートバッファ群
    };

    /*!
        @brief  システムの状態変化が起因のレポートを生成します

        @param[out] pOutReportOutput    レポートバッファの出力(nullptrを許容)
        @param[in]  current         現在時刻
        @param[in]  kind            ステートの種類
        @param[in]  newStateValue   新しいステート値
     */
    template <typename StateValueType>
    nn::Result GenerateSystemTriggerReport(
        ReportOutput* pOutReportOutput,
        const nn::TimeSpan& current,
        SystemDurationKind kind,
        const StateValueType& newStateValue) NN_NOEXCEPT
    {
        const auto newState = CreateSomeValue(newStateValue);
        if(!m_SystemStateChangeManager.IsStateChanged(kind, newState))
        {
            if(pOutReportOutput != nullptr)
            {
                pOutReportOutput->count = 0;
            }
            NN_RESULT_SUCCESS;
        }

        nn::account::Uid uidList[nn::account::UserCountMax];
        const int UserNum = m_MultiUserDurationManager.GetRegisteredUserList(uidList);

        uint64_t seqNoList[nn::account::UserCountMax];
        const int SeqNoNum = UserNum;
        for(int i = 0 ; i < UserNum ; i++)
        {
            // 以降でレポート作成に失敗したとしても、レポート抜けがあることに変わりないのでシーケンシャル番号は必ず進める
            seqNoList[i] = m_MultiUserDurationManager.GetSeqNoAndIncrease(uidList[i]);
        }

        NN_UTIL_SCOPE_EXIT
        {
            m_SystemStateChangeManager.ChangeState(current, kind, newState);
            m_PreviousReportedTime = current;
        };

        // ヘッダ生成
        Header header;
        {
            TriggerString trigger;
            const auto& oldState = m_SystemStateChangeManager.GetState<StateValueType>(kind);
            GenerateTriggerString(&trigger, GetKindName(kind), newState, oldState);

            CreateHeader(&header, current, trigger);
        }

        NN_RESULT_DO(BuildReportBuffer(pOutReportOutput, current, header, uidList, UserNum, seqNoList, SeqNoNum));

        NN_RESULT_SUCCESS;
    }

    /*!
        @brief  ユーザーの状態変化が起因のレポートを生成します

        @param[out] pOutReportOutput    レポートバッファの出力(nullptrを許容)
        @param[in]  uid             ユーザー
        @param[in]  current         現在時刻
        @param[in]  kind            ステートの種類
        @param[in]  newStateValue   新しいステート値
     */
    template <typename StateValueType>
    nn::Result GenerateUserTriggerReport(
        ReportOutput* pOutReportOutput,
        const nn::account::Uid& uid,
        const nn::TimeSpan& current,
        UserDurationKind kind,
        const StateValueType& newStateValue) NN_NOEXCEPT
    {
        const auto newState = CreateSomeValue(newStateValue);
        if(!m_MultiUserDurationManager.IsStateChanged(uid, kind, newState))
        {
            if(pOutReportOutput != nullptr)
            {
                pOutReportOutput->count = 0;
            }
            NN_RESULT_SUCCESS;
        }

        NN_UTIL_SCOPE_EXIT
        {
            m_MultiUserDurationManager.ChangeState(uid, current, kind, newState);
            m_PreviousReportedTime = current;
        };

        // ヘッダ生成
        Header header;
        {
            TriggerString trigger;
            const auto& oldState = m_MultiUserDurationManager.GetState<StateValueType>(uid, kind);
            GenerateTriggerString(&trigger, GetKindName(kind), newState, oldState);

            CreateHeader(&header, current, trigger);
        }

        // 以降でレポート作成に失敗したとしても、レポート抜けがあることに変わりないのでシーケンシャル番号は必ず進める
        uint64_t seqNo = m_MultiUserDurationManager.GetSeqNoAndIncrease(uid);

        NN_RESULT_DO(BuildReportBuffer(pOutReportOutput, current, header, &uid, 1, &seqNo, 1));

        NN_RESULT_SUCCESS;
    }

private:
    const int64_t m_Version = 0;
    const uint64_t m_BootId;
    nn::TimeSpan m_PreviousReportedTime;

    nn::Bit8 m_ReportBuffer[ReportBufferSize];

    SystemStateChangeManager m_SystemStateChangeManager;
    MultiUserDurationManager m_MultiUserDurationManager;

    /*!
        @brief レポートのトリガーを表す文字列
    */
    struct TriggerString
    {
        static const size_t ElementSize = nn::srepo::KeyLengthMax + 1;
        const char* type; //!< レポートの要因となったステート種を表す文字列
        SomeValueString oldState; //!< 変更前のステート値を表す文字列
        SomeValueString newState; //!< 変更後のステート値を表す文字列
    };

    /*!
        @brief レポートのヘッダ
    */
    struct Header
    {
        int64_t version;
        uint64_t bootId;
        nn::TimeSpanType stableDuration;
        nn::TimeSpanType upTime;
        TriggerString trigger;
    };

private:
    template <typename T>
    T GenerateRamdom() const NN_NOEXCEPT
    {
        T ret;
        nn::os::GenerateRandomBytes(&ret, sizeof(T));
        return ret;
    }

    const char* GetKindName(SystemDurationKind kind) const NN_NOEXCEPT;
    const char* GetKindName(UserDurationKind kind) const NN_NOEXCEPT;

    const char* GetStateName(SystemDurationKind kind) const NN_NOEXCEPT;
    const char* GetStateName(UserDurationKind kind) const NN_NOEXCEPT;

    const char* GetDurationName(SystemDurationKind kind) const NN_NOEXCEPT;
    const char* GetDurationName(UserDurationKind kind) const NN_NOEXCEPT;

    void CreateHeader(
        Header* pOut,
        nn::TimeSpanType current,
        const TriggerString& trigger) const NN_NOEXCEPT;

    // トリガー文字列生成デフォルト処理
    void GenerateTriggerString(
        TriggerString* pOut,
        const char* kindName,
        const SomeValueBase& newState,
        const SomeValueBase& oldState) const NN_NOEXCEPT;

    // ControllerStatus専用トリガー文字列生成
    void GenerateTriggerString(
        TriggerString* pOut,
        const char* kindName,
        const SomeValue<ControllerStatus>& newState,
        const SomeValue<ControllerStatus>& oldState) const NN_NOEXCEPT;

    //特定の型 SomeValue<hoge> のトリガー文字列を個別に作るには SomeValue<hoge> を受け付けるGenerateTriggerString(...)を用意する
    // void GenerateTriggerString(
    //     TriggerString* pOut,
    //     const char* kindName,
    //     const SomeValue<hoge>& newState,
    //     const SomeValue<hoge>& oldState) const NN_NOEXCEPT;

    nn::Result BuildReportBuffer(
        ReportOutput* pOut,
        const nn::TimeSpan& current,
        const Header& header,
        const nn::account::Uid* pUidList,
        int uidCount,
        const uint64_t* pSeqNoList,
        int seqNoCount) NN_NOEXCEPT;

    nn::Result BuildSystemReportBuffer(
        size_t* pOutSize,
        nn::Bit8* pBuffer, size_t size, size_t startPosition,
        const nn::TimeSpan& current,
        const Header& header) const NN_NOEXCEPT;

    nn::Result BuildUserReportBuffer(
        size_t* pOutSize,
        nn::Bit8* pBuffer, size_t size, size_t startPosition,
        const nn::TimeSpan& current,
        const nn::account::Uid& uid,
        uint64_t seqNo) const NN_NOEXCEPT;

    nn::Result PushReport(const nn::account::Uid& uid, nn::Bit8* pBuffer, size_t size) const NN_NOEXCEPT;

};

}}}}}
