﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/srepo/detail/service/srepo_Common.h>
#include <nn/srepo/detail/service/core/srepo_StateDuration.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace srepo { namespace detail { namespace service { namespace core {

/*!
    @brief  システムグローバルな継続時間の種類を表す列挙型です。
            システムグローバルでないユーザー単位のものは含まれません。
*/
enum class SystemDurationKind : int32_t
{
    CompletedNetworkRequestType,    //!< 受理済のネットワーク利用要求
    ConnectionOfNotification,       //!< 通知サービスの接続性
    ForegroundProgramId,            //!< フォアグラウンドプログラムID
    DeviceOperationMode,            //!< デバイスの動作モード
    SystemPowerState,               //!< システムのパワーステート
    ControllerStatus,               //!< コントローラ情報

    CountMax,
    // 以降に追加してはいけない
};

/*!
    @brief  ユーザー単位の継続時間の種類を表す列挙型です。
*/
enum class UserDurationKind : int32_t
{
    AccountStatus = 0,
    FriendPresence,

    CountMax,
    // 以降に追加してはいけない
};

/*!
    @brief  複数のステート変化ポイントを管理するクラスです。
    @tparam DurationKind    イベントの種類を表す列挙型を指定してください。
*/
template <typename DurationKind>
class MultiStateChangePointManager
{
public:
    static const int StateDurationCount = static_cast<int>(DurationKind::CountMax); //!< StateDuration構造体の要素数

    /*!
        @brief  初期化します。
        @param[in] current  現在時刻
        @details
                他APIの利用の前に必ず一度だけ実行してください。
    */
    void Initialize(const nn::TimeSpan& current) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(!m_IsInitialized);

        InitializeImpl(current);

        for(auto& value : m_StateChangePointArray)
        {
            NN_SDK_ASSERT(value.pStateValue != nullptr,
                "[srepo::MultiStateChangePointManager] Must set a valid SomeValue class pointer to m_StateChangePointArray[].pStateValue at InitializeImpl()");
            NN_UNUSED(value);
        }

        m_IsInitialized = true;
    }

    /*!
        @brief  ステートの継続時間の配列を生成します。
        @param[out] outArray    出力先配列
        @param[in]  current     現在時刻

        @details
            outArray に DurationKind の列挙型の定義順に値が格納されます。
    */
    void GenerateStateDurationArray(StateDuration outArray[StateDurationCount], const nn::TimeSpan& current) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_IsInitialized);

        int i = 0;
        for(const auto& value : m_StateChangePointArray)
        {
            GenerateStateDuration(&outArray[i++], value, current);
        }
    }

    /*!
        @brief  ステートが変わるかどうか判定します。

        @param[in]  kind        ステートの種類
        @param[in]  newState    新しいステート値
    */
    template <typename StateValuekind>
    bool IsStateChanged(DurationKind kind, const SomeValue<StateValuekind>& newState) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_IsInitialized);

        const auto& previous = GetStateChangePointRef(kind);

        NN_SDK_ASSERT_EQUAL(previous.pStateValue->GetTag(), newState.GetTag());

        return static_cast<const SomeValue<StateValuekind>*>(previous.pStateValue)->GetRef() != newState.GetRef();
    }

    /*!
        @brief  ステートを変更します。

        @param[in]  current     現在時刻
        @param[in]  kind        ステートの種類
        @param[in]  newState    新しいステート値

        @pre
            - IsStateChanged(kind, newState) == true
    */
    template <typename StateValuekind>
    void ChangeState(
        const nn::TimeSpan& current, DurationKind kind, const SomeValue<StateValuekind>& newState) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_IsInitialized);
        NN_SDK_ASSERT(IsStateChanged(kind, newState));

        auto& target = GetStateChangePointRef(kind);

        NN_SDK_ASSERT_EQUAL(target.pStateValue->GetTag(), newState.GetTag());

        target.time = current;
        static_cast<SomeValue<StateValuekind>*>(target.pStateValue)->Set(newState.GetRef());
    }

    /*!
        @brief  ステートを取得します。
        @param[in]  kind    ステートの種類
    */
    template<typename StateType>
    const SomeValue<StateType>& GetState(DurationKind kind) const NN_NOEXCEPT
    {
        auto& target = GetStateChangePointRef(kind);
        NN_SDK_ASSERT_EQUAL(target.pStateValue->GetTag(), SomeValue<StateType>::Tag);
        return *static_cast<const SomeValue<StateType>*>(target.pStateValue);
    }

private:
    /*!
        @brief  ステートの個別の初期化を行います。

        @param[in] current 現在時刻

        @details
            継承先クラスで GetStateChangePointRef(...) で得られる参照に StateChangePoint の初期値を個別に設定してください。
     */
    virtual void InitializeImpl(const nn::TimeSpan& current) NN_NOEXCEPT = 0;

protected:
    StateChangePoint& GetStateChangePointRef(DurationKind kind) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_RANGE(static_cast<int>(kind), 0, StateDurationCount);
        return m_StateChangePointArray[static_cast<int>(kind)];
    }

    const StateChangePoint& GetStateChangePointRef(DurationKind kind) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_RANGE(static_cast<int>(kind), 0, StateDurationCount);
        return m_StateChangePointArray[static_cast<int>(kind)];
    }

private:
    bool m_IsInitialized = false;
    StateChangePoint m_StateChangePointArray[StateDurationCount] {};
};

/*!
    @brief  システムグローバルなステートの変更ポイントを管理するクラスです。
*/
class SystemStateChangeManager : public MultiStateChangePointManager<SystemDurationKind>
{
private:
    virtual void InitializeImpl(const nn::TimeSpan& current) NN_NOEXCEPT NN_OVERRIDE;

    SomeValue<nn::srepo::CompletedNetworkRequestType> m_CompletedNetworkRequestType {nn::srepo::CompletedNetworkRequestType::None};
    SomeValue<bool> m_ConnectionOfNotification {false};
    SomeValue<nn::ncm::ProgramId> m_ForegroundProgramId {nn::ncm::ProgramId::GetInvalidId()};
    SomeValue<nn::srepo::DeviceOperationMode> m_DeviceOperationMode {nn::srepo::DeviceOperationMode::Unknown};
    SomeValue<nn::srepo::SystemPowerState> m_PowerState {nn::srepo::SystemPowerState::FullAwake};
    SomeValue<ControllerStatus> m_ControllerStatus {ControllerStatus::Create(0, 0)};
};

/*!
    @brief  ユーザー1人分のステートの変更ポイントを管理するクラスです。
*/
class OneUserStateChangeManager : public MultiStateChangePointManager<UserDurationKind>
{
private:
    virtual void InitializeImpl(const nn::TimeSpan& current) NN_NOEXCEPT NN_OVERRIDE;

    SomeValue<AccountStatus> m_AccountStatus {AccountStatus::Create(false)};
    SomeValue<nn::srepo::FriendPresence> m_FriendPresence {nn::srepo::FriendPresence::Offline};
};

/*!
    @brief  複数ユーザーの継続時間を管理するクラスです。
*/
class MultiUserDurationManager
{
    NN_DISALLOW_COPY(MultiUserDurationManager);
    NN_DISALLOW_MOVE(MultiUserDurationManager);

public:
    /*!
        @brief 登録可能なユーザー数の最大値です。
    */
    static const int UserRegistrationCountMax = nn::account::UserCountMax;

public:

    MultiUserDurationManager() NN_NOEXCEPT = default;

    /*!
        @brief  ユーザーを登録します。

        @pre
            - uid のユーザーが登録されていない
            - 登録数が合計で UserRegistrationCountMax 未満である

        @post
            - uid のユーザーが登録済である
    */
    void Register(const nn::account::Uid& uid, const nn::TimeSpan& current) NN_NOEXCEPT;

    /*!
        @brief  ユーザーを登録解除します。

        @pre
            - uid のユーザーが登録済である

        @post
            - uid のユーザーが登録されていない

        @details
            本関数実行後は登録解除されたユーザーのステート経過時間は取得できません。
    */
    void Unregister(const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief  登録済ユーザーを取得します。

        @param[out] outUids ユーザーの格納先配列

        @post
            - 0 <= 返り値 < UserRegistrationCountMax

        @return 取得したユーザー数

        @details
                outUids の先頭から返り値分の要素に、登録済ユーザーの Uid が格納されます。
    */
    int GetRegisteredUserList(nn::account::Uid outUids[UserRegistrationCountMax]) const NN_NOEXCEPT;

    /*!
        @brief  継続時間のリストを取得します。

        @param[out] outArray    継続時間の格納先配列
        @param[in]  uid         ユーザー
        @param[in]  current     現在時刻

        @pre
            - uidのユーザーが登録済である
    */
    void GenerateStateDurationArray(
        StateDuration outArray[OneUserStateChangeManager::StateDurationCount],
        const nn::account::Uid& uid,
        const nn::TimeSpan& current) const NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(IsUserRegisterd(uid), "A uid is not registered. (uid:%016llx-%016llx)", uid._data[0], uid._data[1]);
        GetUserData(uid).GetOneUserStateChangeManager().GenerateStateDurationArray(outArray, current);
    }

    /*!
        @brief  ステートが変わるかどうか判定します。

        @param[in]  uid         ユーザー
        @param[in]  kind        ステートの種類
        @param[in]  newState    新しいステート値

        @pre
            - uidのユーザーが登録済である
    */
    template <typename Statekind>
    bool IsStateChanged(const nn::account::Uid& uid, UserDurationKind kind, const SomeValue<Statekind>& newState) const NN_NOEXCEPT
    {
        return GetUserData(uid).GetOneUserStateChangeManager().IsStateChanged(kind, newState);
    }

    /*!
        @brief  ステートを変更します。

        @param[in]  uid         ユーザー
        @param[in]  current     現在時刻
        @param[in]  kind        ステートの種類
        @param[in]  newState    新しいステート値

        @pre
            - uidのユーザーが登録済である
            - IsStateChanged(kind, newState) == true
    */
    template <typename Statekind>
    void ChangeState(
        const nn::account::Uid& uid, const nn::TimeSpan& current, UserDurationKind kind, const SomeValue<Statekind>& newState) NN_NOEXCEPT
    {
        return GetUserData(uid).GetOneUserStateChangeManager().ChangeState(current, kind, newState);
    }

    /*!
        @brief  ステートを取得します。
        @param[in]  uid     ユーザー
        @param[in]  kind    ステートの種類
    */
    template<typename StateType>
    const SomeValue<StateType>& GetState(const nn::account::Uid& uid, UserDurationKind kind) const NN_NOEXCEPT
    {
        return GetUserData(uid).GetOneUserStateChangeManager().GetState<StateType>(kind);
    }

    /*!
        @brief  呼び出しのたびに 1 ずつ増える値を取得します。
        @param[in]  uid     ユーザー

        @details
            uid ごとに、初回呼び出し時は 1 が返ります。

            同じ uid を Unregister() , Register() で再登録をした場合、値は 1 に戻ります。
    */
    uint64_t GetSeqNoAndIncrease(const nn::account::Uid& uid) NN_NOEXCEPT
    {
        return GetUserData(uid).GetSeqNoAndIncrease();
    }

private:
    class UserData
    {
    public:
        UserData(const nn::account::Uid& uid, const nn::TimeSpan& current) NN_NOEXCEPT
            : m_Uid(uid)
            , m_SeqNo(1)
        {
            m_StateChangeManager.Initialize(current);
        }

        OneUserStateChangeManager& GetOneUserStateChangeManager() NN_NOEXCEPT
        {
            return m_StateChangeManager;
        }

        const OneUserStateChangeManager& GetOneUserStateChangeManager() const NN_NOEXCEPT
        {
            return m_StateChangeManager;
        }

        const nn::account::Uid& GetUid() const NN_NOEXCEPT
        {
            return m_Uid;
        }

        uint64_t GetSeqNoAndIncrease() NN_NOEXCEPT
        {
            return m_SeqNo++;
        }

    private:
        nn::account::Uid m_Uid;
        uint64_t m_SeqNo;
        OneUserStateChangeManager m_StateChangeManager;
    };
    nn::util::optional<UserData> m_UserDataArray[UserRegistrationCountMax]; // Registerで構築, Unregisterで破棄

    bool IsUserRegisterd(const nn::account::Uid& uid) const NN_NOEXCEPT;
    const UserData& GetUserData(const nn::account::Uid& uid) const NN_NOEXCEPT;
    UserData& GetUserData(const nn::account::Uid& uid) NN_NOEXCEPT;
};

}}}}}
