﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       TV 設定に関する API の宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_StaticAssert.h>
#include <nn/settings/system/settings_Common.h>
#include <nn/util/util_BitFlagSet.h>

namespace nn { namespace settings { namespace system {

NN_STATIC_ASSERT(sizeof(float) == 4);

/**
 * @brief       TV 設定の制御フラグ定義です。
 */
struct TvFlag
{
    typedef ::nn::util::BitFlagSet<32, TvFlag>::Flag<0>
            Allows4k;               //!< ゲームに 4K の利用を許可するか否か
    typedef ::nn::util::BitFlagSet<32, TvFlag>::Flag<1>
            Allows3d;               //!< ゲームに 3D の利用を許可するか否か
    typedef ::nn::util::BitFlagSet<32, TvFlag>::Flag<2>
            AllowsCec;              //!< CEC を利用するか否か
    typedef ::nn::util::BitFlagSet<32, TvFlag>::Flag<3>
            PreventsScreenBurnIn;   //!< 画面焼け軽減を行うか否か
};

/**
 * @brief       TV 設定の制御フラグ集合を扱う型です。
 */
typedef ::nn::util::BitFlagSet<32, TvFlag> TvFlagSet;

/**
 * @brief       TV の解像度設定を表す列挙型です。
 */
enum TvResolution
{
    TvResolution_Auto,  //!< 自動
    TvResolution_1080p, //!< 1080p
    TvResolution_720p,  //!< 720p
    TvResolution_480p,  //!< 480p
};

/**
 * @brief       HDMI のコンテンツタイプ設定を表す列挙型です。
 */
enum HdmiContentType
{
    HdmiContentType_None,       //!< 無し
    HdmiContentType_Graphics,   //!< グラフィックス
    HdmiContentType_Cinema,     //!< シネマ
    HdmiContentType_Photo,      //!< フォト
    HdmiContentType_Game,       //!< ゲーム
};

/**
 * @brief       RGB レンジ設定を表す列挙型です。
 */
enum RgbRange
{
    RgbRange_Auto,      //!< 自動
    RgbRange_Full,      //!< フル
    RgbRange_Limited,   //!< リミテッド
};

/**
 * @brief       CMU モード設定を表す列挙型です。
 */
enum CmuMode
{
    CmuMode_None,           //!< 無し
    CmuMode_ColorInvert,    //!< 色反転
    CmuMode_HighContrast,   //!< ハイコントラスト
    CmuMode_GrayScale,      //!< グレースケール
};

/**
 * @brief       TV 設定を表す構造体です。
 */
struct TvSettings
{
    TvFlagSet flags;            //!< 制御フラグ
    int32_t tvResolution;       //!< TV の解像度設定
    int32_t hdmiContentType;    //!< HDMI のコンテンツタイプ設定
    int32_t rgbRange;           //!< RGB レンジ設定
    int32_t cmuMode;            //!< CMU モード設定
    uint32_t tvUnderscan;       //!< TV のアンダースキャン設定 [0, 20]
    float tvGamma;              //!< TV のガンマ設定 [0.0, 3.0]
    float contrastRatio;        //!< CMU モード用のコントラスト比設定 [0.0, 1.0]
};

/**
 * @brief       EDID を表す構造体です。
 */
struct Edid
{
    Bit8 data0[128];    //!< EDID ブロック
    Bit8 data1[128];    //!< EDID 拡張ブロック
};

//! @name TV 設定関連 API
//! @{

/**
 * @brief       TV 設定を取得します。
 *
 * @details     TV 設定として指定されている値を取得します。
 *
 * @param[out]  pOutValue                   TV 設定を読み出すバッファ
 *
 * @pre
 *              - pOutValue != nullptr
 */
void GetTvSettings(TvSettings* pOutValue) NN_NOEXCEPT;

/**
 * @brief       TV 設定を変更します。
 *
 * @details     TV 設定を指定された値に変更します。
 *
 * @param[in]   value                       TV 設定
 */
void SetTvSettings(const TvSettings& value) NN_NOEXCEPT;

/**
 * @brief       EDID を取得します。
 *
 * @param[out]  pOutValue                   EDID を読み出すバッファ
 *
 * @pre
 *              - pOutValue != nullptr
 */
void GetEdid(Edid* pOutValue) NN_NOEXCEPT;

/**
 * @brief       EDID を設定します。
 *
 * @param[in]   value                       EDID
 */
void SetEdid(const Edid& value) NN_NOEXCEPT;

//! @}

}}} // nn::settings::system
