﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       6 軸センサーの設定に関する API の宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/settings/system/settings_Common.h>

namespace nn { namespace settings { namespace system {

/**
 * @brief       本体 6 軸センサーの加速度のバイアス値を表す構造体です。
 */
struct ConsoleSixAxisSensorAccelerationBias
{
    float x; //!< 加速度の x 軸バイアス値
    float y; //!< 加速度の y 軸バイアス値
    float z; //!< 加速度の z 軸バイアス値
};

/**
 * @brief       本体 6 軸センサーの角速度のバイアス値を表す構造体です。
 */
struct ConsoleSixAxisSensorAngularVelocityBias
{
    float x; //!< 角速度の x 軸バイアス値
    float y; //!< 角速度の y 軸バイアス値
    float z; //!< 角速度の z 軸バイアス値
};

/**
 * @brief       本体 6 軸センサーの加速度のゲイン値を表す構造体です。
 */
struct ConsoleSixAxisSensorAccelerationGain
{
    float xx; //!< 加速度の x-x ゲイン値
    float xy; //!< 加速度の x-y ゲイン値
    float xz; //!< 加速度の x-z ゲイン値

    float yx; //!< 加速度の y-x ゲイン値
    float yy; //!< 加速度の y-y ゲイン値
    float yz; //!< 加速度の y-z ゲイン値

    float zx; //!< 加速度の z-x ゲイン値
    float zy; //!< 加速度の z-y ゲイン値
    float zz; //!< 加速度の z-z ゲイン値
};

/**
 * @brief       本体 6 軸センサーの角速度のゲイン値を表す構造体です。
 */
struct ConsoleSixAxisSensorAngularVelocityGain
{
    float xx; //!< 角速度の x-x ゲイン値
    float xy; //!< 角速度の x-y ゲイン値
    float xz; //!< 角速度の x-z ゲイン値

    float yx; //!< 角速度の y-x ゲイン値
    float yy; //!< 角速度の y-y ゲイン値
    float yz; //!< 角速度の y-z ゲイン値

    float zx; //!< 角速度の z-x ゲイン値
    float zy; //!< 角速度の z-y ゲイン値
    float zz; //!< 角速度の z-z ゲイン値
};

/**
 * @brief       本体 6 軸センサーの角速度の時間バイアス値を表す構造体です。
 */
struct ConsoleSixAxisSensorAngularVelocityTimeBias
{
    float x; //!< 角速度の時間バイアスの x 軸バイアス値
    float y; //!< 角速度の時間バイアスの y 軸バイアス値
    float z; //!< 角速度の時間バイアスの z 軸バイアス値
};

/**
 * @brief       本体 6 軸センサーの角加速度の係数を表す構造体です。
 */
struct ConsoleSixAxisSensorAngularAcceleration
{
    float xx; //!< 角加速度の係数の x-x ゲイン値
    float xy; //!< 角加速度の係数の x-y ゲイン値
    float xz; //!< 角加速度の係数の x-z ゲイン値

    float yx; //!< 角加速度の係数の y-x ゲイン値
    float yy; //!< 角加速度の係数の y-y ゲイン値
    float yz; //!< 角加速度の係数の y-z ゲイン値

    float zx; //!< 角加速度の係数の z-x ゲイン値
    float zy; //!< 角加速度の係数の z-y ゲイン値
    float zz; //!< 角加速度の係数の z-z ゲイン値
};

//! @name 6 軸センサーの設定関連 API
//! @{

/**
 * @brief       本体 6 軸センサーの加速度のバイアス値を取得します。
 *
 * @param[out]  pOutValue             本体 6 軸センサーの加速度のバイアス値を読み出すバッファ
 *
 * @pre
 *              - pOutValue != nullptr
 */
void GetConsoleSixAxisSensorAccelerationBias(
    ConsoleSixAxisSensorAccelerationBias* pOutValue) NN_NOEXCEPT;

/**
 * @brief       本体 6 軸センサーの加速度のバイアス値を変更します。
 *
 * @details     本体 6 軸センサーの加速度のバイアス値を指定された値に変更します。
 *
 * @param[in]   value                 本体 6 軸センサーの加速度のバイアス値
 */
void SetConsoleSixAxisSensorAccelerationBias(
    const ConsoleSixAxisSensorAccelerationBias& value) NN_NOEXCEPT;

/**
 * @brief       本体 6 軸センサーの角速度のバイアス値を取得します。
 *
 * @param[out]  pOutValue             本体 6 軸センサーの角速度のバイアス値を読み出すバッファ
 *
 * @pre
 *              - pOutValue != nullptr
 */
void GetConsoleSixAxisSensorAngularVelocityBias(
    ConsoleSixAxisSensorAngularVelocityBias* pOutValue) NN_NOEXCEPT;

/**
 * @brief       本体 6 軸センサーの角速度のバイアス値を変更します。
 *
 * @details     本体 6 軸センサーの角速度のバイアス値を指定された値に変更します。
 *
 * @param[in]   value                 本体 6 軸センサーの角速度のバイアス値
 */
void SetConsoleSixAxisSensorAngularVelocityBias(
    const ConsoleSixAxisSensorAngularVelocityBias& value) NN_NOEXCEPT;

/**
 * @brief       本体 6 軸センサーの加速度のゲイン値を取得します。
 *
 * @param[out]  pOutValue             本体 6 軸センサーの加速度のゲイン値を読み出すバッファ
 *
 * @pre
 *              - pOutValue != nullptr
 */
void GetConsoleSixAxisSensorAccelerationGain(
    ConsoleSixAxisSensorAccelerationGain* pOutValue) NN_NOEXCEPT;

/**
 * @brief       本体 6 軸センサーの加速度のゲイン値を変更します。
 *
 * @details     本体 6 軸センサーの加速度のゲイン値を指定された値に変更します。
 *
 * @param[in]   value                 本体 6 軸センサーの加速度のゲイン値
 */
void SetConsoleSixAxisSensorAccelerationGain(
    const ConsoleSixAxisSensorAccelerationGain& value) NN_NOEXCEPT;

/**
 * @brief       本体 6 軸センサーの角速度のゲイン値を取得します。
 *
 * @param[out]  pOutValue             本体 6 軸センサーの角速度のゲイン値を読み出すバッファ
 *
 * @pre
 *              - pOutValue != nullptr
 */
void GetConsoleSixAxisSensorAngularVelocityGain(
    ConsoleSixAxisSensorAngularVelocityGain* pOutValue) NN_NOEXCEPT;

/**
 * @brief       本体 6 軸センサーの角速度のゲイン値を変更します。
 *
 * @details     本体 6 軸センサーの角速度のゲイン値を指定された値に変更します。
 *
 * @param[in]   value                 本体 6 軸センサーの角速度のゲイン値
 */
void SetConsoleSixAxisSensorAngularVelocityGain(
    const ConsoleSixAxisSensorAngularVelocityGain& value) NN_NOEXCEPT;

/**
 * @brief       本体 6 軸センサーの角速度の時間バイアス値を取得します。
 *
 * @param[out]  pOutValue             本体 6 軸センサーの角速度の時間バイアス値を読み出すバッファ
 *
 * @pre
 *              - pOutValue != nullptr
 */
void GetConsoleSixAxisSensorAngularVelocityTimeBias(
    ConsoleSixAxisSensorAngularVelocityTimeBias* pOutValue) NN_NOEXCEPT;

/**
 * @brief       本体 6 軸センサーの角速度の時間バイアス値を変更します。
 *
 * @details     本体 6 軸センサーの角速度の時間バイアス値を指定された値に変更します。
 *
 * @param[in]   value                 本体 6 軸センサーの角速度の時間バイアス値
 */
void SetConsoleSixAxisSensorAngularVelocityTimeBias(
    const ConsoleSixAxisSensorAngularVelocityTimeBias& value) NN_NOEXCEPT;

/**
 * @brief       本体 6 軸センサーの角加速度の係数を取得します。
 *
 * @param[out]  pOutValue             本体 6 軸センサーの角加速度の係数を読み出すバッファ
 *
 * @pre
 *              - pOutValue != nullptr
 */
void GetConsoleSixAxisSensorAngularAcceleration(
    ConsoleSixAxisSensorAngularAcceleration* pOutValue) NN_NOEXCEPT;

/**
 * @brief       本体 6 軸センサーの角加速度の係数を変更します。
 *
 * @details     本体 6 軸センサーの角加速度の係数を指定された値に変更します。
 *
 * @param[in]   value                 本体 6 軸センサーの角加速度の係数
 */
void SetConsoleSixAxisSensorAngularAcceleration(
    const ConsoleSixAxisSensorAngularAcceleration& value) NN_NOEXCEPT;

//! @}

}}} // namespace nn::settings::system
