﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 *  @file
 *  @brief   言語設定に関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>

namespace nn { namespace settings {

/**
 * @brief       言語を表す列挙体です。
 */
enum Language
{
    Language_Japanese,              //!< 日本語 (ja)
    Language_AmericanEnglish,       //!< 米国英語 (en-US)
    Language_French,                //!< フランス語 (fr)
    Language_German,                //!< ドイツ語 (de)
    Language_Italian,               //!< イタリア語 (it)
    Language_Spanish,               //!< スペイン語 (es)
    Language_Chinese,               //!< 中国語 (zh-CN)
    Language_Korean,                //!< 韓国語 (ko)
    Language_Dutch,                 //!< オランダ語 (nl)
    Language_Portuguese,            //!< ポルトガル語 (pt)
    Language_Russian,               //!< ロシア語 (ru)
    Language_Taiwanese,             //!< 台湾語 (zh-TW)
    Language_BritishEnglish,        //!< 英国英語 (en-GB)
    Language_CanadianFrench,        //!< カナダフランス語 (fr-CA)
    Language_LatinAmericanSpanish,  //!< 中南米スペイン語 (es-419)
    Language_SimplifiedChinese,     //!< 簡体字中国語 (zh-Hans)
    Language_TraditionalChinese,    //!< 繁体字中国語 (zh-Hant)
};

/**
 * @brief       言語コードを表す構造体です。
 */
struct LanguageCode
{
    char string[8]; //!< IETF 言語タグ形式の言語コード（NULL 終端）

//! @name コンストラクタ
//! @{

    /**
     * @brief       言語コードを生成します。
     *
     * @param[in]   value                       言語
     *
     * @return      指定された言語の言語コードを返します。
     *
     * @pre
     *              - value は有効な言語
     */
    static LanguageCode Make(Language value) NN_NOEXCEPT;

//! @}

    /**
     * @brief       言語コードが一致しているか否かを判断します。
     *
     * @param[in]   lhs                         言語コード
     * @param[in]   rhs                         言語コード
     *
     * @return      一致しているなら true そうでなければ false を返します。
     */
    friend bool operator==(
        const LanguageCode& lhs, const LanguageCode& rhs) NN_NOEXCEPT;

    /**
     * @brief       言語コードが不一致か否かを判断します。
     *
     * @param[in]   lhs                         言語コード
     * @param[in]   rhs                         言語コード
     *
     * @return      不一致なら true そうでなければ false を返します。
     */
    friend bool operator!=(
        const LanguageCode& lhs, const LanguageCode& rhs) NN_NOEXCEPT
    {
        return !(lhs == rhs);
    }

    /**
     * @brief       言語が一致しているか否かを判断します。
     *
     * @param[in]   lhs                         言語
     * @param[in]   rhs                         言語コード
     *
     * @return      一致しているなら true そうでなければ false を返します。
     */
    friend bool operator==(
        const Language& lhs, const LanguageCode& rhs) NN_NOEXCEPT
    {
        return LanguageCode::Make(lhs) == rhs;
    }

    /**
     * @brief       言語が一致しているか否かを判断します。
     *
     * @param[in]   lhs                         言語コード
     * @param[in]   rhs                         言語
     *
     * @return      一致しているなら true そうでなければ false を返します。
     */
    friend bool operator==(
        const LanguageCode& lhs, const Language& rhs) NN_NOEXCEPT
    {
        return rhs == lhs;
    }

    /**
     * @brief       言語が不一致か否かを判断します。
     *
     * @param[in]   lhs                         言語
     * @param[in]   rhs                         言語コード
     *
     * @return      不一致なら true そうでなければ false を返します。
     */
    friend bool operator!=(
        const Language& lhs, const LanguageCode& rhs) NN_NOEXCEPT
    {
        return !(lhs == rhs);
    }

    /**
     * @brief       言語が不一致か否かを判断します。
     *
     * @param[in]   lhs                         言語コード
     * @param[in]   rhs                         言語
     *
     * @return      不一致なら true そうでなければ false を返します。
     */
    friend bool operator!=(
        const LanguageCode& lhs, const Language& rhs) NN_NOEXCEPT
    {
        return !(rhs == lhs);
    }
};

//! @name 言語設定関連 API
//! @{

/**
 * @brief       利用可能な言語コードの数を返します。
 *
 * @details     言語設定として利用可能な言語コードの数を取得します。
 *              利用可能な言語コードについては GetAvailableLanguageCodes() を参照してください。
 *
 * @return      利用可能な言語コードの数を返します。
 */
int GetAvailableLanguageCodeCount() NN_NOEXCEPT;

/**
 * @brief       利用可能な言語コードの一覧を取得します。
 *
 * @details     言語設定として利用可能な言語コードの配列を取得します。
 *              nn::settings::Language で宣言された言語と対応する言語コードは常に配列に含まれます。
 *              また、言語コードの取得順も nn::settings::Language での宣言順に従います。
 *
 * @param[out]  outCodes                    言語コードを読み出すバッファ（配列）
 * @param[in]   count                       outCodes の数
 *
 * @return      取得した言語コードの数を返します。
 *
 * @pre
 *              - outCodes != nullptr
 * @post
 *              - i が 0 以上かつ戻り値未満ならば、 outCodes[i] は有効な言語コード
 */
int GetAvailableLanguageCodes(LanguageCode outCodes[], int count) NN_NOEXCEPT;

//! @}

}} // nn::settings
