﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       デバイス秘密鍵の生産時設定に関する API の宣言
 */

#pragma once

#include <nn/nn_BitTypes.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/settings/factory/settings_Result.h>

namespace nn { namespace settings { namespace factory {

/**
 * @brief       デバイス秘密鍵 (ECC-B233 版) を表す構造体です。
 */
struct EccB233DeviceKey
{
    uint32_t size;          //!< デバイス秘密鍵 (ECC-B233 版) のバイト数
    ::nn::Bit8 data[84];    //!< デバイス秘密鍵 (ECC-B233 版)
};

/**
 * @brief       デバイス秘密鍵 (RSA-2048 版) を表す構造体です。
 */
struct Rsa2048DeviceKey
{
    uint32_t size;          //!< デバイス秘密鍵 (RSA-2048 版) のバイト数
    ::nn::Bit8 data[580];   //!< デバイス秘密鍵 (RSA-2048 版)
};

/**
 * @brief       デバイス登録用のデバイス秘密鍵 (ECC-B233 版) を取得します。
 *
 * @param[out]  pOutValue                    デバイス秘密鍵を読みだすバッファ
 *
 * @retresult
 * @handleresult{nn::settings::factory::ResultCalibrationDataFileSystemCorrupted}
 * @handleresult{nn::settings::factory::ResultCalibrationDataCrcError}
 * @endretresult
 *
 * @pre
 *              - pOutValue != nullptr
 */
::nn::Result GetEciDeviceKey(EccB233DeviceKey* pOutValue) NN_NOEXCEPT;

/**
 * @brief       デバイス登録用のデバイス秘密鍵 (RSA-2048 版) を取得します。
 *
 * @param[out]  pOutValue                    デバイス秘密鍵を読みだすバッファ
 *
 * @retresult
 * @handleresult{nn::settings::factory::ResultCalibrationDataFileSystemCorrupted}
 * @handleresult{nn::settings::factory::ResultCalibrationDataCrcError}
 * @endretresult
 *
 * @pre
 *              - pOutValue != nullptr
 */
::nn::Result GetEciDeviceKey(Rsa2048DeviceKey* pOutValue) NN_NOEXCEPT;

/**
 * @brief       eTicket 発行用のデバイス秘密鍵を取得します。
 *
 * @param[out]  pOutValue                    デバイス秘密鍵を読みだすバッファ
 *
 * @retresult
 * @handleresult{nn::settings::factory::ResultCalibrationDataFileSystemCorrupted}
 * @handleresult{nn::settings::factory::ResultCalibrationDataCrcError}
 * @endretresult
 *
 * @pre
 *              - pOutValue != nullptr
 */
::nn::Result GetEticketDeviceKey(Rsa2048DeviceKey* pOutValue) NN_NOEXCEPT;

}}} // nn::settings::factory
