﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief PSM ライブラリの生産工程用の API 宣言。
 */

#pragma once

#include <nn/nn_Macro.h>

#include <nn/psm/psm_ManufactureTypes.h>

namespace nn { namespace psm {

/**
 * @brief      充電を有効化します。
 * @pre
 *  - PSM ライブラリが初期化済の状態である
 * @details    充電を有効化し、本体に充電器が接続されると自動的に充電を開始する状態にします。
 */
void EnableBatteryCharging() NN_NOEXCEPT;

/**
 * @brief      充電を無効化します。
 * @pre
 *  - PSM ライブラリが初期化済の状態である
 * @details    充電を無効化し、本体に充電器が接続されても充電を開始しない状態にします。
 *             充電の無効化は EnableBatteryCharging() を呼び出すか、SoC の電源が切れるまで続きます。
 */
void DisableBatteryCharging() NN_NOEXCEPT;

/**
 * @brief      充電が有効化されているかどうかを取得します。
 * @return     充電が有効化されているかどうか
 * @pre
 *  - PSM ライブラリが初期化済の状態である
 * @details    充電が有効化されているかどうかを取得します。
 */
bool IsBatteryChargingEnabled() NN_NOEXCEPT;

/**
 * @brief      電池劣化量を取得します。
 * @return     電池劣化量 [%]
 * @pre
 *  - PSM ライブラリが初期化済の状態である
 * @details    電池劣化量を取得します。
 */
double GetBatteryAgePercentage() NN_NOEXCEPT;

/**
 * @brief      給電経路制御機能を有効にします。
 * @pre
 *  - PSM ライブラリが初期化済の状態である
 * @details    PSM ライブラリの給電経路制御機能を有効にします。@n
 *             この機能は PSM ライブラリの生産工程用ビルドでのみ使用可能です。
 */
void EnableSupplyRouteControl() NN_NOEXCEPT;

/**
 * @brief      給電経路制御機能を無効にします。
 * @pre
 *  - PSM ライブラリが初期化済の状態である
 * @details    PSM ライブラリの給電経路制御機能を無効にします。@n
 *             この機能は PSM ライブラリの生産工程用ビルドでのみ使用可能です。
 */
void DisableSupplyRouteControl() NN_NOEXCEPT;

/**
 * @brief      給電経路を設定します。
 * @param[in]  @ref nn::psm::SupplyRoute 型の給電経路
 * @pre
 *  - PSM ライブラリが初期化済の状態である
 * @details    PSM ライブラリの給電経路を設定します。
 *             この機能は PSM ライブラリの生産工程用ビルドでのみ使用可能です。@n
 *             設定した給電経路が意図せず切り替わることを避けるために、
 *             この機能を使用する前に @ref nn::psm::DisableSupplyRouteControl を使用して
 *             PSM ライブラリの給電経路制御機能を無効にすることを推奨します。@n
 *             この機能を使用した後は @ref nn::psm::EnableSupplyRouteControl を使用して
 *             PSM ライブラリの給電経路制御機能を有効に戻すことを推奨します。
 */
void SetSupplyRoute(SupplyRoute supplyRoute) NN_NOEXCEPT;

}} // namespace nn::psm
