﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @file
    @brief  Public header defining PSC API exposed to PM control service
*/

#pragma once

#include <nn/os/os_SystemEvent.h>
#include <nn/psc.h>
#include <nn/psc/sfdl/psc.sfdl.h>
#include <nn/psc/psc_PmControlTypes.h>

namespace nn      {
namespace psc     {

/**
    @brief      Power control module.

    @details    This class exposes PSC interface to higher level PM control service.
                Using this class, such service may dispatch system
                wide requests to registered PM modules.

                Sample code:

                @code
                #include <nn/psc.h>
                #include <nn/psc/control/psc_PmControl.h>

                nn::psc::control::PmControl g_Control;

                void ControllerInitialize()
                {
                    nn::Result result = g_Control.Initialize(nn::os::EventClearMode_ManualClear);
                    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                }

                void ControllerFinalize()
                {
                    g_Control.Finalize();
                }

                nn::Result ControllerDispatch(nn::psc::PmState state, nn::psc::PmFlagSet flags, nn::psc::PmTransitionOrder order)
                {
                    nn::Result result = g_Control.DispatchRequest(state, flags, order);
                    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

                    g_Control.GetEventPointer()->Wait();
                    g_Control.GetEventPointer()->Clear();

                    // result from dispatch operation
                    return g_Control.GetResult();
                }

                @endcode
*/
class PmControl
{
    NN_DISALLOW_COPY(PmControl);
    NN_DISALLOW_MOVE(PmControl);

private:
    nn::sf::SharedPointer<nn::psc::sf::IPmControl> m_InterfaceImpl;
    nn::os::SystemEvent m_SystemEvent;
    bool m_Initialized;

public:
    /**
        @brief      Send system wide request to power managed modules.
        @param[in]  state                           PM state to request.
        @param[in]  flags                           Additional flags.
        @param[in]  order                           Determines order in which requests are dispatched to PmModules.
        @return     nn::ResultSuccess               Operation succesfull.
        @return     nn::psc::ResultNotInitialized   PSC controller is not initialized.
        @return     nn::psc::ResultInvalidArgument  One of the arguments is not valid.

        @pre        @ref Initialize() must be called.
        @details    This function will dispatch a request to all managed modules. Requests
                    will be sent in accordance with the dependency information collected by PSC.
    */
    nn::Result DispatchRequest(PmState state, PmFlagSet flags, PmTransitionOrder order)
    NN_NOEXCEPT;

    /**
        @brief      Get state of PSC controller.
        @param[out] pStateOut                       Pointer to @ref PmControlState variable.
        @return     nn::ResultSuccess               Operation succesfull.
        @return     nn::psc::ResultNotInitialized   PSC controller is not initialized.

        @pre        @ref Initialize() must be called.
        @details    This function returns current state of the PSC controller.
    */
    nn::Result GetState(PmControlState* pStateOut)
    NN_NOEXCEPT;

    /**
        @brief      Get result from last operation.
        @return     nn::ResultSuccess                   Operation successfull.
        @return     nn::psc::ResultNotInitialized       PSC controller is not initialized.
        @return     nn::psc::ResultCancel               Operation was canceled.
        @return     nn::psc::ResultMaxDependencyLevels  Too many dependency levels.
        @return     nn::psc::ResultModuleNotInitialized Dependent module is not initialized.

        @pre        @ref Initialize() must be called.
        @details    When PSC signals an event due to completed operation,
                    PM control service may call this function to obtain result code
                    from that operation.
    */
    nn::Result GetResult()
    NN_NOEXCEPT;

    /**
        @brief      Cancel operation.
        @return     nn::ResultSuccess                   Operation succesfull.
        @return     nn::psc::ResultNotInitialized       PSC controller is not initialized.

        @pre        @ref Initialize() must be called.
        @details    PM control service may choose to cancel ongoing operation.
                    In that case PSC will signal an event and will set result
                    of the operation to nn::psc::ResultCancel. Note that
                    after cancel operation, state of the modules may be inconsitent -
                    some have completed transition, others not. PSC will not attempt
                    to roll back operation. It will be responsibility of PM
                    control service to dispatch a new request in order to bring all
                    modules into consitent state.
    */
    nn::Result Cancel()
    NN_NOEXCEPT;

    /**
        @brief      Obtain handle to system event associated with this module.
        @return     Pointer to system event.
        @pre        @ref Initialize() must be called

        @details    Clients wishing to implement event handling logic may obtain
                    pointer to the system event associated with this module.
    */
    nn::os::SystemEvent* GetEventPointer()
    NN_NOEXCEPT;

    /*
        @brief      Output dependency information and module state for all registered modules.
        @return     nn::ResultSuccess                   Operation successfull.
        @return     nn::psc::ResultNotInitialized       PSC controller is not initialized.

        @details    This function will output to the log dependency information
                    for all known PM modules. It can be used for debugging, for instance,
                    after a timeout was detected.

                    This call is only enabled on DEBUG or DEVELOP builds
    */
    nn::Result PrintModuleInformation()
    NN_NOEXCEPT;

    /*
        @brief      Return dependency information and module state for all registered modules.
        @return     nn::ResultSuccess                   Operation successfull.
        @return     nn::psc::ResultNotInitialized       PSC controller is not initialized.
        @return     nn::psc::ResultBufferOverflow       Size of given buffer is not enough.

        @details    This function will output to the log dependency information
                    for all known PM modules. It can be used for debugging, for instance,
                    after a timeout was detected.
    */
    nn::Result GetModuleInformation(
        nn::psc::util::TransitionInformation* pOutTransitionInfo,
        int *pOutModuleCount,
        nn::psc::util::ModuleTransitionInformation* pOutModuleInfo,
        int maxModuleCount,
        int *pOutDependencyCount,
        nn::psc::util::DependencyInformation* pOutDependencyInfo,
        int maxDependencyCount
    ) NN_NOEXCEPT;

    /**
        @brief      Initialize power control module.
        @param[in]  eventClearMode                      Clear mode for notification event
        @return     nn::ResultSuccess                   Operation successfull.
        @return     nn::psc::ResultAlreadyInitialized   Already initialized.
        @return     nn::psc::ResultOutOfMemory          Out of memory condition was detected.
        @pre        Cliend should be provisioned to access "psc:c" port.
    */
    nn::Result Initialize(nn::os::EventClearMode eventClearMode)
    NN_NOEXCEPT;

    /**
        @brief      Finalize power control module.
        @return     nn::ResultSuccess                   Operation successfull.
    */
    nn::Result Finalize()
    NN_NOEXCEPT;

    PmControl()  NN_NOEXCEPT;
    ~PmControl() NN_NOEXCEPT;
};

}}
