﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/prepo/detail/service/prepo_Common.h>

namespace nn { namespace prepo { namespace detail { namespace service { namespace core {

/*!
    @brief      システム情報収集モジュールです。
*/
class SystemInfo
{
public:
    /*!
        @brief      システム情報を収集します。

        @param[out] outSize 書き込まれたバイト数。
        @param[in]  buffer  バッファ。
        @param[in]  size    バッファサイズ。
        @param[in]  appId   アプリケーション ID。
        @param[in]  eventId イベント ID。

        @return     処理結果。

        @details
                    動作モードが未設定の場合は、ResultOperationModeUndefined を返します。
    */
    static nn::Result Collect(size_t* outSize, void* buffer, size_t size,
        nn::ApplicationId appId, const char* eventId) NN_NOEXCEPT;

    /*!
        @brief      システム情報を収集します。

        @param[out] outSize 書き込まれたバイト数。
        @param[in]  buffer  バッファ。
        @param[in]  size    バッファサイズ。
        @param[in]  appId   アプリケーション ID。
        @param[in]  eventId イベント ID。
        @param[in]  uid     ユーザーアカウント。

        @return     処理結果。

        @details
                    動作モードが未設定の場合は、ResultOperationModeUndefined を返します。
    */
    static nn::Result Collect(size_t* outSize, void* buffer, size_t size,
        nn::ApplicationId appId, const char* eventId, const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      システム情報を収集します。

        @param[out] outSize 書き込まれたバイト数。
        @param[in]  buffer  バッファ。
        @param[in]  size    バッファサイズ。
        @param[in]  appId   アプリケーション ID。
        @param[in]  eventId イベント ID。

        @return     処理結果。

        @details
                    動作モードが未設定の場合は、設定されるまでブロックします。
    */
    static nn::Result CollectSync(size_t* outSize, void* buffer, size_t size,
        nn::ApplicationId appId, const char* eventId) NN_NOEXCEPT;

    /*!
        @brief      システム情報を収集します。

        @param[out] outSize 書き込まれたバイト数。
        @param[in]  buffer  バッファ。
        @param[in]  size    バッファサイズ。
        @param[in]  appId   アプリケーション ID。
        @param[in]  eventId イベント ID。
        @param[in]  uid     ユーザーアカウント。


        @return     処理結果。

        @details
                    動作モードが未設定の場合は、設定されるまでブロックします。
    */
    static nn::Result CollectSync(size_t* outSize, void* buffer, size_t size,
        nn::ApplicationId appId, const char* eventId, const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      現在の動作モードを設定します。

        @param[in]  mode    動作モード。
    */
    static void SetOperationMode(int64_t mode) NN_NOEXCEPT;

    /*!
        @brief      システム情報の収集に必要なバッファのサイズ。
    */
    static const size_t RequiredBufferSize = 328;

private:
    //
    static nn::Result CollectApplicationId(msgpack::OutputStreamParam& stream, nn::ApplicationId appId) NN_NOEXCEPT;
    static nn::Result CollectEventId(msgpack::OutputStreamParam& stream, const char* eventId) NN_NOEXCEPT;
    static nn::Result CollectOperationMode(msgpack::OutputStreamParam& stream) NN_NOEXCEPT;
    static nn::Result CollectLocalClockRecordAt(msgpack::OutputStreamParam& stream) NN_NOEXCEPT;
    static nn::Result CollectNetworkClockRecordAt(msgpack::OutputStreamParam& stream) NN_NOEXCEPT;
    static nn::Result CollectNetworkServiceAccountId(msgpack::OutputStreamParam& stream, nn::account::Uid uid) NN_NOEXCEPT;
    static nn::Result CollectOsVersion(msgpack::OutputStreamParam& stream) NN_NOEXCEPT;
    static nn::Result CollectReportIdentifier(msgpack::OutputStreamParam& stream) NN_NOEXCEPT;
    static nn::Result CollectRebootlessSystemUpdateVersion(msgpack::OutputStreamParam& stream) NN_NOEXCEPT;

private:
    //
    static nn::Result GetOperationMode(int64_t* out) NN_NOEXCEPT;
    //
    static nn::Result GetAccountId(nn::account::NetworkServiceAccountId* out, const nn::account::Uid& uid) NN_NOEXCEPT;
};

}}}}}
