﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/prepo/detail/service/prepo_Common.h>
#include <curl/curl.h>

namespace nn { namespace prepo { namespace detail { namespace service { namespace core {

/*!
    @brief      レポートデータのアップローダです。
*/
class ReportUploader
{
private:
    NN_DISALLOW_COPY(ReportUploader);
    NN_DISALLOW_MOVE(ReportUploader);

private:
    /*!
        @brief      コンストラクタです。
    */
    ReportUploader() NN_NOEXCEPT;

public:
    /*!
        @brief      インスタンスを取得します。

        @return     インスタンス。
    */
    static ReportUploader& GetInstance() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(ReportUploader, s_Instance);
        return s_Instance;
    }

public:
    /*!
        @brief      レポートデータをアップロードします。

        @param[in]  authToken   デバイス認証トークン。
        @param[in]  data        レポートデータ。
        @param[in]  size        レポートデータのサイズ。

        @return     処理結果。

        @pre
            - authToken != nulltpr。
            - authToken が null 終端されている。
            - data != nulltpr。
            - size > 0。

        @details
                    本関数は、シングルスレッド専用です。
    */
    nn::Result Upload(const char* authToken, const Bit8* data, size_t size) NN_NOEXCEPT;

private:
    /*!
        @brief      入力ストリーム型です。
    */
    struct InputStream
    {
        const Bit8* data; //!< データの先頭ポインタ。
        size_t size;      //!< データサイズ。
        size_t position;  //!< 読み込み位置。

        /*!
            @brief      読み込み位置のポインタを取得します。

            @return     読み込み位置のポインタ。
        */
        inline const Bit8* GetReadPointer() NN_NOEXCEPT
        {
            return data + position;
        }

        /*!
            @brief      データの残りサイズを取得します。

            @return     データの残りサイズ。
        */
        inline size_t GetRemainSize() const NN_NOEXCEPT
        {
            return size - position;
        }
    };

private:
    //
    InputStream m_Stream;
    //
    nn::Result SetupCurl(curl_slist** outHeaders, CURL* curl, const char* authToken) NN_NOEXCEPT;
    //
    nn::Result Perform(const char* authToken) NN_NOEXCEPT;

private:
    //
    static CURLcode SslCtxFunction(CURL* curl, void* ssl, void* param) NN_NOEXCEPT;
    //
    static size_t HttpReadFunction(char* buffer, size_t size, size_t count, void* param) NN_NOEXCEPT;
    //
    static size_t HttpWriteFunction(char* buffer, size_t size, size_t count, void* param) NN_NOEXCEPT;
};

}}}}}
