﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#pragma once

/**
* @file
* @brief  共有フォントに関するシステム向けのヘッダファイルです。
*/

#include <nn/pl/pl_SharedFontApi.h>
#include <nn/settings/settings_Language.h>

namespace nn { namespace pl {

/**
* @brief        共有フォントを言語ごとに定められた優先度順に取得します。
* @param[out]   pOutFontCount   取得可能なフォントの数の格納先。
* @param[out]   outTypes        フォントの種類の格納先。
* @param[out]   outAddresses    フォントのアドレスの格納先。
* @param[out]   outSizes        フォントのサイズの格納先。
* @param[in]    outCountMax     outTypes と outAddresses、outSizes の要素数。
* @param[in]    languageCode    言語。
* @return       全てのフォントがロード済みでフォントの取得に成功した場合は true。
* @pre
*               - pOutFontCount != nullptr
*               - outTypes != nullptr
*               - outAddresses != nullptr
*               - outSizes != nullptr
* @post
*               - 返り値が true の場合
*                   - outTypes[0 ~ min(*pOutFontCount, outCountMax) - 1] が有効なフォントの種類を保持
*                   - outAddresses[0 ~ min(*pOutFontCount, outCountMax) - 1] が有効なアドレスを保持
*                   - outSizes[0 ~ min(*pOutFontCount, outCountMax) - 1] が有効な値を保持
* @details      共有フォントを languageCode に指定した言語で定められた優先度順に取得します。
*               インデックス 0 のデータから順に高優先度で描画することで、その言語に適したフォントで文字の描画を行うことができます。
*               複数のフォントを組み合わせて使う方法についてはフォントライブラリのマニュアルなどを参照ください。
*/
bool GetSharedFontInOrderOfPriority(int* pOutFontCount, SharedFontType outTypes[], const void* outAddresses[], size_t outSizes[], int outCountMax, settings::LanguageCode languageCode) NN_NOEXCEPT;

/**
* @brief  共有フォントの種類を表します（システム向け）。
*/
enum SharedFontTypeForSystem
{
    SharedFontTypeForSystem_Standard,                //!< 日米欧
    SharedFontTypeForSystem_ChineseSimple,           //!< 簡体字
    SharedFontTypeForSystem_ChineseSimpleExtension,  //!< 簡体字（追加文字）
    SharedFontTypeForSystem_ChineseTraditional,      //!< 繁体字
    SharedFontTypeForSystem_Korean,                  //!< ハングル
    SharedFontTypeForSystem_NintendoExtension,       //!< 任天堂外字
    SharedFontTypeForSystem_NintendoExtension2,      //!< 任天堂外字（4.0.0NUP以降システム向け）
    SharedFontTypeForSystem_Max,                     //!< 列挙子の総数
};

/**
* @brief        共有フォントを言語ごとに定められた優先度順に取得します（システム向け）。
* @param[out]   pOutFontCount   取得可能なフォントの数の格納先。
* @param[out]   outTypes        フォントの種類の格納先。
* @param[out]   outAddresses    フォントのアドレスの格納先。
* @param[out]   outSizes        フォントのサイズの格納先。
* @param[in]    outCountMax     outTypes と outAddresses、outSizes の要素数。
* @param[in]    languageCode    言語。
* @return       全てのフォントがロード済みでフォントの取得に成功した場合は true。
* @pre
*               - pOutFontCount != nullptr
*               - outTypes != nullptr
*               - outAddresses != nullptr
*               - outSizes != nullptr
* @post
*               - 返り値が true の場合
*                   - outTypes[0 ~ min(*pOutFontCount, outCountMax) - 1] が有効なフォントの種類を保持
*                   - outAddresses[0 ~ min(*pOutFontCount, outCountMax) - 1] が有効なアドレスを保持
*                   - outSizes[0 ~ min(*pOutFontCount, outCountMax) - 1] が有効な値を保持
* @details      共有フォントを languageCode に指定した言語で定められた優先度順に取得します。
*               インデックス 0 のデータから順に高優先度で描画することで、その言語に適したフォントで文字の描画を行うことができます。
*               複数のフォントを組み合わせて使う方法についてはフォントライブラリのマニュアルなどを参照ください。
*/
bool GetSharedFontInOrderOfPriorityForSystem(int* pOutFontCount, SharedFontTypeForSystem outTypes[], const void* outAddresses[], size_t outSizes[], int outCountMax, settings::LanguageCode languageCode) NN_NOEXCEPT;

}}
