﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/*
 * @file
 * @brief  PlayDataManager が使用しシステム・本体機能に公開する型を定義します。
 */

#include <nn/nn_Common.h>
#include <nn/nn_StaticAssert.h>
#include <nn/account/account_Types.h>
#include <nn/applet/applet_Types.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/ncm/ncm_ProgramId.h>
#include <nn/ncm/ncm_StorageId.h>
#include <nn/oe/oe_OperationModeApis.h>
#include <nn/time/time_PosixTime.h>

namespace nn { namespace pdm {

//! @brief システムの動作モードの種類です。
enum class OperationMode : uint8_t
{
    Handheld = oe::OperationMode_Handheld,  //!< 携帯モード
    Console  = oe::OperationMode_Console,   //!< 据え置きモード
};

//! @brief アプレットに関するイベントの種類です。
enum class AppletEventType : uint8_t
{
    Launch          = 0,    //!< アプレットの開始
    Exit            = 1,    //!< アプレットの終了
    InFocus         = 2,    //!< アプレットのインフォーカス状態
    OutOfFocus      = 3,    //!< アプレットのアウトフォーカス状態
    Background      = 4,    //!< アプレットのバックグラウンドフォーカス状態
    Terminate       = 5,    //!< アプレットの強制終了
    AbnormalExit    = 6,    //!< アプレットの異常終了
};

//! @brief 電源状態の変更に関するイベントの種類です。
enum class PowerStateChangeEventType : uint8_t
{
    On                      = 0,    //!< 電源オン
    Off                     = 1,    //!< 正常な電源オフ
    Sleep                   = 2,    //!< スリープ状態への移行
    Awake                   = 3,    //!< スリープ状態から起床（アプリが実行できる状態への復帰）
    BackgroundServicesAwake = 4,    //!< バックグラウンド処理のみ有効な状態への移行（ユーザーからの見た目はスリープ状態と同等）
    Terminate               = 5,    //!< 強制終了
};

//! @brief ユーザーアカウントに関するイベントの種類です。
enum class UserAccountEventType : uint8_t
{
    Open    = 0,    //!< オープン（ログイン）
    Close   = 1,    //!< クローズ（ログアウト）
    NetworkServiceAccountAvailable   = 2,   //!< ネットワークサービスアカウントが利用可能
    NetworkServiceAccountUnavailable = 3,   //!< ネットワークサービスアカウントが利用不可
};

//! @brief イベントの発生時刻に関する情報です。
struct EventTimeData
{
    uint32_t    eventIndex;         //!< イベントのインデックス
    uint32_t    userClockTime;      //!< ユーザ時計時刻（nn::time::InputPosixTimeMin からの経過分）
    uint32_t    networkClockTime;   //!< ネットワーク時計時刻（nn::time::InputPosixTimeMin からの経過分）。ネットワーク時計が無効な場合は 0。
};

//! @brief イベントの発生時刻に関する詳細情報です。
struct DetailedEventTimeData
{
    uint32_t        eventIndex;                     //!< イベントのインデックス
    time::PosixTime userClockTime;                  //!< ユーザー時計時刻
    time::PosixTime networkClockTime;               //!< ネットワーク時計時刻。ネットワーク時計時刻が無効な場合は 0。
    int64_t         steadyClockTimeValueSinceBase;  //!< 単調増加時計時刻の基準点からの経過秒。基準点は pdm 独自の値です。
};

//! @brief アプリケーションのイベントです。
struct ApplicationEvent
{
    ncm::ApplicationId  applicationId;  //!< イベント対象のアプリケーションID
    EventTimeData       eventTime;      //!< イベントの発生時刻に関する情報
    AppletEventType     eventType;      //!< イベントの種類
};

//! @brief アカウントのイベントです。
struct AccountEvent
{
    account::Uid            uid;        //!< イベント対象のユーザーアカウントID
    DetailedEventTimeData   eventTime;  //!< イベントの発生時刻に関する情報
    UserAccountEventType    eventType;  //!< イベントの種類
};

//! @brief アプリケーションの統計情報です。
struct PlayStatistics
{
    ncm::ApplicationId  applicationId;  //!< アプリケーションID
    EventTimeData       firstEventTime; //!< 対象のプログラムの最初のイベントの情報
    EventTimeData       latestEventTime;//!< 対象のプログラムの最後のイベントの情報
    uint32_t            totalPlayTime;  //!< 合計プレイ時間（分単位）
    uint32_t            totalPlayCount; //!< 合計プレイ回数
};

}}
