﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/pdm/pdm_PrivateTypes.h>

namespace nn { namespace pdm { namespace detail {

/**
 * @brief   排他オブジェクト付きオンメモリバッファインタフェース
 *
 * @details 本クラスのデストラクタは仮想化していません。
 *          派生先クラスのデストラクタを呼び出すようにしてください。
 */
class LockableMemoryBuffer
{
    NN_DISALLOW_COPY(LockableMemoryBuffer);
    NN_DISALLOW_MOVE(LockableMemoryBuffer);

public:
    /**
     * @brief   ミューテックスによる排他施錠を実施します。
     *
     * @details @ref std::lock_guard などの利用を想定した小文字名称定義です。
     */
    void lock() const NN_NOEXCEPT
    {
        m_Mutex.Lock();
    }

    /**
     * @brief   ミューテックスによる排他解除を実施します。
     *
     * @details @ref std::lock_guard などの利用を想定した小文字名称定義です。
     */
    void unlock() const NN_NOEXCEPT
    {
        m_Mutex.Unlock();
    }

    /**
     * @brief   管理対象バッファのオンメモリ先頭アドレスを返します。
     *          本メソッドでの排他保証は派生先に依存します。
     */
    virtual Bit8* GetPointer() NN_NOEXCEPT = 0;

    /**
     * @brief   管理対象バッファのオンメモリ先頭アドレスを返します。 ( const )
     *          本メソッドでの排他保証は派生先に依存します。
     */
    virtual const Bit8* GetPointer() const NN_NOEXCEPT = 0;

    /**
     * @brief   管理対象バッファの容量をバイト単位で返します。
     *          本メソッドでの排他保証は派生先に依存します。
     */
    virtual uint32_t GetCapacity() const NN_NOEXCEPT = 0;

protected:
    /**
     * @brief   コンストラクタ。
     */
    explicit LockableMemoryBuffer() NN_NOEXCEPT = default;

private:
    mutable os::SdkRecursiveMutex m_Mutex;  //!< 排他用オブジェクト
};



/**
 * @brief   メモリ内蔵型テンポラリバッファクラス
 */
template<uint32_t TByteCapacity>
class LockableMemoryBuiltIn : public LockableMemoryBuffer
{
    NN_DISALLOW_COPY(LockableMemoryBuiltIn);
    NN_DISALLOW_MOVE(LockableMemoryBuiltIn);

public:
    static const uint32_t ByteCapacity = TByteCapacity; //!< 内蔵バッファ容量( byte )

    /**
     * @brief   コンストラクタ
     */
    NN_IMPLICIT LockableMemoryBuiltIn() NN_NOEXCEPT = default;

    /**
     * @brief   管理対象バッファのオンメモリ先頭アドレスを返します。
     *          本関数の実施に伴う排他処理はありません。
     */
    virtual Bit8* GetPointer() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_Buffer;
    }

    /**
     * @brief   管理対象バッファのオンメモリ先頭アドレスを返します。
     *          本関数の実施に伴う排他処理はありません。
     */
    virtual const Bit8* GetPointer() const NN_NOEXCEPT NN_OVERRIDE
    {
        return m_Buffer;
    }

    /**
     * @brief   管理対象バッファの容量をバイト単位で返します。
     *          本関数の実施に伴う排他処理はありません。
     */
    virtual uint32_t GetCapacity() const NN_NOEXCEPT NN_OVERRIDE
    {
        return ByteCapacity;
    }

private:
    Bit8 m_Buffer[ByteCapacity];    //!< 内蔵バッファ実体。
};



/**
 * @brief   クエリ支援ユーティリティ
 */
class QueryServiceHelper
{
    NN_DISALLOW_COPY(QueryServiceHelper);
    NN_DISALLOW_MOVE(QueryServiceHelper);

public:
    explicit QueryServiceHelper() NN_NOEXCEPT {}

    static void Initialize(nn::pdm::EventTimeData* outValue) NN_NOEXCEPT;
    static void Initialize(PlayStatistics* outValue) NN_NOEXCEPT;
    inline static void Initialize(PlayStatistics* outValue, const ncm::ApplicationId& applicationId) NN_NOEXCEPT;

    inline static void ConvertTotalTimeFromSecondsToMinutes(PlayStatistics* pInOutValue) NN_NOEXCEPT;
    inline static void ConvertTotalTimeFromSecondsToMinutes(PlayStatistics outList[], int count) NN_NOEXCEPT;

    inline static int SelectPlayStatisticsIndexToReplace(const pdm::PlayStatistics list[], int count) NN_NOEXCEPT;
};



//----------------------------------------------------------------------------
//! @name インラインメソッド実装
//@{
inline void QueryServiceHelper::ConvertTotalTimeFromSecondsToMinutes(PlayStatistics* pInOutValue) NN_NOEXCEPT
{
    pInOutValue->totalPlayTime = pInOutValue->totalPlayTime / 60;
}

inline void QueryServiceHelper::ConvertTotalTimeFromSecondsToMinutes(PlayStatistics outList[], int count) NN_NOEXCEPT
{
    for (int i = 0; i < count; ++i)
    {
        ConvertTotalTimeFromSecondsToMinutes(&outList[i]);
    }
}

inline void QueryServiceHelper::Initialize(PlayStatistics* outValue, const ncm::ApplicationId& applicationId) NN_NOEXCEPT
{
    Initialize(outValue);
    outValue->applicationId = applicationId;
}

inline int QueryServiceHelper::SelectPlayStatisticsIndexToReplace(const pdm::PlayStatistics list[], int count) NN_NOEXCEPT
{
    NN_UNUSED(list);
    NN_UNUSED(count);
    // TODO: プレイ時間や最終起動日を見て適切なエントリーを選択してドロップする。
    return 0;
}

//@}

}}}
