﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief 電源、クロック、電圧を操作する機能の API 宣言。
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>

#include "pcv_Types.h"

namespace nn {
namespace pcv {

/**
 * @brief       指定したモジュールに供給される電力の供給・非供給状態を設定します。
 *
 * @param[in]   moduleId 制御対象モジュール
 * @param[in]   enabled 電力を供給するか否か
 * @return      処理の結果を返します。
 * @retval      ResultSuccess 成功しました。
 *
 * @pre         Initialize 関数を実行してライブラリを初期化している必要があります。
 *              対象モジュールが給電状態の変更に対応している必要があります。
 * @post        モジュールが給電状態の変更に対応している場合 enabled 引数に従って供給・非供給状態になります。
 * @details     moduleId を指定してシステム上の当該モジュールの給電状態を変更できます。
 *              enabled 引数に従って供給・非供給状態になります。
 */
Result SetPowerEnabled(Module moduleId, bool enabled) NN_NOEXCEPT;

/**
 * @brief       指定したモジュールに供給可能なクロックをリストとして取得します。
 *
 * @param[out]  pOutType クロックのリストの表記形式
 * @param[out]  pOutRates クロックのリスト
 * @param[out]  pOutCount クロックのリストの長さ
 * @param[in]   moduleId 制御対象モジュール
 * @param[in]   maxCount クロックのリストの最大の長さ
 * @return      処理の結果を返します。
 * @retval      ResultSuccess 成功しました。
 *
 * @pre         Initialize 関数を実行してライブラリを初期化している必要があります。
 *              対象モジュールがクロックレートの変更に対応している必要があります。
 * @post
 * @details     指定したモジュールに供給可能なクロックをリストとして取得します。
 *              @a pOutRates に格納されるリストの形式は @a pOutType に返される値で示されます。各形式については @ref ClockRatesListType を参照してください。
 */
Result GetPossibleClockRates(ClockRatesListType* pOutType, ClockHz* pOutRates,
                             int* pOutCount, Module moduleId, int maxCount) NN_NOEXCEPT;

/**
 * @brief       指定したモジュールのクロックの供給・非供給状態を設定します。
 *
 * @param[in]   moduleId 制御対象モジュール
 * @param[in]   enabled クロックを供給するか否か
 * @return      処理の結果を返します。
 * @retval      ResultSuccess 成功しました。
 *
 * @pre         Initialize 関数を実行してライブラリを初期化している必要があります。
 *              対象モジュールがクロックの制御に対応している必要があります。
 * @post        モジュールがクロック制御に対応している場合 enabled 引数に従って供給・非供給状態になります。
 * @details     moduleId を指定してシステム上の当該モジュールのクロックを制御します。
 *              モジュールがクロック制御に対応している場合 enabled 引数に従って供給・非供給状態になります。
 */
Result SetClockEnabled(Module moduleId, bool enabled) NN_NOEXCEPT;

/**
 * @brief       指定したモジュールのクロックレートを変更します。
 *
 * @param[in]   moduleId 制御対象モジュール
 * @param[in]   clockRateHz 設定するクロックレート
 * @return      処理の結果を返します。
 * @retval      ResultSuccess 成功しました。
 *
 * @pre         Initialize 関数を実行してライブラリを初期化している必要があります。
 *              対象モジュールがクロックレートの変更に対応している必要があります。
 *              @a clockRateHz は対象モジュールに GetPossibleClockRates を実行した際に得られる ClockRatesListType に
 *              応じた値になっている必要があります。ClockRatesListType_Discrete の場合 SetClockRate で
 *              正確な値を引数に取らせなければなりません。ClockRatesListType_Range の場合 SetClockRate で
 *              正確な値を引数に取らせる必要はありません。
 * @post        モジュールがクロックレート変更に対応しており、
 *              clockRateHz に適切なクロックレートが指定された場合、
 *              指定されたクロックレートに変更されます。
 * @details     モジュールの ClockRatesListType が ClockRatesListType_Range の場合、設定されるクロックレートは、
 *              上限クロックレート以下かつ指定クロックレート以上で設定可能な最小のクロックレートとなります。
 *              指定クロックレート以上に設定不可能な場合は、指定クロックレート以下で設定可能な最大のクロックレートとなります。
 */
Result SetClockRate(Module moduleId, ClockHz clockRateHz) NN_NOEXCEPT;


/**
 * @brief       Return the clock rate for the specified module.
 *
 * @param[out]  pClockRateHz 設定するクロックレート
 * @param[in]   moduleId 制御対象モジュール
 * @return      The clock rate for the module is returned through pClockRateHz.
 * @retval      ResultSuccess if successful.  Else, error result.
 *
 * @pre         Initialize the PCV library
 * @post
 * @details
 */
Result GetClockRate(ClockHz* pClockRateHz, Module moduleId) NN_NOEXCEPT;

/**
 * @brief       指定したモジュールの現在の状態を取得します。
 *
 * @param[out]  pOutState モジュールの現在の状態
 * @param[in]   moduleId 対象モジュール
 * @return      処理の結果を返します。
 * @retval      ResultSuccess 成功しました。
 *
 * @pre         Initialize 関数を実行してライブラリを初期化している必要があります。
 * @post
 * @details     モジュールの現在の状態を取得し ModuleState 構造体の各要素に値を格納します。
 */
Result GetState(ModuleState *pOutState, Module moduleId) NN_NOEXCEPT;

/**
 * @brief       Sets the minimum voltage used for a module.  This feature is to enable
 *              fast clock rate setting to the specified clock rate.  The trade-off
 *              of doing this is possible higher power consumption than normal at lower
 *              clock rates.
 *
 * @param[in]   moduleId 制御対象モジュール
 * @param[in]   clockRateHz - the clock rate used to set the minimum voltage used
 * @return      処理の結果を返します。
 * @retval      ResultSuccess if successful.  Otherwise, an error result is returned.
 *
 * @pre         Initialize the PCV library.
 * @post
 * @details     The minimum voltage used for moduleId will be the minimum voltage that
 *              is needed to support clock rate clockRateHz. Clock rates below clockRateHz
 *              will not use a lower voltage.
 *
 *              At lower clock rates, it is some times possible to use lower voltages.
 *              Thus, when switching from a lower clock rate to a higher clock rate, it
 *              may be necessary to raise the voltage to support the higher clock rate.
 *              Raising voltages can take noticeable latency.  Fixing the voltage higher
 *              than needed can eliminate that latency to change from the lower to
 *              higher clock rate.
 */
Result SetMinVClockRate(Module moduleId, ClockHz clockRateHz) NN_NOEXCEPT;

/**
 * @brief       指定したモジュールのリセットをアサート状態を切り替えます。
 *
 * @param[in]   moduleId 対象モジュール
 * @param[in]   asserted アサート・デアサートの指定
 * @return      処理の結果を返します。
 * @retval      ResultSuccess 成功しました。
 *
 * @pre         Initialize 関数を実行してライブラリを初期化している必要があります。
 *              対象モジュールがリセットの制御に対応している必要があります。
 * @post        指定されたモジュールがリセットのアサート状態の制御に対応していた場合、
 *              モジュールのリセットのアサート状態を切り替えます。
 */
Result SetReset(Module moduleId, bool asserted) NN_NOEXCEPT;

/**
 * @brief       Enables or disables the power domain.
 *
 * @param[in]   powerDomain - power domain to enable or disable.
 * @param[in]   enabled - if true, enables.  Else, disables.
 * @return
 * @retval      ResultSuccess if successful.  Otherwise, an error result is returned.
 *
 * @pre         Initialize the PCV library.
 * @post
 */
Result SetVoltageEnabled(PowerDomain powerDomain, bool enabled) NN_NOEXCEPT;

/**
 * @brief       Returns if the power domain is enabled
 *
 * @param[out]  pEnabled
 * @param[in]   powerDomain
 * @return      Returns state through pEnabled.
 * @retval      ResultSuccess if successful.  Otherwise, an error result is returned.
 *
 * @pre         Initialize the PCV library.
 * @post
 * @details
 */
Result GetVoltageEnabled(bool* pEnabled, PowerDomain powerDomain) NN_NOEXCEPT;

/**
 * @brief       Returns the minimum, maximum, and step size voltages for the power domain.
 *
 * @param[out]  pMinVolt
 * @param[out]  pMaxVolt
 * @param[out]  pStepVolt
 * @param[in]   powerDomain
 * @retval      ResultSuccess if successful.  Otherwise, an error result is returned.
 *
 * @pre         Initialize the PCV library.
 * @post
 * @details
 */
Result GetVoltageRange(MicroVolt* pMinVolt, MicroVolt* pMaxVolt, MicroVolt* pStepVolt, PowerDomain powerDomain) NN_NOEXCEPT;

/**
 * @brief       Sets the voltage for a power domain.
 *
 * @param[in]   powerDomain
 * @param[in]   microVolt - volage value in microvolts
 * @return
 * @retval      ResultSuccess if successful.  Otherwise, an error result is returned.
 *
 * @pre         Initialize the PCV library.
 * @post
 * @details
 */
Result SetVoltageValue(PowerDomain powerDomain, MicroVolt microVolt) NN_NOEXCEPT;

/**
 * @brief       Get the voltage for a power domain.
 *
 * @param[out]  pMicroVolt - returned voltage in microvolts.
 * @param[in]   powerDomain
 * @return      Returned voltage through pMicroVolt.
 * @retval      ResultSuccess if successful.  Otherwise, an error result is returned.
 *
 * @pre         Initialize the PCV library.
 * @post
 * @details
 */
Result GetVoltageValue(MicroVolt* pMicroVolt, PowerDomain powerDomain) NN_NOEXCEPT;

/**
 * @brief       Get the temperature thresholds that the PCV library is interested in.
 *
 * @param[out]  pThresholds - buffer of maxCount entries to return threshold values
 * @param[out]  pOutCount - number of thresholds returned
 * @param[in]   maxCount - size of pThresholds buffer in number of entries
 * @return      pThresholds and pOutCount
 * @retval      ResultSuccess if successful.  Otherwise, an error result is returned.
 *
 * @pre         Initialize the PCV library.
 * @post
 * @details
 */
Result GetTemperatureThresholds(TemperatureThreshold* pThresholds, int* pOutCount, int maxCount) NN_NOEXCEPT;

/**
 * @brief       Sets PCV's temperature state.
 *
 * @param[in]   temperatureMilliC - temperature in millicelius
 * @return
 * @retval      ResultSuccess if successful.  Otherwise, an error result is returned.
 *
 * @pre         Initialize the PCV library.
 * @post
 * @details     Used in conjunction with GetTemperatureThresholds(). GetTemperatureThresholds() is
 *              used to get the thresholds that the PCV is interested in.  When those thresholds
 *              are crossed, SetTemperature() is used to inform PCV of the current temperature.
 *              Informing PCV of the temperature is for correctness and/or optimization of voltage
 *              settings.
 */
Result SetTemperature(MilliC temperatureMilliC) NN_NOEXCEPT;

} // pcv
} // nn
