﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/account/account_Types.h>

namespace nn { namespace pctl {

/**
 * @brief 連携処理の状態を表す定数値です。
 */
enum PairingState
{
    PairingState_None = 0,              //!< 未連携であるか連携処理が行われていません。
    PairingState_Processing,            //!< 連携処理中です。
    PairingState_Active                 //!< 連携済みです。
};

/**
 * @brief アカウントの状態を表す定数値です。
 */
enum PairingAccountState
{
    PairingAccountState_Active = 0,     //!< アカウントは有効です。
    PairingAccountState_Deleted         //!< アカウントが削除されています。
};

/**
 * @brief 曜日を表す定数値です。
 */
enum Week
{
    Week_Sunday = 0,                    //!< 日曜日
    Week_Monday,                        //!< 月曜日
    Week_Tuesday,                       //!< 火曜日
    Week_Wednesday,                     //!< 水曜日
    Week_Thursday,                      //!< 木曜日
    Week_Friday,                        //!< 金曜日
    Week_Saturday,                      //!< 土曜日
    Week_TotalCount                     //!< (曜日の数を表します。直接使用しないでください。)
};

/**
 * @brief プレイタイマーの時間切れにおける制限モードを表す定数値です。
 */
enum PlayTimerMode : uint8_t
{
    PlayTimerMode_Alarm = 0,            //!< アラームモード
    PlayTimerMode_Suspend               //!< 強制中断モード
};

/**
 * @brief プレイタイマー処理の1日ごとの設定値を表す構造体です。
 */
struct PlayTimerDaySettings
{
    bool isBedtimeEnabled;              //!< おやすみタイム設定(プレイ可能時刻を指定時刻までとする設定)が有効かどうか
    uint8_t bedtimeHour;                //!< おやすみタイム設定の時刻のうち時を表す値(0～23)
    uint8_t bedtimeMinute;              //!< おやすみタイム設定の時刻のうち分を表す値(0～59)
    bool isLimitTimeEnabled;            //!< 1日の遊べる時間の設定が有効かどうか
    uint16_t limitTime;                 //!< 1日の遊べる時間(単位: 分)
};

/**
 * @brief プレイタイマー処理の設定値を表す構造体です。
 *
 * @details
 * isEnabled が false である場合、残りのフィールドの値は未定義です。
 */
struct PlayTimerSettings
{
    bool isEnabled;                     //!< 設定が有効かどうか
    PlayTimerMode playTimerMode;        //!< 時間切れの際の動作モード
    bool isWeekSettingsUsed;            //!< 各曜日ごとの設定値を用いるかどうか
    NN_PADDING1;
    PlayTimerDaySettings dailySettings;                 //!< 全日共通の設定値(isWeekSettingsUsed == false の場合)
    PlayTimerDaySettings weekSettings[Week_TotalCount]; //!< 各曜日ごとの設定値(isWeekSettingsUsed == true の場合)
};

}}
