﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

namespace nn { namespace pctl {

/**
 * @name 修理・修復ツール向けペアレンタルコントロール処理関数
 * @{
 */

/**
 * @brief ペアレンタルコントロールの機能をすべて無効化します。
 * @return 無効化したら true、既に無効化されていた場合は false
 *
 * @details
 * 本関数による無効化を実施すると、ペアレンタルコントロールや連携状態の設定が
 * 無い状態と同等の扱いになり、それらの設定に基づく機能制限が
 * 行われなくなるほか、サーバー連携時のイベント蓄積処理や設定同期処理なども停止されます。
 * 各種制限の設定そのものは削除されないため、@ref PostEnableAllFeatures で有効な状態に
 * 戻すと、無効にする前の制限状態に戻り、サーバー連携時に行われる処理が行われるようになります。
 *
 * 本関数による無効化はペアレンタルコントロールシステムに即時反映されます。
 *
 * なお、本関数による無効化は永続化され、本体システムの再起動などでも
 * 無効化の状態はそのまま続きます。(本体初期化が行われた場合は
 * 他の制限設定と一緒に本関数による無効化設定も失われます。)
 * 本体初期化でセーブデータ削除が行われるなどがない限り無効化の状態は残り続けますので、
 * 無効化を必要とする処理が完了したら必ず @ref PostEnableAllFeatures を呼び出して
 * 元に戻すようにしてください。
 * (この挙動はシステム向けの DisableFeaturesForReset とは異なります。)
 *
 * システム向けの DisableFeaturesForReset が先に呼び出された状態で
 * 本関数を呼び出した場合は、改めて必要な無効化処理と永続化を行い、
 * 関数は true を返します。
 */
bool DisableAllFeatures() NN_NOEXCEPT;

/**
 * @brief 無効化されたペアレンタルコントロールの機能を有効化するようにします。
 * @return 有効化したら true、呼び出し時点に無効化されていなかった場合は false
 *
 * @details
 * 本関数は永続化された無効化状態を解除しますが、有効化を
 * ペアレンタルコントロールシステムに即時反映せず、無効化状態を維持させます。
 * 有効化は本体システムを再起動することで反映されます。
 *
 * 無効化の状態から有効化した後も無効化状態は続きますが、その状態で
 * 本関数を呼び出すと戻り値は false になります。
 */
bool PostEnableAllFeatures() NN_NOEXCEPT;

/**
 * @brief ペアレンタルコントロールの機能がすべて無効化されているかどうかを返します。
 * @param[out] outIsEnabledOnNextBoot 次回起動時に有効化されるかどうかを受け取るポインター。
 *     この値が不要である場合は nullptr を指定することができます。
 * @return 無効化されている場合は true
 *
 * @details
 * @ref DisableAllFeatures が既に呼び出されており、@ref PostEnableAllFeatures が呼び出されて
 * 本体の再起動が行われていない状態の場合、本関数は *outIsEnabledOnNextBoot に true を
 * 設定しつつ戻り値として true を返します。@n
 * また、無効化されていない状態であれば *outIsEnabledOnNextBoot と戻り値はともに false になります。
 */
bool IsAllFeaturesDisabled(bool* outIsEnabledOnNextBoot) NN_NOEXCEPT;

/**
 * @}
 */

}}
