﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os/os_Event.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_SdkThread.h>
#include <nn/ovln/format/ovln_PctlMessage.h>
#include <nn/pctl/pctl_TypesForAuthentication.h>
#include <nn/pctl/detail/service/pctl_ParentalControlStates.h>
#include <nn/pctl/detail/service/system/pctl_Settings.h>
#include <nn/time/time_PosixTime.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_LockGuard.h>

namespace nn { namespace pctl { namespace detail { namespace service { namespace system {

/**
 * @brief 解除コードの形式が有効かどうかを返します。
 * @param[in] pinCode チェックするコード
 * @param[in] pinCodeLength pinCode の長さ(NULL文字含まず)
 */
inline bool IsValidPinCode(const char* pinCode, size_t pinCodeLength) NN_NOEXCEPT
{
    if (pinCodeLength < MinPinCodeLength || pinCodeLength > MaxPinCodeLength) // NULL文字除いて範囲外かどうかの確認
    {
        return false;
    }
    // NULL文字を含まない長さなのでチェックしてから1減らす
    while (pinCodeLength--)
    {
        if (!(*pinCode >= '0' && *pinCode <= '9'))
        {
            return false;
        }
        ++pinCode;
    }
    return true;
}

class SettingsManager
{
public:
    static const size_t ThreadStackSize = 12 * 1024;

    SettingsManager(void* threadStackBuffer, size_t stackBufferSize) NN_NOEXCEPT;
    ~SettingsManager() NN_NOEXCEPT;

    // @brief スレッドを開始します。
    // @details 終了はデストラクターで行います。
    void StartThread() NN_NOEXCEPT;

    // @brief 「他の人との自由なコミュニケーション」が制限されるかどうかを返します。
    nn::Result ConfirmFreeCommunicationRestriction(const ParentalControlStates& states, nn::ncm::ApplicationId applicationId, bool freeCommunicationFlag) const NN_NOEXCEPT;
    // @brief アプリケーション起動時の制限チェックを行います。
    nn::Result ConfirmLaunchApplicationPermission(const ParentalControlStates& states, nn::ncm::ApplicationId applicationId, const int8_t* ratingAgeData, size_t ratingAgeDataSize, bool freeCommunicationFlag) const NN_NOEXCEPT;
    // @brief アプリケーション再開時の制限チェックを行います。
    nn::Result ConfirmResumeApplicationPermission(const ParentalControlStates& states, nn::ncm::ApplicationId applicationId, const int8_t* ratingAgeData, size_t ratingAgeDataSize, bool freeCommunicationFlag) const NN_NOEXCEPT;
    // @brief SNS投稿の制限チェックを行います。
    nn::Result ConfirmSnsPostPermission(const ParentalControlStates& states) const NN_NOEXCEPT;
    // @brief 立体視機能の制限チェックを行います。
    nn::Result ConfirmStereoVisionPermission(const ParentalControlStates& states) const NN_NOEXCEPT;
    // @brief 本体設定系の制限チェックを行います。
    nn::Result ConfirmSystemSettingsPermission() const NN_NOEXCEPT;
    // @brief 立体視機能制限の設定が行えるかどうかのチェックを行います。
    nn::Result ConfirmStereoVisionRestrictionConfigurable() const NN_NOEXCEPT;
    // @brief 指定した動画が再生できるかどうかのチェックを行います。
    nn::Result ConfirmPlayableApplicationVideo(const ParentalControlStates& states, nn::ncm::ApplicationId applicationId, const int8_t* ratingAgeData, size_t ratingAgeDataSize) const NN_NOEXCEPT;

    // @brief 解除コード(暗証番号)のチェックを行います。
    nn::Result CheckPinCode(const char* pinCode, size_t pinCodeWithNullCharLength) const NN_NOEXCEPT;

    // @brief 現在制限が発生するものを返します。
    int GetRestrictedFeatures() const NN_NOEXCEPT;
    // @brief 制限設定が有効かどうかを返します。
    bool IsRestrictionEnabled() const NN_NOEXCEPT;
    // @brief 既定で利用するレーティング団体を取得します。
    //        未設定の場合は現在のリージョン設定などから取得した値を返します。
    nn::ns::RatingOrganization GetDefaultRatingOrganization() NN_NOEXCEPT;
    // @brief レーティング制限を行うための制限年齢値を取得します。
    uint8_t GetRatingAge() const NN_NOEXCEPT;
    // @brief 「安心レベル」を取得します。
    SafetyLevel GetSafetyLevel() const NN_NOEXCEPT;
    // @brief 現在有効な設定値を取得します。
    void GetCurrentSettings(SafetyLevelSettings* pSettings) const NN_NOEXCEPT;
    // @brief 「安心レベル」がカスタムである際に用いられる設定値を取得します。
    void GetCustomSettings(SafetyLevelSettings* pSettings) const NN_NOEXCEPT;
    // @brief 既定で利用するレーティング団体を変更します。
    void SetDefaultRatingOrganization(nn::ns::RatingOrganization organization) NN_NOEXCEPT;
    // @brief 「安心レベル」を変更します。
    void SetSafetyLevel(SafetyLevel safetyLevel) NN_NOEXCEPT;
    // @brief 「安心レベル」がカスタムである時の設定値を変更します。
    void SetCustomSafetyLevelSettings(const SafetyLevelSettings& settings) NN_NOEXCEPT;
    // @brief 立体視機能の利用が制限されているかどうかを返します。
    bool GetStereoVisionRestriction() const NN_NOEXCEPT;
    // @brief 立体視機能の制限を設定します。
    // @details 暗証番号設定が無い場合は無視します。
    void SetStereoVisionRestriction(bool restricted) NN_NOEXCEPT;
    // @brief 設定を削除します。
    void DeleteSettings() NN_NOEXCEPT;
    // @brief 解除コードを設定または非設定状態にします。
    nn::Result SetPinCode(const char* pinCode, size_t pinCodeWithNullCharLength) NN_NOEXCEPT;
    // @brief 解除コードの桁数を取得します。
    int GetPinCodeLength() NN_NOEXCEPT;
    // @brief 解除コードを取得します。
    size_t GetPinCode(char* outPinCode, size_t maxLengthWithNull) NN_NOEXCEPT;
    // @brief ランダムの解除コードを生成し、「制限なし」の設定値を返します。
    void GenerateRandomPinCodeAndNoRestriction(Settings& outSettings) NN_NOEXCEPT;

    // @brief 「お問い合わせコード」を生成します。
    void GenerateInquiryCode(InquiryCode* outCode) NN_NOEXCEPT;
    // @brief マスターキーが正しいかどうかを返します。
    bool CheckMasterKey(const InquiryCode& codeData, const char* masterKey, size_t masterKeyLength) NN_NOEXCEPT;

    // @brief 内部で保持する生の Settings データを返します。
    void GetSettings(Settings& outSettings) NN_NOEXCEPT;

    // @brief settings に含まれるデータをペアコン設定データとして保存します。
    void SetSettings(const Settings& settings) NN_NOEXCEPT;

    // @brief 有効なデータの数を返します。
    size_t GetFreeCommunicationApplicationListCount() const NN_NOEXCEPT;

    // @brief 設定値データを取得し配列に格納します。
    size_t GetFreeCommunicationApplicationList(nn::pctl::FreeCommunicationApplicationInfo* outArray, size_t offset, size_t count) const NN_NOEXCEPT;

    // @brief アプリケーション別の「他の人との自由なコミュニケーション」に関する利用制限のリストを更新します。
    // @details この関数は保存を行いません。
    void UpdateFreeCommunicationApplicationList(const nn::pctl::FreeCommunicationApplicationInfo* pTitleInfoArray, size_t count) NN_NOEXCEPT;
    // @brief ApplicationIdをリストに追加して保存します。既に存在している場合は何もしません。
    // @details id がリストに存在する場合は何もしないため、制限があるアプリを指定して毎回呼び出すことができます。
    void AddFreeCommunicationApplicationList(nn::ncm::ApplicationId id) NN_NOEXCEPT;
    // @brief 特定のタイトルをリストから削除します。(主にデバッグ用)
    // @details id がリストに存在しない場合は何もしないため、制限がないアプリを指定して毎回呼び出すことができます。
    void DeleteFreeCommunicationApplicationList(nn::ncm::ApplicationId id) NN_NOEXCEPT;
    // @brief リストを空にします(リセットします)。
    void ClearFreeCommunicationApplicationList() NN_NOEXCEPT;

    // @brief 有効なデータの数を返します。
    size_t GetExemptApplicationListCount() const NN_NOEXCEPT;

    // @brief 設定値データを取得し配列に格納します。
    size_t GetExemptApplicationList(nn::pctl::ExemptApplicationInfo* outArray, size_t offset, size_t count) const NN_NOEXCEPT;

    // @brief アプリケーション別の制限対象外リストを更新します。
    // @details この関数は保存を行いません。
    void UpdateExemptApplicationList(const nn::pctl::ExemptApplicationInfo* pTitleInfoArray, size_t count) NN_NOEXCEPT;
    // @brief ApplicationIdをリストに追加して保存します。既に存在している場合は何もしません。
    // @details id がリストに存在する場合は何もしないため、対象外アプリを指定して毎回呼び出すことができます。
    void AddExemptApplicationList(nn::ncm::ApplicationId id) NN_NOEXCEPT;
    // @brief 特定のタイトルをリストから削除します。(主にデバッグ用)
    // @details id がリストに存在しない場合は何もしないため、対象外ではないアプリを指定して毎回呼び出すことができます。
    void DeleteExemptApplicationList(nn::ncm::ApplicationId id) NN_NOEXCEPT;
    // @brief リストを空にします(リセットします)。
    void ClearExemptApplicationList() NN_NOEXCEPT;

    // @brief アプリケーションが起動したことをハンドルします。
    void PostApplicationLaunched(nn::ncm::ApplicationId id, bool hasFreeCommunication) NN_NOEXCEPT;
    // @brief アプリケーションが終了したことをハンドルします。
    void PostApplicationTerminated(nn::ncm::ApplicationId id, nn::util::optional<nn::time::PosixTime> timestamp) NN_NOEXCEPT;
    // @brief アプリケーションが起動・再開判定に特定の原因で失敗したことをハンドルします。
    void PostApplicationRejected(nn::ncm::ApplicationId id) NN_NOEXCEPT;
    // @brief アプリケーションのダウンロードが開始したことをハンドルします。
    void PostApplicationDownloadStarted(nn::ncm::ApplicationId id) NN_NOEXCEPT;
    // @brief 解除コード間違いが発生したことをハンドルします。
    void PostWrongPinCode() NN_NOEXCEPT;

    // @brief すべての設定を適用して保存します。
    // @return データが適用可能で保存されたら true
    // @details unknownTitleCallback と notTitleInListCallback は
    //     FreeCommunicationApplicationSettings::UpdateDataFromList に渡されるコールバック関数です。
    bool StoreAllSettings(nn::ovln::format::PctlSettingTypeFlagSet* outChangedType,
        const Settings* pSettings,
        const FreeCommunicationApplicationSettings* pFreeCommunicationSettings,
        void(*unknownTitleCallback)(nn::ncm::ApplicationId id),
        void(*notTitleInListCallback)(nn::ncm::ApplicationId id),
        const ExemptApplicationSettings* pExemptionSettings) NN_NOEXCEPT;

    // @brief ペアコン全機能を無効化状態にします。
    // @details isRecoveryMode が true の場合(修理ツール向け)は無効化状態である旨を保存し、再起動後も無効化状態を継続させます。
    //     isRecoveryMode が false の場合は無効化状態である旨を保存せず、再起動で無効化状態を破棄します(通常の状態に戻ります)。
    bool DisableAllFeatures(bool isRecoveryMode) NN_NOEXCEPT;
    // @brief ペアコン全機能を有効化状態にする設定を保存します。保存のみ行い有効化状態自体はしません。
    bool PostEnableAllFeatures() NN_NOEXCEPT;
    // @brief ペアコン全機能が無効化されているかどうかを返します。
    bool IsAllFeaturesDisabled() const
    {
        return m_IsDisabledAll;
    }
    // @brief ペアコン全機能が無効化されているかどうかを返します。
    bool IsAllFeaturesDisabled(bool* outIsEnabledOnNextBoot) const
    {
        *outIsEnabledOnNextBoot = m_IsEnabledOnNextBoot;
        return m_IsDisabledAll;
    }

    // @brief 設定を読み込みます。
    // @details スレッドでは処理を行わず、呼び出し元スレッドでそのまま処理を行います。
    void LoadSettings() NN_NOEXCEPT;

    // @brief マスターキー用の初期化を行います。
    // @details LoadSettings より後、StartThread より前に実行する必要があります。
    void InitializeForMasterKey() NN_NOEXCEPT;

private:
    template <typename TEvent>
    class EventHolderT
    {
    public:
        EventHolderT() NN_NOEXCEPT :
            m_Event(nn::os::EventClearMode::EventClearMode_ManualClear)
        {
            nn::os::InitializeMultiWaitHolder(&m_WaitHolder, m_Event.GetBase());
        }
        ~EventHolderT() NN_NOEXCEPT
        {
            nn::os::FinalizeMultiWaitHolder(&m_WaitHolder);
        }

        TEvent m_Event;
        nn::os::MultiWaitHolderType m_WaitHolder;
    };
    typedef EventHolderT<nn::os::Event> EventHolder;

    void InitializeDefaultRatingOrganizationIfNeeded() NN_NOEXCEPT;

    void SaveGeneralSettingsNoLock() NN_NOEXCEPT;
    void SaveApplicationSettingsNoLock() NN_NOEXCEPT;
    void SaveExemptionSettingsNoLock() NN_NOEXCEPT;

    void PostSaveGeneralSettingsNoLock() NN_NOEXCEPT
    {
        m_EventHolderSaveGeneralSettings.m_Event.Signal();
    }

    void PostSaveApplicationSettingsNoLock() NN_NOEXCEPT
    {
        m_EventHolderSaveAppplicationSettings.m_Event.Signal();
    }

    void PostSaveExemptionSettingsNoLock() NN_NOEXCEPT
    {
        m_EventHolderSaveExemptionSettings.m_Event.Signal();
    }

    void ThreadProc() NN_NOEXCEPT;
    static void StaticThreadProc(void* pThis) NN_NOEXCEPT
    {
        reinterpret_cast<SettingsManager*>(pThis)->ThreadProc();
    }

    void OnApplicationLaunched(nn::ncm::ApplicationId id, bool hasFreeCommunication) NN_NOEXCEPT;
    void OnApplicationTerminated(nn::ncm::ApplicationId id, nn::util::optional<nn::time::PosixTime> timestamp) NN_NOEXCEPT;
    void OnApplicationRejected(nn::ncm::ApplicationId id) NN_NOEXCEPT;
    void OnApplicationAddedToList(nn::ncm::ApplicationId id) NN_NOEXCEPT;
    void OnApplicationRemovedFromList(nn::ncm::ApplicationId id) NN_NOEXCEPT;
    void OnApplicationDownloadStarted(nn::ncm::ApplicationId id) NN_NOEXCEPT;
    void OnWrongPinCode() NN_NOEXCEPT;
    // @brief 解除コードが変更された際の処理を行います。(SetPinCode から呼び出します。)
    void OnChangedPinCode() NN_NOEXCEPT;

    bool IsEqualPinCodeNoLock(const char* pinCode, size_t pinCodeWithNullCharLength) const NN_NOEXCEPT;

private:
    nn::os::ThreadType m_TaskThread;
    nn::os::MultiWaitType m_MultiWait;
    EventHolder m_EventHolderExit;
    EventHolder m_EventHolderSaveGeneralSettings;
    EventHolder m_EventHolderSaveAppplicationSettings;
    EventHolder m_EventHolderSaveExemptionSettings;
    EventHolder m_EventHolderApplicationLaunched;
    EventHolder m_EventHolderApplicationTerminated;
    EventHolder m_EventHolderApplicationRejected;
    EventHolder m_EventHolderApplicationAddedToList;
    EventHolder m_EventHolderApplicationRemovedFromList;
    EventHolder m_EventHolderApplicationDownloadStarted;
    EventHolder m_EventHolderWrongPinCode;
    // m_GeneralSettings の書き込み時にロックを行うための Mutex
    // (書き込んだら m_EventHolderSaveGeneralSettings がシグナルされる)
    mutable nn::os::SdkMutex m_MutexGeneralSettings;
    // m_FreeCommunicationSettings の書き込み時にロックを行うための Mutex
    // (書き込んだら m_EventHolderSaveAppplicationSettings がシグナルされる)
    mutable nn::os::SdkMutex m_MutexApplicationSettings;
    // m_ExemptionSettings の書き込み時にロックを行うための Mutex
    // (書き込んだら m_EventHolderSaveExemptionSettings がシグナルされる)
    mutable nn::os::SdkMutex m_MutexExemptionSettings;
    // m_EventHolderApplicationLaunched などのイベントやり取りを行うための Mutex
    // (イベントに付随するパラメーター受け渡しのため用いる)
    mutable nn::os::SdkMutex m_MutexMessage;
    bool m_IsDisabledAll;
    bool m_IsEnabledOnNextBoot;
    bool m_HasLaunchedApplicationFreeCommunication;
    bool m_IsDisabledAllForRecovery;
    NN_PADDING4;
    nn::ncm::ApplicationId m_ApplicationIdLaunched;
    nn::ncm::ApplicationId m_ApplicationIdTerminated;
    nn::ncm::ApplicationId m_ApplicationIdRejected;
    nn::util::optional<nn::time::PosixTime> m_TimestampTerminated;
    nn::ncm::ApplicationId m_ApplicationIdAddedToList;
    nn::ncm::ApplicationId m_ApplicationIdRemovedFromList;
    nn::ncm::ApplicationId m_ApplicationIdDownloadStarted;

    Settings m_GeneralSettings;
    FreeCommunicationApplicationSettings m_FreeCommunicationSettings;
    ExemptApplicationSettings m_ExemptionSettings;
    // マスターキー発行に用いるお問い合わせコード生成用データを一時的に保持する変数
    uint64_t m_DataForMasterKeyInquiry;
};


inline bool SettingsManager::IsRestrictionEnabled() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexGeneralSettings);
    return m_GeneralSettings.current.pinCode[0] != 0;
}

inline uint8_t SettingsManager::GetRatingAge() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexGeneralSettings);
    return m_GeneralSettings.current.ratingAge;
}

inline SafetyLevel SettingsManager::GetSafetyLevel() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexGeneralSettings);
    return static_cast<SafetyLevel>(m_GeneralSettings.current.safetyLevel);
}

inline void SettingsManager::GetCurrentSettings(SafetyLevelSettings* pSettings) const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexGeneralSettings);
    pSettings->ratingAge = m_GeneralSettings.current.ratingAge;
    pSettings->snsPostRestriction = (m_GeneralSettings.current.isSnsPostRestricted != 0);
    pSettings->freeCommunicationRestriction = (m_GeneralSettings.current.isFreeCommunicationRestrictedByDefault != 0);
}

inline void SettingsManager::GetCustomSettings(SafetyLevelSettings* pSettings) const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexGeneralSettings);
    pSettings->ratingAge = m_GeneralSettings.custom.ratingAge;
    pSettings->snsPostRestriction = (m_GeneralSettings.custom.isSnsPostRestricted != 0);
    pSettings->freeCommunicationRestriction = (m_GeneralSettings.custom.isFreeCommunicationRestrictedByDefault != 0);
}

inline bool SettingsManager::GetStereoVisionRestriction() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexGeneralSettings);
    return m_GeneralSettings.current.isStereoVisionRestricted != 0;
}

inline int SettingsManager::GetPinCodeLength() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexGeneralSettings);
    return static_cast<int>(nn::util::Strnlen(m_GeneralSettings.current.pinCode, MaxPinCodeLength));
}

inline size_t SettingsManager::GetFreeCommunicationApplicationListCount() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexApplicationSettings);
    return m_FreeCommunicationSettings.GetAvailableCount();
}

inline size_t SettingsManager::GetFreeCommunicationApplicationList(nn::pctl::FreeCommunicationApplicationInfo* outArray, size_t offset, size_t count) const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexApplicationSettings);
    return m_FreeCommunicationSettings.GetRestrictedValues(outArray, offset, count);
}

inline size_t SettingsManager::GetExemptApplicationListCount() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexExemptionSettings);
    return m_ExemptionSettings.GetAvailableCount();
}

inline size_t SettingsManager::GetExemptApplicationList(nn::pctl::ExemptApplicationInfo* outArray, size_t offset, size_t count) const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexExemptionSettings);
    return m_ExemptionSettings.GetExemptedValues(outArray, offset, count);
}

inline bool SettingsManager::IsEqualPinCodeNoLock(const char* pinCode, size_t pinCodeWithNullCharLength) const NN_NOEXCEPT
{
    if (pinCodeWithNullCharLength == 0 || !IsValidPinCode(pinCode, pinCodeWithNullCharLength - 1))
    {
        return false;
    }

    int length = nn::util::Strnlen(m_GeneralSettings.current.pinCode, MaxPinCodeLength);
    return length == static_cast<int>(pinCodeWithNullCharLength - 1) &&
        std::memcmp(m_GeneralSettings.current.pinCode, pinCode, sizeof(char) * length) == 0;
}

}}}}}
