﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/pctl/detail/service/json/pctl_Json.h>
#include <nn/nn_Result.h>
#include <nn/util/util_Optional.h>

#include <limits>

namespace nn { namespace pctl { namespace detail { namespace service { namespace json {

enum ExpectType : uint8_t
{
    ExpectType_Key = 0,
    ExpectType_AnyKey,
    ExpectType_String,
    ExpectType_Int64,
    ExpectType_Uint64,
    ExpectType_Double,
    ExpectType_Boolean,
    ExpectType_ObjectBegin,
    ExpectType_ObjectEnd,
    ExpectType_ArrayBegin,
    ExpectType_ArrayEnd,

    ExpectType_AllowAnyKey = 0x20,
    ExpectType_ArrayRepeatValues = 0x40,
    ExpectType_AllowNull = 0x80,
    ExpectType_TypeMask = 0x1F
};

/**
 * @brief 決められたJSONデータをハンドルするために用いるデータ定義用構造体です。
 *        NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS ～ NN_DETAIL_PCTL_JSON_END_EXPECTS を用いて定義します。
 */
struct ExpectDataDefinition
{
    void* data;
    uint8_t type;
};

/**
 * @brief 事前に定義されたJSONデータをハンドルするためのクラスです。
 */
class JsonDataHandler
{
public:
    static const int MaxDepth = 32;

    JsonDataHandler() NN_NOEXCEPT :
        m_Param(nullptr),
        m_pDataArray(nullptr),
        m_DataCount(0),
        m_CurrentDepth(0)
    {
    }

    JsonDataHandler(void* param, const ExpectDataDefinition* dataArray, int dataCount) NN_NOEXCEPT :
        m_Param(param),
        m_pDataArray(dataArray),
        m_DataCount(dataCount),
        m_CurrentDepth(0)
    {
        InitializeHandler();
    }

    template <size_t N>
    JsonDataHandler(void* param, const ExpectDataDefinition (& dataArray)[N]) NN_NOEXCEPT :
        m_Param(param),
        m_pDataArray(dataArray),
        m_DataCount(static_cast<int>(N)),
        m_CurrentDepth(0)
    {
        NN_STATIC_ASSERT(N <= INT_MAX);
        InitializeHandler();
    }

    void InitializeHandler() NN_NOEXCEPT
    {
        auto& info = m_DepthInfo[0];
        info.subArray = m_pDataArray;
        info.subArrayLength = m_DataCount;
        info.currentPosition = 0;
        info.index = 0;
        info.isObject = false;
        info.isRepeat = false;
    }

    void SetExpectData(void* param, const ExpectDataDefinition* dataArray, int dataCount) NN_NOEXCEPT
    {
        m_Param = param;
        m_pDataArray = dataArray;
        m_DataCount = dataCount;
        m_CurrentDepth = 0;
        InitializeHandler();
    }

    template <size_t N>
    void SetExpectData(void* param, const ExpectDataDefinition (& dataArray)[N]) NN_NOEXCEPT
    {
        NN_STATIC_ASSERT(N <= INT_MAX);
        m_Param = param;
        m_pDataArray = dataArray;
        m_DataCount = static_cast<int>(N);
        m_CurrentDepth = 0;
        InitializeHandler();
    }

public:
    /*!
        @brief      オブジェクトの開始イベントです。

        @return     処理を継続するかどうか。

        @details
                    本関数は、RapidJSON 用関数です。
    */
    bool StartObject() NN_NOEXCEPT;

    /*!
        @brief      オブジェクトの終了イベントです。

        @param[in]  numObjects  オブジェクト数。

        @return     処理を継続するかどうか。

        @details
                    本関数は、RapidJSON 用関数です。
    */
    bool EndObject(RAPIDJSON_NAMESPACE::SizeType numObjects) NN_NOEXCEPT;

    /*!
        @brief      配列の開始イベントです。

        @return     処理を継続するかどうか。

        @details
                    本関数は、RapidJSON 用関数です。
    */
    bool StartArray() NN_NOEXCEPT;

    /*!
        @brief      配列の終了イベントです。

        @param[in]  numElements 要素数。

        @return     処理を継続するかどうか。

        @details
                    本関数は、RapidJSON 用関数です。
    */
    bool EndArray(RAPIDJSON_NAMESPACE::SizeType numElements) NN_NOEXCEPT;

    /*!
        @brief      キーの出現イベントです。

        @param[in]  key     キー。
        @param[in]  length  文字列長。

        @return     処理を継続するかどうか。

        @details
                    本関数は、RapidJSON 用関数です。
    */
    bool Key(const char* key, RAPIDJSON_NAMESPACE::SizeType length, bool) NN_NOEXCEPT;

    /*!
        @brief      値（null）の出現イベントです。

        @return     処理を継続するかどうか。
    */
    bool Null() NN_NOEXCEPT;

    /*!
        @brief      値（bool）の出現イベントです。

        @param[in]  value   値。

        @return     処理を継続するかどうか。
    */
    bool Bool(bool value) NN_NOEXCEPT;

    /*!
        @brief      値（int32_t）の出現イベントです。

        @param[in]  value   値。

        @return     処理を継続するかどうか。
    */
    bool Int(int32_t value) NN_NOEXCEPT;

    /*!
        @brief      値（uint32_t）の出現イベントです。

        @param[in]  value   値。

        @return     処理を継続するかどうか。
    */
    bool Uint(uint32_t value) NN_NOEXCEPT;

    /*!
        @brief      値（int64_t）の出現イベントです。

        @param[in]  value   値。

        @return     処理を継続するかどうか。
    */
    bool Int64(int64_t value) NN_NOEXCEPT;

    /*!
        @brief      値（uint64_t）の出現イベントです。

        @param[in]  value   値。

        @return     処理を継続するかどうか。
    */
    bool Uint64(uint64_t value) NN_NOEXCEPT;

    /*!
        @brief      値（double）の出現イベントです。

        @param[in]  value   値。

        @return     処理を継続するかどうか。
    */
    bool Double(double value) NN_NOEXCEPT;

    /*!
        @brief      値（string）の出現イベントです。

        @param[in]  value   値。
        @param[in]  length  文字列長。

        @return     処理を継続するかどうか。
    */
    bool String(const char* value, RAPIDJSON_NAMESPACE::SizeType length, bool) NN_NOEXCEPT;

protected:
    void* GetParam() const NN_NOEXCEPT
    {
        return m_Param;
    }

private:
    // 現在の階層に対するチェック位置を更新します。
    void UpdateCurrentPositionForCurrentDepth(int lastPosition) NN_NOEXCEPT;
    // startPosition にあるデータに対してその次のチェック位置を計算します。
    // startPosition のデータがObject/Arrayの場合はその終了位置も計算します。
    int CalculateNextPosition(int startPosition) const NN_NOEXCEPT;
    // StartObject/StartArray時にそれらを無視することができる状態である場合、
    // 階層の情報を更新して値をすべて無視するようにします。
    bool TryToIgnoreObjectOrArray(bool isObject) NN_NOEXCEPT;
    // 現在の階層においてキーや値を無視できる状態かどうかを返し、
    // 必要に応じて状態を更新します。
    bool TryToIgnoreKeyValue() NN_NOEXCEPT;
    // 次に来るデータを無視する状態に設定します。
    void SetIgnoreNextData() NN_NOEXCEPT;

    void* m_Param;
    const ExpectDataDefinition* m_pDataArray;
    int m_DataCount;
    int m_CurrentDepth;
    struct DepthInfo
    {
        const ExpectDataDefinition* subArray;
        int subArrayLength;
        // isObject == true && currentPosition == startPosition の場合は「無視状態」として
        // 次に来るキーや値を無視する処理を行う
        int currentPosition;
        int index;
        bool isObject;
        bool isRepeat;
    };
    DepthInfo m_DepthInfo[MaxDepth];
};

/**
 * @brief 文字列値を受け付けるコールバック関数の型です。
 * @param[out] outIgnoreValue 値を無視するかどうかを受け取るポインター。
 *     true を返すと値の部分を無視します。
 * @param[in] param 解析開始時に指定される任意のデータ
 * @param[in] index 同じ階層内で何番目にヒットしたかを表す値(開始値は 0)
 * @param[in] value 解析された値
 * @param[in] length value の長さ
 * @return 正しく受け付けられたら true、それ以外であれば false
 */
typedef bool (* HandleAnyKeyFunction)(bool* outIgnoreValue, void* param, int index, const char* value, size_t length);
/**
 * @brief 文字列値を受け付けるコールバック関数の型です。
 * @param[in] param 解析開始時に指定される任意のデータ
 * @param[in] index 同じ階層内で何番目にヒットしたかを表す値(開始値は 0)
 * @param[in] value 解析された値。nullptr であれば null 値。
 * @param[in] length value の長さ
 * @return 正しく受け付けられたら true、それ以外であれば false
 */
typedef bool (* HandleStringValueFunction)(void* param, int index, const char* value, size_t length);
/**
 * @brief int64_t 値を受け付けるコールバック関数の型です。
 * @param[in] param 解析開始時に指定される任意のデータ
 * @param[in] index 同じ階層内で何番目にヒットしたかを表す値(開始値は 0)
 * @param[in] value 解析された値
 * @return 正しく受け付けられたら true、それ以外であれば false
 */
typedef bool (* HandleInt64ValueFunction)(void* param, int index, int64_t value);
/**
 * @brief int64_t 値または null 値を受け付けるコールバック関数の型です。
 * @param[in] param 解析開始時に指定される任意のデータ
 * @param[in] index 同じ階層内で何番目にヒットしたかを表す値(開始値は 0)
 * @param[in] value 解析された値。nn::util::nullopt であれば null 値。
 * @return 正しく受け付けられたら true、それ以外であれば false
 */
typedef bool (* HandleInt64OrNullValueFunction)(void* param, int index, nn::util::optional<int64_t> value);
/**
 * @brief uint64_t 値を受け付けるコールバック関数の型です。
 * @param[in] param 解析開始時に指定される任意のデータ
 * @param[in] index 同じ階層内で何番目にヒットしたかを表す値(開始値は 0)
 * @param[in] value 解析された値
 * @return 正しく受け付けられたら true、それ以外であれば false
 */
typedef bool (* HandleUint64ValueFunction)(void* param, int index, uint64_t value);
/**
 * @brief uint64_t 値または null 値を受け付けるコールバック関数の型です。
 * @param[in] param 解析開始時に指定される任意のデータ
 * @param[in] index 同じ階層内で何番目にヒットしたかを表す値(開始値は 0)
 * @param[in] value 解析された値。nn::util::nullopt であれば null 値。
 * @return 正しく受け付けられたら true、それ以外であれば false
 */
typedef bool (* HandleUint64OrNullValueFunction)(void* param, int index, nn::util::optional<uint64_t> value);
/**
 * @brief double 値を受け付けるコールバック関数の型です。
 * @param[in] param 解析開始時に指定される任意のデータ
 * @param[in] index 同じ階層内で何番目にヒットしたかを表す値(開始値は 0)
 * @param[in] value 解析された値
 * @return 正しく受け付けられたら true、それ以外であれば false
 */
typedef bool (* HandleDoubleValueFunction)(void* param, int index, double value);
/**
 * @brief double 値または null 値を受け付けるコールバック関数の型です。
 * @param[in] param 解析開始時に指定される任意のデータ
 * @param[in] index 同じ階層内で何番目にヒットしたかを表す値(開始値は 0)
 * @param[in] value 解析された値。nn::util::nullopt であれば null 値。
 * @return 正しく受け付けられたら true、それ以外であれば false
 */
typedef bool (* HandleDoubleOrNullValueFunction)(void* param, int index, nn::util::optional<double> value);
/**
 * @brief bool 値を受け付けるコールバック関数の型です。
 * @param[in] param 解析開始時に指定される任意のデータ
 * @param[in] index 同じ階層内で何番目にヒットしたかを表す値(開始値は 0)
 * @param[in] value 解析された値
 * @return 正しく受け付けられたら true、それ以外であれば false
 */
typedef bool (* HandleBooleanValueFunction)(void* param, int index, bool value);
/**
 * @brief bool 値または null 値を受け付けるコールバック関数の型です。
 * @param[in] param 解析開始時に指定される任意のデータ
 * @param[in] index 同じ階層内で何番目にヒットしたかを表す値(開始値は 0)
 * @param[in] value 解析された値。nn::util::nullopt であれば null 値。
 * @return 正しく受け付けられたら true、それ以外であれば false
 */
typedef bool (* HandleBooleanOrNullValueFunction)(void* param, int index, nn::util::optional<bool> value);
/**
 * @brief オブジェクト値を受け付けるコールバック関数の型です。深い階層の値読み取りが続きます。
 * @param[in] param 解析開始時に指定される任意のデータ
 * @param[in] index 同じ階層内で何番目にヒットしたかを表す値(開始値は 0)
 * @return 正しく受け付けられるなら true、それ以外であれば false
 */
typedef bool (* HandleObjectValueFunction)(void* param, int index);
/**
 * @brief 配列値を受け付けるコールバック関数の型です。深い階層の値読み取りが続きます。
 * @param[in] param 解析開始時に指定される任意のデータ
 * @param[in] index 同じ階層内で何番目にヒットしたかを表す値(開始値は 0)
 * @return 正しく受け付けられるなら true、それ以外であれば false
 */
typedef bool (* HandleArrayValueFunction)(void* param, int index);

}}}}}

/**
 * @brief JSONの期待フォーマット定義を開始します。
 * @param specifier 'static' や 'const' などの指定
 * @param dataName 変数名
 */
#define NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS(specifier, dataName) \
    specifier ::nn::pctl::detail::service::json::ExpectDataDefinition dataName[] = {
/**
 * @brief JSONの期待フォーマット定義を終了します。
 */
#define NN_DETAIL_PCTL_JSON_END_EXPECTS() \
    };
/**
 * @brief 指定した種類を期待します。
 */
#define NN_DETAIL_PCTL_JSON_EXPECT(type, data) \
    { (data), static_cast<uint8_t>((type)) },
/**
 * @brief 指定したキー名を期待します。
 */
#define NN_DETAIL_PCTL_JSON_EXPECT_KEY(name) \
    NN_DETAIL_PCTL_JSON_EXPECT(::nn::pctl::detail::service::json::ExpectType::ExpectType_Key, const_cast<char*>((name)))
/**
 * @brief 任意のキー名を期待します。
 * @details
 * NN_DETAIL_PCTL_JSON_EXPECT_KEY で一致するキー名が現れた場合はそちらを優先します。
 */
#define NN_DETAIL_PCTL_JSON_EXPECT_ANY_KEY(func) \
    NN_DETAIL_PCTL_JSON_EXPECT(::nn::pctl::detail::service::json::ExpectType::ExpectType_AnyKey, reinterpret_cast<void*>(static_cast<::nn::pctl::detail::service::json::HandleAnyKeyFunction>((func))))
/**
 * @brief 文字列値を期待します。ヒットした場合指定の関数を呼び出します。
 */
#define NN_DETAIL_PCTL_JSON_EXPECT_VALUE_STRING(func) \
    NN_DETAIL_PCTL_JSON_EXPECT(::nn::pctl::detail::service::json::ExpectType::ExpectType_String, reinterpret_cast<void*>(static_cast<::nn::pctl::detail::service::json::HandleStringValueFunction>((func))))
/**
 * @brief 数値を期待します。ヒットした場合符号付き64ビット整数として指定の関数を呼び出します。
 */
#define NN_DETAIL_PCTL_JSON_EXPECT_VALUE_INT64(func) \
    NN_DETAIL_PCTL_JSON_EXPECT(::nn::pctl::detail::service::json::ExpectType::ExpectType_Int64, reinterpret_cast<void*>(static_cast<::nn::pctl::detail::service::json::HandleInt64ValueFunction>((func))))
/**
 * @brief 数値または null 値を期待します。数値がヒットした場合符号付き64ビット整数として、null 値がヒットした場合は nn::util::nullopt を用いて指定の関数を呼び出します。
 */
#define NN_DETAIL_PCTL_JSON_EXPECT_VALUE_INT64_NULLABLE(func) \
    NN_DETAIL_PCTL_JSON_EXPECT(::nn::pctl::detail::service::json::ExpectType::ExpectType_Int64 | ::nn::pctl::detail::service::json::ExpectType::ExpectType_AllowNull, reinterpret_cast<void*>(static_cast<::nn::pctl::detail::service::json::HandleInt64OrNullValueFunction>((func))))
/**
 * @brief 数値を期待します。ヒットした場合符号なし64ビット整数として指定の関数を呼び出します。
 */
#define NN_DETAIL_PCTL_JSON_EXPECT_VALUE_UINT64(func) \
    NN_DETAIL_PCTL_JSON_EXPECT(::nn::pctl::detail::service::json::ExpectType::ExpectType_Uint64, reinterpret_cast<void*>(static_cast<::nn::pctl::detail::service::json::HandleUint64ValueFunction>((func))))
/**
 * @brief 数値または null 値を期待します。数値がヒットした場合符号なし64ビット整数として、null 値がヒットした場合は nn::util::nullopt を用いて指定の関数を呼び出します。
 */
#define NN_DETAIL_PCTL_JSON_EXPECT_VALUE_UINT64_NULLABLE(func) \
    NN_DETAIL_PCTL_JSON_EXPECT(::nn::pctl::detail::service::json::ExpectType::ExpectType_Uint64 | ::nn::pctl::detail::service::json::ExpectType::ExpectType_AllowNull, reinterpret_cast<void*>(static_cast<::nn::pctl::detail::service::json::HandleUint64OrNullValueFunction>((func))))
/**
 * @brief 数値を期待します。ヒットした場合倍精度浮動小数点の値として指定の関数を呼び出します。
 */
#define NN_DETAIL_PCTL_JSON_EXPECT_VALUE_DOUBLE(func) \
    NN_DETAIL_PCTL_JSON_EXPECT(::nn::pctl::detail::service::json::ExpectType::ExpectType_Double, reinterpret_cast<void*>(static_cast<::nn::pctl::detail::service::json::HandleDoubleValueFunction>((func))))
/**
 * @brief 数値または null 値を期待します。数値がヒットした場合倍精度浮動小数点の値として、null 値がヒットした場合は nn::util::nullopt を用いて指定の関数を呼び出します。
 */
#define NN_DETAIL_PCTL_JSON_EXPECT_VALUE_DOUBLE_NULLABLE(func) \
    NN_DETAIL_PCTL_JSON_EXPECT(::nn::pctl::detail::service::json::ExpectType::ExpectType_Double | ::nn::pctl::detail::service::json::ExpectType::ExpectType_AllowNull, reinterpret_cast<void*>(static_cast<::nn::pctl::detail::service::json::HandleDoubleOrNullValueFunction>((func))))
/**
 * @brief bool 値を期待します。ヒットした場合指定の関数を呼び出します。
 */
#define NN_DETAIL_PCTL_JSON_EXPECT_VALUE_BOOLEAN(func) \
    NN_DETAIL_PCTL_JSON_EXPECT(::nn::pctl::detail::service::json::ExpectType::ExpectType_Boolean, reinterpret_cast<void*>(static_cast<::nn::pctl::detail::service::json::HandleBooleanValueFunction>((func))))
/**
 * @brief bool 値または null 値を期待します。ヒットした場合 bool 値であればその値を、null 値がヒットした場合は nn::util::nullopt を用いて指定の関数を呼び出します。
 */
#define NN_DETAIL_PCTL_JSON_EXPECT_VALUE_BOOLEAN_NULLABLE(func) \
    NN_DETAIL_PCTL_JSON_EXPECT(::nn::pctl::detail::service::json::ExpectType::ExpectType_Boolean | ::nn::pctl::detail::service::json::ExpectType::ExpectType_AllowNull, reinterpret_cast<void*>(static_cast<::nn::pctl::detail::service::json::HandleBooleanOrNullValueFunction>((func))))
/**
 * @brief オブジェクトの開始を期待します。未知のキーを許可します。
 */
#define NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN(isNullAllowed) \
    NN_DETAIL_PCTL_JSON_EXPECT(::nn::pctl::detail::service::json::ExpectType::ExpectType_ObjectBegin | ::nn::pctl::detail::service::json::ExpectType::ExpectType_AllowAnyKey | ((isNullAllowed) ? (::nn::pctl::detail::service::json::ExpectType::ExpectType_AllowNull) : 0), nullptr)
/**
 * @brief オブジェクトの開始を期待します。未知のキーを許可します。
 *        また、開始時にコールバック呼び出しを行います。
 */
#define NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN_CALLBACK(isNullAllowed, func) \
    NN_DETAIL_PCTL_JSON_EXPECT(::nn::pctl::detail::service::json::ExpectType::ExpectType_ObjectBegin | ::nn::pctl::detail::service::json::ExpectType::ExpectType_AllowAnyKey | ((isNullAllowed) ? (::nn::pctl::detail::service::json::ExpectType::ExpectType_AllowNull) : 0), reinterpret_cast<void*>(static_cast<::nn::pctl::detail::service::json::HandleObjectValueFunction>((func))))
/**
 * @brief オブジェクトの開始を期待します。決まったキーのみを受け付け、未知のキーを許可しません。(キーが現れなかったかどうかは判定しません。)
 */
#define NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN_FIXED(isNullAllowed) \
    NN_DETAIL_PCTL_JSON_EXPECT(::nn::pctl::detail::service::json::ExpectType::ExpectType_ObjectBegin | ((isNullAllowed) ? (::nn::pctl::detail::service::json::ExpectType::ExpectType_AllowNull) : 0), nullptr)
/**
 * @brief オブジェクトの開始を期待します。決まったキーのみを受け付け、未知のキーを許可しません。(キーが現れなかったかどうかは判定しません。)
 *        また、開始時にコールバック呼び出しを行います。
 */
#define NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN_FIXED_CALLBACK(isNullAllowed, func) \
    NN_DETAIL_PCTL_JSON_EXPECT(::nn::pctl::detail::service::json::ExpectType::ExpectType_ObjectBegin | ((isNullAllowed) ? (::nn::pctl::detail::service::json::ExpectType::ExpectType_AllowNull) : 0), reinterpret_cast<void*>(static_cast<::nn::pctl::detail::service::json::HandleObjectValueFunction>((func))))
/**
 * @brief オブジェクトの終了を期待します。
 */
#define NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END() \
    NN_DETAIL_PCTL_JSON_EXPECT(::nn::pctl::detail::service::json::ExpectType::ExpectType_ObjectEnd, nullptr)
/**
 * @brief 配列の開始を期待します。配列に含まれる値は1種類固定(配列内の先頭データを利用)で任意の要素数を受け付けます。
 */
#define NN_DETAIL_PCTL_JSON_EXPECT_ARRAY_BEGIN(isNullAllowed) \
    NN_DETAIL_PCTL_JSON_EXPECT(::nn::pctl::detail::service::json::ExpectType::ExpectType_ArrayBegin | ::nn::pctl::detail::service::json::ExpectType::ExpectType_ArrayRepeatValues | ((isNullAllowed) ? (::nn::pctl::detail::service::json::ExpectType::ExpectType_AllowNull) : 0), nullptr)
/**
 * @brief 配列の開始を期待します。配列に含まれる値は1種類固定(配列内の先頭データを利用)で任意の要素数を受け付けます。
 *        また、開始時にコールバック呼び出しを行います。
 */
#define NN_DETAIL_PCTL_JSON_EXPECT_ARRAY_BEGIN_CALLBACK(isNullAllowed, func) \
    NN_DETAIL_PCTL_JSON_EXPECT(::nn::pctl::detail::service::json::ExpectType::ExpectType_ArrayBegin | ::nn::pctl::detail::service::json::ExpectType::ExpectType_ArrayRepeatValues | ((isNullAllowed) ? (::nn::pctl::detail::service::json::ExpectType::ExpectType_AllowNull) : 0), reinterpret_cast<void*>(static_cast<::nn::pctl::detail::service::json::HandleArrayValueFunction>((func))))
/**
 * @brief 配列の開始を期待します。固定長の配列を受け付けます。
 */
#define NN_DETAIL_PCTL_JSON_EXPECT_ARRAY_BEGIN_FIXED(isNullAllowed) \
    NN_DETAIL_PCTL_JSON_EXPECT(::nn::pctl::detail::service::json::ExpectType::ExpectType_ArrayBegin | ((isNullAllowed) ? (::nn::pctl::detail::service::json::ExpectType::ExpectType_AllowNull) : 0), nullptr)
/**
 * @brief 配列の開始を期待します。固定長の配列を受け付けます。
 *        また、開始時にコールバック呼び出しを行います。
 */
#define NN_DETAIL_PCTL_JSON_EXPECT_ARRAY_BEGIN_FIXED_CALLBACK(isNullAllowed, func) \
    NN_DETAIL_PCTL_JSON_EXPECT(::nn::pctl::detail::service::json::ExpectType::ExpectType_ArrayBegin | ((isNullAllowed) ? (::nn::pctl::detail::service::json::ExpectType::ExpectType_AllowNull) : 0), reinterpret_cast<void*>(static_cast<::nn::pctl::detail::service::json::HandleArrayValueFunction>((func))))
/**
 * @brief 配列の終了を期待します。
 */
#define NN_DETAIL_PCTL_JSON_EXPECT_ARRAY_END() \
    NN_DETAIL_PCTL_JSON_EXPECT(::nn::pctl::detail::service::json::ExpectType::ExpectType_ArrayEnd, nullptr)
