﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/nn_SdkAssert.h>
#include <nn/util/util_LockGuard.h>
#include <atomic>

namespace nn { namespace pctl { namespace detail { namespace service { namespace common {

/*!
    @brief      キャンセル可能オブジェクトです。
*/
class Cancelable
{
public:
    class Listener
    {
    public:
        virtual void SignalFromPctl() NN_NOEXCEPT = 0;
    };

public:
    /*!
        @brief      コンストラクタです。
    */
    Cancelable() NN_NOEXCEPT
        : m_IsCanceled(false),
          m_pListener(nullptr)
    {
    }

    /*!
        @brief      キャンセルします。
    */
    void Cancel() NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(m_Lock);
        m_IsCanceled = true;

        if (m_pListener)
        {
            m_pListener->SignalFromPctl();
        }
    }

    /*!
        @brief      キャンセルした状態をリセットします。
    */
    void Reset() NN_NOEXCEPT
    {
        m_IsCanceled = false;
    }

    /*!
        @brief      キャンセルされたかどうかを取得します。

        @return     キャンセルされたかどうか。
    */
    bool IsCanceled() const NN_NOEXCEPT
    {
        return m_IsCanceled;
    }

    inline void AttachListener(Listener* pListener) NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(m_Lock);
        NN_SDK_ASSERT(pListener);
        NN_SDK_ASSERT(!m_pListener);
        m_pListener = pListener;

        if (m_IsCanceled)
        {
            m_pListener->SignalFromPctl();
        }
    }

    inline void DetachListener() NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(m_Lock);
        NN_SDK_ASSERT(m_pListener);
        m_pListener = nullptr;
    }

private:
    //
    mutable os::SdkMutex m_Lock;
    std::atomic<bool> m_IsCanceled;
    Listener* m_pListener;
};

}}}}}
