﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/omm/omm_Api.h>
#include <nn/omm/srv/omm_Common.private.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_Thread.h>
#include <nn/usb/pd/usb_Pd.h>
#include <nn/usb/pd/usb_PdCradleLib.h>
#include <nn/usb/pd/usb_PdCradleVdm.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace omm { namespace srv {

    class MultiWaitEvent
    {
    public:
        MultiWaitEvent() NN_NOEXCEPT;

        template<typename T>
        int Link(T* eventType) NN_NOEXCEPT
        {
            NN_SDK_ASSERT(m_Count < MaxWaitHolderCount);

            os::InitializeMultiWaitHolder(&m_MultiWaitHolder[m_Count], eventType);
            os::LinkMultiWaitHolder(&m_MultiWait, &m_MultiWaitHolder[m_Count]);

            int outIndex = m_Count;
            m_Count++;

            return outIndex;
        }

        int WaitAny() NN_NOEXCEPT;

    private:
        static const int MaxWaitHolderCount = 8;
        os::MultiWaitType m_MultiWait;
        os::MultiWaitHolderType m_MultiWaitHolder[MaxWaitHolderCount];
        int m_Count;
    };

    class CradleObserver
    {
    public:
        class IHandler
        {
        public:
            virtual void OnMounted() NN_NOEXCEPT = 0;
            virtual void OnUnmounted() NN_NOEXCEPT = 0;
        };

        explicit CradleObserver(IHandler* handler) NN_NOEXCEPT;
        ~CradleObserver() NN_NOEXCEPT;
        void Initialize(const usb::pd::CradleSession& cradleSession, ConsoleStyle style) NN_NOEXCEPT;
        void Start() NN_NOEXCEPT;
        void Stop() NN_NOEXCEPT;
        bool IsStarted() NN_NOEXCEPT
        {
            return m_IsStarted;
        }
        bool HasCradle() NN_NOEXCEPT;
        bool IsGoodCradleMounted() NN_NOEXCEPT;
        bool IsAnyCradleMounted() NN_NOEXCEPT;
        CradleStatus GetCradleStatus() NN_NOEXCEPT;
        void HandleStateExplicitly() NN_NOEXCEPT;

        Result GetCradleFwVersion(
            usb::pd::VdmPdcHFwVersion*      pOutPdcHFwVersion,
            usb::pd::VdmPdcAFwVersion*      pOutPdcAFwVersion,
            usb::pd::VdmMcuFwVersion*       pOutMcuFwVersion,
            usb::pd::VdmDp2HdmiFwVersion*   pOutDp2HdmiVersion) NN_NOEXCEPT;

    private:
        enum class State : Bit8
        {
            Boot,
            NotMounted,
            Mounted,
            Stop
        };

        IHandler* m_Handler;
        os::ThreadType m_Thread;
        bool m_IsStarted;
        State m_State;

        enum class Signal : Bit8
        {
            Status,
            Stop,
            Timeout
        };

        Signal Wait() NN_NOEXCEPT;

        MultiWaitEvent m_MultiWait;
        int m_StopIndex;
        int m_StatusIndex;
        os::Event m_StopEvent;
        os::SystemEventType m_CradleStatusEvent;
        usb::pd::Session m_PdSession;
        usb::pd::CradleSession m_PdCradleSession;

        void HandleState() NN_NOEXCEPT;
        void ExecuteObservationLoop() NN_NOEXCEPT;

        State HandleBootState() NN_NOEXCEPT;
        State HandleNotMountedState() NN_NOEXCEPT;
        State HandleWaitChargedState() NN_NOEXCEPT;
        State HandleMountedState() NN_NOEXCEPT;

        Result SubmitErrorReportCradleFirmwareContext() NN_NOEXCEPT;

        ConsoleStyle m_ConsoleStyle;
    };
}}}
