﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <array>
#include <nn/account/account_Types.h>
#include <nn/nn_Common.h>
#include <nn/olsc/srv/util/olsc_MountContext.h>
#include <nn/olsc/srv/util/olsc_MountTypes.h>
#include <nn/os/os_ReaderWriterLock.h>
#include <nn/os/os_SdkConditionVariable.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace olsc { namespace srv { namespace util {

class MountManagerBase
{
public:
    virtual void Release(MounterId mounter, ReferenceMode referenceMode) NN_NOEXCEPT = 0;
    virtual const char* GetMountName(MounterId mounter) const NN_NOEXCEPT = 0;
    virtual const char* GetRootPath(MounterId mounter) const NN_NOEXCEPT = 0;

    template<typename MountContextType>
    MountContextType GetMountContext(MounterId mounter) NN_NOEXCEPT
    {
        return MountContextType(this, mounter);
    }
};

template<typename TagType, int MaxTagCount, int MaxMounterCount>
class MountManager : public MountManagerBase
{
public:
    MountManager() NN_NOEXCEPT;
    void RegisterTag(TagType tag, const MountInfo& mountInfo) NN_NOEXCEPT;
    nn::util::optional<MounterId> TryAcquire(TagType tag, const account::Uid& uid, ReferenceMode referenceMode) NN_NOEXCEPT;
    MounterId Acquire(TagType tag, const account::Uid& uid, ReferenceMode referenceMode) NN_NOEXCEPT;


    virtual void Release(MounterId mounter, ReferenceMode referenceMode) NN_NOEXCEPT NN_OVERRIDE;
    virtual const char* GetMountName(MounterId mounter) const NN_NOEXCEPT NN_OVERRIDE;
    virtual const char* GetRootPath(MounterId mounter) const NN_NOEXCEPT NN_OVERRIDE;

    ReadMount AcquireForRead(TagType tag, const account::Uid& uid) NN_NOEXCEPT;
    WriteMount AcquireForWrite(TagType tag, const account::Uid& uid) NN_NOEXCEPT;

private:
    class Mounter
    {
    public:
        Mounter(const account::Uid& uid, TagType tag) NN_NOEXCEPT;
        ~Mounter() NN_NOEXCEPT;

        TagType GetTag() const NN_NOEXCEPT;
        const account::Uid& GetUid() const NN_NOEXCEPT;
        ReferenceMode GetReferenceMode() const NN_NOEXCEPT;
        int GetReferenceCount() const NN_NOEXCEPT;

        bool IsReferenced() const NN_NOEXCEPT;
        const char* GetMountName() const NN_NOEXCEPT;
        const char* GetRootPath() const NN_NOEXCEPT;
        MounterId GetId() const NN_NOEXCEPT;

        Result EnsureAndMount(const MountInfo& mountInfo) NN_NOEXCEPT;

        bool TryLock(ReferenceMode referenceMode) NN_NOEXCEPT;
        void Unlock(ReferenceMode referenceMode) NN_NOEXCEPT;
    private:
        Result Ensure(const MountInfo& mountInfo) NN_NOEXCEPT;
        Result Mount(const MountInfo& mountInfo) NN_NOEXCEPT;
        void Unmount() NN_NOEXCEPT;
        void InitializeId() NN_NOEXCEPT;

        account::Uid m_Uid;
        TagType m_Tag;
        ReferenceMode m_ReferenceMode;
        int m_RefCount;
        char m_MountName[16];
        char m_RootPath[16];
        MounterId m_Id;
        os::ReaderWriterLock m_ReaderWriterLock;
    };

    struct RegisteredTag
    {
        TagType tag;
        MountInfo mountInfo;
    };

    std::array<RegisteredTag, MaxTagCount> m_RegisteredTags;
    int m_RegisteredTagCount;
    os::SdkRecursiveMutex m_TagListLock;
    std::array<nn::util::optional<Mounter>, MaxMounterCount> m_Mounters;

    mutable os::SdkRecursiveMutex m_MounterLock;
    os::SdkConditionVariable m_AcquiableCondition;

    using MounterHolder = nn::util::optional<Mounter>;
    Mounter* FindMounterById(MounterId id) NN_NOEXCEPT;
    const Mounter* FindMounterById(MounterId id) const NN_NOEXCEPT;

    const RegisteredTag& GetRegisterdTag(TagType tag) NN_NOEXCEPT;
    void LogCurrentState() const NN_NOEXCEPT;
    void LogAcquire(nn::util::optional<MounterId> result, TagType tag, const account::Uid& uid, ReferenceMode referenceMode) const NN_NOEXCEPT;
    void LogTryAcquireFailed(TagType tag, const account::Uid& uid, ReferenceMode referenceMode) const NN_NOEXCEPT;
};

enum class DefaultSaveDataTag : int
{
    DeviceSave = 0,
    UserSettingsSave = 1,
    UserSeriesInfoSave = 2,
};

class DefaultMountManager : public MountManager<DefaultSaveDataTag, 3, 6>
{
public:
    DefaultMountManager(const MountInfo& deviceSaveInfo, const MountInfo& userSttingSaveInfo, const MountInfo& userSeriesInfoSaveInfo) NN_NOEXCEPT;
    ReadMount AcquireDeviceSaveForRead() NN_NOEXCEPT;
    WriteMount AcquireDeviceSaveForWrite() NN_NOEXCEPT;

    ReadMount AcquireUserSettingsSaveForRead(const account::Uid& uid) NN_NOEXCEPT;
    WriteMount AcquireUserSettingsSaveForWrite(const account::Uid& uid) NN_NOEXCEPT;

    ReadMount AcquireUserSeriesInfoSaveForRead(const account::Uid& uid) NN_NOEXCEPT;
    WriteMount AcquireUserSeriesInfoSaveForWrite(const account::Uid& uid) NN_NOEXCEPT;
};

}}}}

#include <nn/olsc/srv/util/olsc_MountManagerImpl.h>
