﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 *  @file
 *  @brief  サーバーに対して発行する API です。
 */

#include <nn/account/account_Types.h>
#include <nn/nn_ApplicationId.h>

#include <nn/olsc/srv/database/olsc_SaveDataArchiveInfoCache.h>
#include <nn/olsc/srv/olsc_InternalTypes.h>

#include <nn/util/util_Execution.h>

#include <curl/curl.h>

namespace nn { namespace olsc { namespace srv { namespace transfer {

    /**
    *       @brief      ストレージサーバに対して、セーブデータの新規アップロード開始を要求します。
    *
    *       @param[out]  out   サーバーから返ってくる SaveDataArchive。
    *       @param[in]   uid   SDA を作成するユーザー情報。
    *       @param[in]   appId   SDA を作成するアプリ情報。
    *       @param[in]   info   作成するセーブデータの SeriesInfo。
    *       @param[in]   dataSize   作成するセーブデータのサイズ。
    *       @param[in]   updatedTime   最後にデータが更新された日時。
    *       @param[in]   nsaIdToken   使用する NSA ID トークン。
    *       @param[in]   curlHandle   内部の通信処理に使用する libcURL のハンドル。
    *       @param[in]   workBuffer   ワークバッファ。
    *       @param[in]   workBufferSize   ワークバッファのサイズ。
    *       @param[in]   pCancelable   中止を検知するためのオブジェクト。

    *
    *       @return     処理結果を返します。
    *
    */
    Result StartSaveDataArchiveUpload(SaveDataArchiveInfo* out, const account::Uid& uid, ApplicationId appId, const SeriesInfo& info, size_t dataSize, time::PosixTime updatedTime, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT;

    /**
    *       @brief      ストレージサーバに対して、セーブデータの差分アップロード開始を要求します。
    *
    *       @param[out]  pOutSda   サーバーから返ってくる SaveDataArchiveInfo。
    *       @param[out]  pOutCfCount   サーバーから返ってくる ComponentFileInfo の数。
    *       @param[in]   componentFileList   サーバーから返ってくる ComponentFileInfo のリスト。
    *       @param[in]   listCount   componentFileList の配列の数。
    *       @param[in]   id   アップロードする SaveDataArchive のid。
    *       @param[in]   info   更新後のセーブデータの SeriesInfo。
    *       @param[in]   dataSize   更新後のセーブデータのサイズ。
    *       @param[in]   updatedTime   最後にデータが更新された日時。
    *       @param[in]   nsaIdToken   使用する NSA ID トークン。
    *       @param[in]   curlHandle   内部の通信処理に使用する libcURL のハンドル。
    *       @param[in]   workBuffer   ワークバッファ。
    *       @param[in]   workBufferSize   ワークバッファのサイズ。
    *       @param[in]   pCancelable   中止を検知するためのオブジェクト。

    *
    *       @return     処理結果を返します。
    *
    */
    Result StartSaveDataArchiveDifferentialUpload(SaveDataArchiveInfo* pOutSda, int* pOutCfCount, ComponentFileInfo componentFileList[], size_t listCount, SaveDataArchiveId id, const SeriesInfo& info, size_t dataSize, time::PosixTime updatedTime, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT;

    /**
    *       @brief      ストレージサーバに対して、 指定した id の SaveDataArchive の新規アップロード完了処理を要求します。
    *
    *       @param[in]   id   SaveDataArchive の ID。
    *       @param[in]   keySeed   SaveDataArchive の keySeed。
    *       @param[in]   mac   InitialData の mac。
    *       @param[in]   nsaIdToken   使用する NSA ID トークン。
    *       @param[in]   curlHandle   内部の通信処理に使用する libcURL のハンドル。
    *       @param[in]   workBuffer   ワークバッファ。
    *       @param[in]   workBufferSize   ワークバッファのサイズ。
    *       @param[in]   pCancelable   中止を検知するためのオブジェクト。
    *
    *       @return     処理結果を返します。
    *
    */
    Result FinishSaveDataArchiveUpload(SaveDataArchiveId id, const srv::KeySeed& keySeed, const srv::InitialDataMac& mac, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT;

    /**
    *       @brief      ストレージサーバに対して、 指定した id の SaveDataArchive の差分アップロード完了処理を要求します。
    *
    *       @param[in]   id   SaveDataArchive の ID。
    *       @param[in]   mac   InitialData の mac。
    *       @param[in]   nsaIdToken   使用する NSA ID トークン。
    *       @param[in]   curlHandle   内部の通信処理に使用する libcURL のハンドル。
    *       @param[in]   workBuffer   ワークバッファ。
    *       @param[in]   workBufferSize   ワークバッファのサイズ。
    *       @param[in]   pCancelable   中止を検知するためのオブジェクト。
    *
    *       @return     処理結果を返します。
    *
    */
    Result FinishSaveDataArchiveDifferentialUpload(SaveDataArchiveId id, const srv::InitialDataMac& mac, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT;


    /**
    *       @brief      ストレージサーバに対して、 指定した id の SaveDataArchive を削除を要求します。
    *
    *       @param[in]   id   SaveDataArchive の ID。
    *       @param[in]   nsaIdToken   使用する NSA ID トークン。
    *       @param[in]   curlHandle   内部の通信処理に使用する libcURL のハンドル。
    *       @param[in]   workBuffer   ワークバッファ。
    *       @param[in]   workBufferSize   ワークバッファのサイズ。
    *       @param[in]   pCancelable   中止を検知するためのオブジェクト。
    *
    *       @return     処理結果を返します。
    *
    */
    Result DeleteSaveDataArchive(SaveDataArchiveId id, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT;

    /**
    *       @brief      ストレージサーバに対して、 指定した id の SaveDataArchive のアップロード延長を要求します。
    *
    *       @param[out]  pOutCount   サーバーから返ってくる ComponentFileInfo の数。
    *       @param[in]   componentFileInfoList   サーバーから返ってくる ComponentFileInfo のリスト。
    *       @param[in]   listCount   componentFileInfoList の配列の数
    *       @param[in]   id   SaveDataArchive の ID。
    *       @param[in]   nsaIdToken   使用する NSA ID トークン。
    *       @param[in]   curlHandle   内部の通信処理に使用する libcURL のハンドル。
    *       @param[in]   workBuffer   ワークバッファ。
    *       @param[in]   workBufferSize   ワークバッファのサイズ。
    *       @param[in]   pCancelable   中止を検知するためのオブジェクト。
    *
    *       @return     処理結果を返します。
    *
    */
    Result ExtendSaveDataArchiveUploadTimeout(SaveDataArchiveInfo* pOutSda, int* pOutCfCount, ComponentFileInfo componentFileList[], size_t listCount, SaveDataArchiveId sdaId, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT;
}}}} // namespace nn::olsc::srv::transfer
