﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/account.h>
#include <nn/nn_Common.h>
#include <nn/olsc/srv/util/olsc_MountManager.h>
#include <nn/os/os_SdkConditionVariable.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/util/util_Optional.h>

#include <array>

namespace nn { namespace olsc { namespace srv {

template<typename ObjectType, int MaxObjectCount>
class UserReusableObjectManager
{
    NN_DISALLOW_COPY(UserReusableObjectManager);
public:
    class ObjectHolder
    {
        NN_DISALLOW_COPY(ObjectHolder);

    public:
        ObjectHolder() NN_NOEXCEPT;
        ~ObjectHolder() NN_NOEXCEPT;
        ObjectHolder(ObjectHolder&& rhs) NN_NOEXCEPT;
        ObjectHolder& operator=(ObjectHolder&& rhs) NN_NOEXCEPT;

        ObjectType* operator->() NN_NOEXCEPT;
        const ObjectType* operator->() const NN_NOEXCEPT;

        ObjectType& Get() NN_NOEXCEPT;
        const ObjectType& Get() const NN_NOEXCEPT;

    private:
        friend class UserReusableObjectManager<ObjectType, MaxObjectCount>;
        ObjectHolder(ObjectType* object, UserReusableObjectManager<ObjectType, MaxObjectCount>* manager) NN_NOEXCEPT;

        ObjectType* m_Object;
        UserReusableObjectManager<ObjectType, MaxObjectCount>* m_Manager;
    };

    UserReusableObjectManager() NN_NOEXCEPT
    {}

    ObjectHolder Acquire(const account::Uid& uid) NN_NOEXCEPT;
    bool TryAcquire(ObjectHolder* out, const account::Uid& uid) NN_NOEXCEPT;
    void Release(ObjectType* object) NN_NOEXCEPT;

protected:
    virtual void EmplaceObject(nn::util::optional<ObjectType>& toEmplace, const account::Uid& uid) NN_NOEXCEPT = 0;

private:

    struct Entry
    {
        nn::util::optional<ObjectType> object;
        account::Uid uid;
        int refCount;
    };

    std::array<Entry, MaxObjectCount> m_EntryHolder = {};
    os::SdkRecursiveMutex m_EntryHolderLock;
    os::SdkConditionVariable m_AcquirableCondition;

    void LogCurrentStatus() NN_NOEXCEPT;
};

}}} // namespace nn::olsc::srv


// -----------------------------------------------------------------------------
// 実装
// -----------------------------------------------------------------------------

#include <nn/olsc/detail/olsc_Log.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/result/result_HandlingUtility.h>

#if !defined(NN_SDK_BUILD_RELEASE)
#define NN_OLSC_USER_REUSABLE_OBJECT_MANAGER_LOG_ENABLE
#endif

namespace nn { namespace olsc { namespace srv {

template<typename ObjectType, int MaxObjectCount>
UserReusableObjectManager<ObjectType, MaxObjectCount>::ObjectHolder::ObjectHolder() NN_NOEXCEPT
    : m_Object(nullptr), m_Manager(nullptr)
{}

template<typename ObjectType, int MaxObjectCount>
UserReusableObjectManager<ObjectType, MaxObjectCount>::ObjectHolder::~ObjectHolder() NN_NOEXCEPT
{
    if (m_Manager)
    {
        m_Manager->Release(m_Object);
    }
}

template<typename ObjectType, int MaxObjectCount>
UserReusableObjectManager<ObjectType, MaxObjectCount>::ObjectHolder::ObjectHolder(ObjectHolder&& rhs) NN_NOEXCEPT
{
    this->m_Object = rhs.m_Object;
    this->m_Manager = rhs.m_Manager;

    rhs.m_Manager = nullptr;
    rhs.m_Object = nullptr;
}

template<typename ObjectType, int MaxObjectCount>
typename UserReusableObjectManager<ObjectType, MaxObjectCount>::ObjectHolder& UserReusableObjectManager<ObjectType, MaxObjectCount>::ObjectHolder::operator=(ObjectHolder&& rhs) NN_NOEXCEPT
{
    ObjectHolder t(std::move(rhs));

    std::swap(this->m_Object, t.m_Object);
    std::swap(this->m_Manager, t.m_Manager);

    return *this;
}

template<typename ObjectType, int MaxObjectCount>
UserReusableObjectManager<ObjectType, MaxObjectCount>::ObjectHolder::ObjectHolder(ObjectType* object, UserReusableObjectManager* manager) NN_NOEXCEPT
    : m_Object(object), m_Manager(manager)
{}

template<typename ObjectType, int MaxObjectCount>
ObjectType* UserReusableObjectManager<ObjectType, MaxObjectCount>::ObjectHolder::operator->() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Object);
    return m_Object;
}

template<typename ObjectType, int MaxObjectCount>
const ObjectType* UserReusableObjectManager<ObjectType, MaxObjectCount>::ObjectHolder::operator->() const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Object);
    return m_Object;
}

template<typename ObjectType, int MaxObjectCount>
ObjectType& UserReusableObjectManager<ObjectType, MaxObjectCount>::ObjectHolder::Get() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Object);
    return *m_Object;
}

template<typename ObjectType, int MaxObjectCount>
const ObjectType& UserReusableObjectManager<ObjectType, MaxObjectCount>::ObjectHolder::Get() const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Object);
    return *m_Object;
}

// -------------------------------------------------------------------------------------------------------

template<typename ObjectType, int MaxObjectCount>
typename UserReusableObjectManager<ObjectType, MaxObjectCount>::ObjectHolder UserReusableObjectManager<ObjectType, MaxObjectCount>::Acquire(const account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_ASSERT(uid != account::InvalidUid);
    std::lock_guard<decltype(m_EntryHolderLock)> lock(m_EntryHolderLock);
    ObjectHolder holder;
    while (!TryAcquire(&holder, uid))
    {
        m_AcquirableCondition.Wait(m_EntryHolderLock);
    }

    return holder;
}

template<typename ObjectType, int MaxObjectCount>
bool UserReusableObjectManager<ObjectType, MaxObjectCount>::TryAcquire(ObjectHolder* out, const account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_ASSERT(uid != account::InvalidUid);
    std::lock_guard<decltype(m_EntryHolderLock)> lock(m_EntryHolderLock);

    Entry* empty = nullptr;
    Entry* overwritable = nullptr;
    for (auto& entry : m_EntryHolder)
    {
        if (!entry.object)
        {
            empty = &entry;
            continue;
        }

        if (entry.uid == uid)
        {
            entry.refCount++;
            *out = std::move(ObjectHolder(&entry.object.value(), this));

            return true;
        }

        if (entry.refCount == 0)
        {
            overwritable = &entry;
        }
    }

    Entry* toEmplace = (empty) ? empty : overwritable;
    if (!toEmplace)
    {
        return false;
    }

    EmplaceObject(toEmplace->object, uid);
    toEmplace->refCount++;
    toEmplace->uid = uid;

    LogCurrentStatus();

    *out = std::move(ObjectHolder(&toEmplace->object.value(), this));

    return true;
}


template<typename ObjectType, int MaxObjectCount>
void UserReusableObjectManager<ObjectType, MaxObjectCount>::Release(ObjectType* object) NN_NOEXCEPT
{
    NN_SDK_ASSERT(object);

    std::lock_guard<decltype(m_EntryHolderLock)> lock(m_EntryHolderLock);
    for (auto& entry : m_EntryHolder)
    {
        if (entry.object && (&entry.object.value() == object))
        {
            entry.refCount--;
            NN_SDK_ASSERT(entry.refCount >= 0);
            if (entry.refCount == 0)
            {
                m_AcquirableCondition.Signal();
            }
            return;
        }
    }
    NN_ABORT("Must not come here.\n");
}


template<typename ObjectType, int MaxObjectCount>
void UserReusableObjectManager<ObjectType, MaxObjectCount>::LogCurrentStatus() NN_NOEXCEPT
{
#if defined(NN_OLSC_USER_REUSABLE_OBJECT_MANAGER_LOG_ENABLE)
    std::lock_guard<decltype(m_EntryHolderLock)> lock(m_EntryHolderLock);
    NN_DETAIL_OLSC_TRACE("CurrentStatus %016p\n", this);
    for (auto& entry : m_EntryHolder)
    {
        NN_DETAIL_OLSC_TRACE("  %016llx-%016llx: %d\n", entry.uid._data[0], entry.uid._data[1], entry.refCount);
    }
#endif
}

}}} //namespace nn::olsc::srv
