﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/account/account_Types.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/crypto/crypto_Md5Generator.h>
#include <nn/fs/fs_SaveDataManagementPrivate.h>
#include <nn/fs/fs_SaveDataTransferForCloudBackUp.h>
#include <nn/nn_ApplicationId.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/olsc/olsc_TransferTaskTypes.h>
#include <nn/time/time_PosixTime.h>

#include <array>

namespace nn { namespace olsc { namespace srv {

    typedef Bit64   SaveDataArchiveId;
    typedef Bit64   DeviceId;
    typedef Bit64   ComponentFileId;

    static const int MaxApplicationCount = 2048;
    static const int MaxUserCount = account::UserCountMax;
    static const int MaxTransferTaskCount = MaxApplicationCount * MaxUserCount;

    // TODO : fs に定義を移してもらう予定
    struct KeySeed
    {
        static const size_t Size = 16;
        char data[Size];
    };
    // TODO : fs に定義を移してもらう予定
    struct InitialDataMac
    {
        static const size_t Size = 16;
        char data[Size];
    };

    enum class SaveDataArchiveStatus : Bit8
    {
        Fixed,
        Uploading
    };

    enum class ComponentFileStatus : Bit8
    {
        Uploading,
        Fixed,
        HandOver
    };


    enum class ComponentFileType : Bit8
    {
        Meta,
        Save,
    };

    enum class PolicyType : Bit8
    {
        AllOk,
        AllNg
    };


    struct SeriesInfo
    {
        Bit64   seriesId;
        fs::SaveDataCommitId    commitId;

        NN_EXPLICIT_OPERATOR bool() const NN_NOEXCEPT;
    };

    const Bit64 InvalidSeriesInfoId = 0;
    const SeriesInfo InvalidSeriesInfo = { InvalidSeriesInfoId };

    inline SeriesInfo::operator bool() const NN_NOEXCEPT
    {
        return this->seriesId != InvalidSeriesInfoId;
    }

    struct NsaIdToken
    {
        char data[account::NetworkServiceAccountIdTokenLengthMax + 1]; // null 終端される
    };

    struct SaveDataArchiveInfo
    {
        static const int DigestSize = crypto::Sha256Generator::HashSize;
        SaveDataArchiveId                 id;
        account::NetworkServiceAccountId  nsaId;
        ApplicationId                     applicationId;
        account::Uid                      userId;
        DeviceId                          deviceId;
        size_t                            dataSize;
        SeriesInfo                        seriesInfo;
        SaveDataArchiveStatus             status;
        bool                              autoBackup;//UNUSED
        bool                              hasThumbnail;
        uint32_t                          launchRequiredVersion;
        uint32_t                          numOfPartitions;
        time::PosixTime                   savedAt;
        time::PosixTime                   timeoutAt;
        time::PosixTime                   finishedAt;
        time::PosixTime                   createdAt;
        time::PosixTime                   updatedAt;
    };

    struct ClientArgument
    {
        nn::fs::SaveDataChunkId chunkId;
    };

    struct ComponentFileDigest
    {
        static const size_t    Size = crypto::Sha256Generator::HashSize;
        Bit8                     data[Size];
    };

    struct SaveDataChunkDigest
    {
        static const size_t  Size = crypto::Md5Generator::HashSize;
        Bit8                     data[Size];
    };

    struct ComponentFileInfo
    {
        // TODO : サーバー側の Url 最大長短縮の処理実装後 512 - 1 に戻す
        static const size_t    MaxUrlLength = 2048 - 1;

        ComponentFileId                   id;   // storage_file の ID
        SaveDataArchiveId                 sdaId;// 対応する SDA の ID
        ClientArgument                    clientArgument;
        ComponentFileType                 type;
        ComponentFileStatus               status;
        size_t                            componentFileSize; // データサイズ (byte)
        ComponentFileDigest               componentFileDigest;
        size_t                            saveDataChunkSize; // データサイズ (byte)
        SaveDataChunkDigest               saveDataChunkDigest;
        char                              url[MaxUrlLength + 1];
        time::PosixTime                   createdAt;
        time::PosixTime                   updatedAt;
    };

    struct TransferTaskConfig
    {
        TransferTaskKind    kind;
        SeriesInfo          cachedSi;
        union {
            struct {
                bool saveDataExists;
                fs::SaveDataCommitId currentCommitId;   // saveDataExists が true の場合のみ有効な値
            } dlInfo;
            struct {
                bool force;
            } ulInfo;
        };
    };

    struct TransferTaskDetailInfo
    {
        enum class State : Bit8
        {
            NotFinished,
            Success,
            Error,
        };

        TransferTaskId          id;
        account::Uid            uid;
        ApplicationId           appId;
        TransferTaskConfig      config;
        State                   state;
        TransferTaskRank        rank;

        inline static TransferTaskDetailInfo Make(TransferTaskId id, const account::Uid& uid, ApplicationId appId, const TransferTaskConfig& config, State state, TransferTaskRank rank) NN_NOEXCEPT
        {
            return TransferTaskDetailInfo{
                id, uid, appId, config, state, rank
            };
        }
    };

    struct PolicyInfo
    {
        PolicyType type;
    };

    enum class SeriesPosition : Bit8 {
        Equal,                      //!<    ローカルとサーバが一致
        ServerDataIsNewer = 1,      //!<    サーバ側が新しい
        LocalDataIsNewer = 2,       //!<    ローカル側が新しい
        Discrete = 3,               //!<    系列が不一致
    };

    struct TransferTaskSuspendedInfo
    {
        TransferTaskId      id;
        SaveDataArchiveId   sdaId;
        bool isDifferentialUpload;
        std::unique_ptr<fs::SaveDataTransferManagerForCloudBackUp> manager;
        std::unique_ptr<fs::ISaveDataDivisionExporter> exporter;
        bool IsValid() NN_NOEXCEPT
        {
            return manager && exporter;
        }
    };

    // ------------------------------------------------------------------------------------------

    inline bool operator==(const SaveDataArchiveInfo& lhs, const SaveDataArchiveInfo& rhs) NN_NOEXCEPT
    {
        return lhs.id == rhs.id;
    }

    // inline bool operator!=(const SaveDataArchive& lhs, const SaveDataArchive& rhs) NN_NOEXCEPT
    // {
    //     return !(lhs == rhs);
    // }

    inline bool operator==(const ClientArgument& lhs, const ClientArgument& rhs) NN_NOEXCEPT
    {
        return lhs.chunkId == rhs.chunkId;
    }


    inline bool operator==(const ComponentFileInfo& lhs, const ComponentFileInfo& rhs) NN_NOEXCEPT
    {
        return lhs.id == rhs.id;
    }

    inline bool operator!=(const ComponentFileInfo& lhs, const ComponentFileInfo& rhs) NN_NOEXCEPT
    {
        return !(lhs.id == rhs.id);
    }

    inline bool operator==(const SeriesInfo& lhs, const SeriesInfo& rhs) NN_NOEXCEPT
    {
        return lhs.seriesId == rhs.seriesId
            && lhs.commitId == rhs.commitId;
    }

    inline bool operator!=(const SeriesInfo& lhs, const SeriesInfo& rhs) NN_NOEXCEPT
    {
        return !(lhs == rhs);
    }

    inline bool operator==(const TransferTaskConfig& lhs, const TransferTaskConfig& rhs) NN_NOEXCEPT
    {
        auto common = lhs.kind == rhs.kind
            && lhs.cachedSi == rhs.cachedSi;

        switch (lhs.kind)
        {
        case TransferTaskKind::Upload:
            return common &&
                lhs.ulInfo.force == rhs.ulInfo.force;
        case TransferTaskKind::Download:
            return common &&
                lhs.dlInfo.currentCommitId == rhs.dlInfo.currentCommitId;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    inline bool operator==(const TransferTaskDetailInfo& lhs, const TransferTaskDetailInfo& rhs) NN_NOEXCEPT
    {
        return lhs.id == rhs.id
            && lhs.uid == rhs.uid
            && lhs.appId == rhs.appId
            && lhs.config == rhs.config
            && lhs.state == rhs.state
            && lhs.rank == rhs.rank;
    }

    inline bool operator!=(const TransferTaskDetailInfo& lhs, const TransferTaskDetailInfo& rhs) NN_NOEXCEPT
    {
        return !(lhs == rhs);
    }

    inline bool operator==(const PolicyInfo& lhs, const PolicyInfo& rhs) NN_NOEXCEPT
    {
        return lhs.type == rhs.type;
    }
    inline bool operator!=(const PolicyInfo& lhs, const PolicyInfo& rhs) NN_NOEXCEPT
    {
        return !(lhs == rhs);
    }

}}}
