﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <array>
#include <functional>
#include <nn/nn_Result.h>
#include <nn/nn_StaticAssert.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/olsc_ResultPrivate.h>
#include <nn/olsc/srv/util/olsc_MountContext.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_BitArray.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace olsc { namespace srv { namespace database {

/**

    @brief      インデックスのみをキーとする単純なデータストア

*/
template<typename MountManagerType, typename EntryType, int MaxEntryCount>
class Datastore
{
    NN_DISALLOW_COPY(Datastore);
    NN_DISALLOW_MOVE(Datastore);
public:

    NN_IMPLICIT Datastore(MountManagerType& mounter) NN_NOEXCEPT :
        m_IsOpened(false), m_MountManager(mounter)
    {}

    bool Put(int* outIndex, const EntryType& entry) NN_NOEXCEPT;
    bool Get(EntryType* out, int index) const NN_NOEXCEPT;
    void Remove(int index) NN_NOEXCEPT;
    bool Replace(int index, const EntryType& entry) NN_NOEXCEPT;

    int GetCount() const NN_NOEXCEPT;
    void Cleanup() NN_NOEXCEPT;

    // std::lock_guard 用
    void lock() NN_NOEXCEPT;
    void unlock() NN_NOEXCEPT;
    void lock() const NN_NOEXCEPT;
    void unlock() const NN_NOEXCEPT;

protected:
    Result InitializeUnsafe() NN_NOEXCEPT;
    virtual const char* GetMetadataFileRelativePath() const NN_NOEXCEPT = 0;
    virtual const char* GetEntryFileRelativePath() const NN_NOEXCEPT = 0;
    virtual util::ReadMount AcquireReadMount(MountManagerType& mountManager) const NN_NOEXCEPT = 0;
    virtual util::WriteMount AcquireWriteMount(MountManagerType& mountManager) const NN_NOEXCEPT = 0;

private:
    struct Metadata
    {
        Bit64 occupyListBuffer[(MaxEntryCount + 63) / 64]; // MaxEntry ビット必要
    };
    static const int MaxPathLen = 64;
    static const int EntryFileSize = MaxEntryCount * sizeof(EntryType);


    Metadata m_Metadata;
    bool m_IsOpened;
    mutable os::SdkRecursiveMutex m_Lock;
    MountManagerType& m_MountManager;
    nn::util::BitArray m_OccupyList;

    const char* MakeMetadataFilePath(char* out, size_t maxOutSize, const char* rootPath) const NN_NOEXCEPT;
    const char* MakeEntryFilePath(char* out, size_t maxOutSize, const char* rootPath) const NN_NOEXCEPT;

    Result ReadMetadataFile() NN_NOEXCEPT;
    Result WriteMetadataFile() const NN_NOEXCEPT;
    Result ReadEntry(EntryType* out, int index) const NN_NOEXCEPT;
    Result WriteEntry(const EntryType& entry, int index) const NN_NOEXCEPT;
    bool HasSpace() const NN_NOEXCEPT;
};

}}}} // namespace nn::olsc::srv::database

// ---------------------------------------------------------------------------------
// 実装
// ---------------------------------------------------------------------------------

#include <mutex>
#include <nn/fs.h>
#include <nn/olsc/detail/olsc_Log.h>
#include <nn/olsc/srv/util/olsc_File.h>
#include <nn/util/util_TFormatString.h>

#define NN_OLSC_DEFINE_SIMPLE_DATASTORE_METHOD(rtype, methodInfo) \
template <typename MounterType, typename EntryType, int MaxEntryCount> \
inline rtype Datastore<MounterType, EntryType, MaxEntryCount>::methodInfo NN_NOEXCEPT

namespace nn { namespace olsc { namespace srv { namespace database {

// ---------------------------------------------------------------------------------
// Private
// ---------------------------------------------------------------------------------

NN_OLSC_DEFINE_SIMPLE_DATASTORE_METHOD(Result, ReadMetadataFile())
{
    auto readMount = AcquireReadMount(m_MountManager);

    char path[MaxPathLen];
    MakeMetadataFilePath(path, sizeof(path), readMount.GetRootPath());

    NN_RESULT_TRY(util::ReadFile(path, &m_Metadata, sizeof(m_Metadata), 0))
        NN_RESULT_CATCH(fs::ResultPathNotFound)
        {
            m_Metadata = {};
        }
    NN_RESULT_END_TRY;
    NN_RESULT_SUCCESS;
}

NN_OLSC_DEFINE_SIMPLE_DATASTORE_METHOD(Result, WriteMetadataFile() const)
{
    auto writeMount = AcquireWriteMount(m_MountManager);

    char path[MaxPathLen];
    NN_RESULT_DO(util::WriteFile(MakeMetadataFilePath(path, sizeof(path), writeMount.GetRootPath()), &m_Metadata, sizeof(m_Metadata)));
    NN_RESULT_SUCCESS;
}

NN_OLSC_DEFINE_SIMPLE_DATASTORE_METHOD(Result, ReadEntry(EntryType* out, int index) const)
{
    NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());
    NN_SDK_ASSERT(m_IsOpened);

    auto readMount = AcquireReadMount(m_MountManager);

    char path[MaxPathLen];
    NN_RESULT_DO(util::ReadFile(MakeEntryFilePath(path, sizeof(path), readMount.GetRootPath()), out, sizeof(EntryType), index * sizeof(EntryType)));
    NN_RESULT_SUCCESS;
}

NN_OLSC_DEFINE_SIMPLE_DATASTORE_METHOD(Result, WriteEntry(const EntryType& entry, int index) const)
{
    NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());
    NN_SDK_ASSERT(m_IsOpened);

    auto writeMount = AcquireWriteMount(m_MountManager);

    char path[MaxPathLen];
    MakeEntryFilePath(path, sizeof(path), writeMount.GetRootPath());

    auto modifyFile = [&entry, index](const char* path) -> Result {
        return util::ModifyFile(path, &entry, sizeof(entry), index * sizeof(entry));
    };

    NN_RESULT_TRY(modifyFile(path))
        NN_RESULT_CATCH(fs::ResultPathNotFound)
        {
            NN_RESULT_DO(util::CreateFile(path, sizeof(EntryType) * MaxEntryCount));
            NN_RESULT_DO(modifyFile(path));
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

NN_OLSC_DEFINE_SIMPLE_DATASTORE_METHOD(const char*, MakeMetadataFilePath(char* out, size_t maxOutSize, const char* rootPath) const)
{
    NN_ABORT_UNLESS(static_cast<size_t>(nn::util::TSNPrintf(out, maxOutSize, "%s%s", rootPath, GetMetadataFileRelativePath())) < maxOutSize);
    return out;
}

NN_OLSC_DEFINE_SIMPLE_DATASTORE_METHOD(const char*, MakeEntryFilePath(char* out, size_t maxOutSize, const char* rootPath) const)
{
    NN_ABORT_UNLESS(static_cast<size_t>(nn::util::TSNPrintf(out, maxOutSize, "%s%s", rootPath, GetEntryFileRelativePath())) < maxOutSize);
    return out;
}

// ---------------------------------------------------------------------------------
// Protected
// ---------------------------------------------------------------------------------

NN_OLSC_DEFINE_SIMPLE_DATASTORE_METHOD(Result, InitializeUnsafe())
{
    NN_SDK_ASSERT(!m_IsOpened);

    NN_RESULT_DO(ReadMetadataFile());

    m_OccupyList.ResetWorkMemory(m_Metadata.occupyListBuffer, sizeof(m_Metadata.occupyListBuffer), MaxEntryCount);
    m_IsOpened = true;

    NN_RESULT_SUCCESS;
}

// ---------------------------------------------------------------------------------
// Public
// ---------------------------------------------------------------------------------

NN_OLSC_DEFINE_SIMPLE_DATASTORE_METHOD(void, Remove(int index))
{
    NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());
    NN_SDK_ASSERT(m_IsOpened);

    m_OccupyList[index] = false;
    NN_ABORT_UNLESS_RESULT_SUCCESS(WriteMetadataFile());
}

NN_OLSC_DEFINE_SIMPLE_DATASTORE_METHOD(bool, Put(int* outIndex, const EntryType& entry))
{
    NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());
    NN_SDK_ASSERT(m_IsOpened);

    if (!HasSpace())
    {
        return false;
    }

    int index = 0;
    while (m_OccupyList.test(index))
    {
        index++;
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(WriteEntry(entry, index));

    m_OccupyList[index] = true;
    NN_ABORT_UNLESS_RESULT_SUCCESS(WriteMetadataFile());

    *outIndex = index;
    return true;
}

NN_OLSC_DEFINE_SIMPLE_DATASTORE_METHOD(bool, Replace(int index, const EntryType& entry))
{
    NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());
    NN_SDK_ASSERT(m_IsOpened);

    if (!m_OccupyList[index])
    {
        return false;
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(WriteEntry(entry, index));

    return true;
}

NN_OLSC_DEFINE_SIMPLE_DATASTORE_METHOD(bool, Get(EntryType* out, int index) const)
{
    NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());
    NN_SDK_ASSERT(m_IsOpened);
    NN_ABORT_UNLESS(index >= 0 && index < MaxEntryCount);

    if (!m_OccupyList[index])
    {
        return false;
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(ReadEntry(out, index));
    return true;
}

NN_OLSC_DEFINE_SIMPLE_DATASTORE_METHOD(int, GetCount() const)
{
    NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());
    NN_SDK_ASSERT(m_IsOpened);

    return m_OccupyList.count();
}

NN_OLSC_DEFINE_SIMPLE_DATASTORE_METHOD(bool, HasSpace() const)
{
    NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());
    NN_SDK_ASSERT(m_IsOpened);

    return !m_OccupyList.all();
}

NN_OLSC_DEFINE_SIMPLE_DATASTORE_METHOD(void, Cleanup())
{
    NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());
    NN_SDK_ASSERT(m_IsOpened);

    auto writeMount = AcquireWriteMount(m_MountManager);

    char path[MaxPathLen];

    NN_ABORT_UNLESS_RESULT_SUCCESS(util::DeleteFile(MakeEntryFilePath(path, sizeof(path), writeMount.GetRootPath())));
    NN_ABORT_UNLESS_RESULT_SUCCESS(util::DeleteFile(MakeMetadataFilePath(path, sizeof(path), writeMount.GetRootPath())));

    m_OccupyList.reset();
}

NN_OLSC_DEFINE_SIMPLE_DATASTORE_METHOD(void, lock())
{
    m_Lock.Lock();
}

NN_OLSC_DEFINE_SIMPLE_DATASTORE_METHOD(void, unlock())
{
    m_Lock.Unlock();
}

NN_OLSC_DEFINE_SIMPLE_DATASTORE_METHOD(void, lock() const)
{
    m_Lock.Lock();
}

NN_OLSC_DEFINE_SIMPLE_DATASTORE_METHOD(void, unlock() const)
{
    m_Lock.Unlock();
}

}}}} // namespace nn::olsc::srv::database

#undef NN_OLSC_DEFINE_SIMPLE_DATASTORE_METHOD
