﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <array>
#include <functional>
#include <nn/nn_Result.h>
#include <nn/nn_StaticAssert.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/olsc_ResultPrivate.h>
#include <nn/olsc/srv/database/olsc_CachedReaderWriter.h>
#include <nn/olsc/srv/util/olsc_MountContext.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace olsc { namespace srv { namespace database {

/**

    @brief      線形にエントリを永続化するDB、先頭部分のみメモリ上にキャッシュする

    @detail     派生クラスのコンストラクタなどで InitializeUnsafe を呼び出すようにしてください。
*/
template<typename MountManagerType, typename EntryType, int MaxEntryCount, int CacheCount, int ReadBufferCount>
class DataArray
{
    NN_DISALLOW_COPY(DataArray);
    NN_DISALLOW_MOVE(DataArray);
public:
    NN_STATIC_ASSERT(ReadBufferCount > 0);

    using CacheBuffer = std::array<EntryType, CacheCount>;
    using ReadBuffer = std::array<EntryType, ReadBufferCount>;

    DataArray(CacheBuffer* cacheBuffer, ReadBuffer* readBuffer, MountManagerType& mounter) NN_NOEXCEPT :
        m_Metadata({}), m_IsOpened(false), m_Cache(cacheBuffer->data(), static_cast<int>(cacheBuffer->size())), m_MountManager(mounter),
        m_ReaderWriter(readBuffer->data(), static_cast<int>(readBuffer->size()), MaxEntryCount)
    {}

    Result PushFront(const EntryType& entry) NN_NOEXCEPT;
    Result PushBack(const EntryType& entry) NN_NOEXCEPT;
    Result Remove(const EntryType& entry) NN_NOEXCEPT;

    using Predicate = std::function<bool(const EntryType&)>;

    Result RemoveIf(const Predicate& pred) NN_NOEXCEPT;
    Result RemoveAll(const Predicate& pred) NN_NOEXCEPT;
    nn::util::optional<EntryType> FindIf(const Predicate& pred) const NN_NOEXCEPT;
    Result Replace(int index, const EntryType& entry) NN_NOEXCEPT;
    Result ReplaceIf(const Predicate& pred, const EntryType& entry) NN_NOEXCEPT;
    void ForEach(const Predicate& pred) const NN_NOEXCEPT;

    int GetCount() const NN_NOEXCEPT;

    Result At(EntryType* out, int index) const NN_NOEXCEPT;
    EntryType operator[](int index) const NN_NOEXCEPT;

    int List(EntryType out[], int maxOutCount, int offset) const NN_NOEXCEPT;

    bool HasEntry(const EntryType& entry) const NN_NOEXCEPT;
    bool HasSpace() const NN_NOEXCEPT;

    Result Cleanup() NN_NOEXCEPT;

    // std::lock_guard 用
    void lock() NN_NOEXCEPT;
    void unlock() NN_NOEXCEPT;
    void lock() const NN_NOEXCEPT;
    void unlock() const NN_NOEXCEPT;

protected:
    virtual const char* GetMetadataFileRelativePath() const NN_NOEXCEPT = 0;
    virtual const char* GetEntryFileRelativePath() const NN_NOEXCEPT = 0;
    virtual util::ReadMount AcquireReadMount(MountManagerType& mountManager) const NN_NOEXCEPT = 0;
    virtual util::WriteMount AcquireWriteMount(MountManagerType& mountManager) NN_NOEXCEPT = 0;

    Result InitializeUnsafe() NN_NOEXCEPT;
private:
    struct Metadata
    {
        int entryCount;
    };
    static const int MaxPathLen = 64;
    static const int IndexFileSize = MaxEntryCount * sizeof(EntryType);

    class Cache
    {
    public:
        NN_IMPLICIT Cache(EntryType cacheBuffer[], int cacheBufferCount) NN_NOEXCEPT : m_CacheBuffer(cacheBuffer), m_CachedCount(0), m_CacheBufferCount(cacheBufferCount)
        {}
        using Filler = std::function<Result(EntryType dest[], int count)>;
        Result Fill(Filler filler, int count) NN_NOEXCEPT;
        const EntryType& At(int index) NN_NOEXCEPT;

        using Reader = std::function<Result(const EntryType data[], int count)>;
        Result Read(Reader reader, int offset, int count) NN_NOEXCEPT;

        void PushBack(const EntryType& entry) NN_NOEXCEPT;
        void PushFront(const EntryType& entry) NN_NOEXCEPT;
        void Update(int index, const EntryType& entry) NN_NOEXCEPT;
        int GetCachedCount() const NN_NOEXCEPT;
        void Clear() NN_NOEXCEPT;

    private:
        EntryType* m_CacheBuffer;
        int m_CachedCount;
        int m_CacheBufferCount;
    };

    Metadata m_Metadata;
    bool m_IsOpened;
    mutable os::SdkRecursiveMutex m_Lock;
    mutable Cache m_Cache;
    MountManagerType& m_MountManager;
    mutable CachedReaderWriter<EntryType> m_ReaderWriter;

    Result ReadMetadataFile() NN_NOEXCEPT;
    Result WriteMetadataFile() NN_NOEXCEPT;
    Result FillCache(Cache& cache, int offsetIndex, int entryCount) const NN_NOEXCEPT;
    Result UpdateEntryFile(const EntryType entriss[], int entryCount, int offsetIndex, const char* path) NN_NOEXCEPT;
    nn::util::optional<int> FindIndex(const EntryType& entry) const NN_NOEXCEPT;
    nn::util::optional<int> FindIndex(const Predicate& pred) const NN_NOEXCEPT;
    void ClearMetaAndCache() NN_NOEXCEPT;
    const char* MakeMetadataFilePath(char* out, size_t maxOutSize, const char* rootPath) const NN_NOEXCEPT;
    const char* MakeEntryFilePath(char* out, size_t maxOutSize, const char* rootPath) const NN_NOEXCEPT;
};

}}}} // namespace nn::olsc::srv::database

// ---------------------------------------------------------------------------------
// 実装
// ---------------------------------------------------------------------------------

#include <algorithm>
#include <mutex>
#include <nn/fs.h>
#include <nn/olsc/detail/olsc_Log.h>
#include <nn/olsc/srv/util/olsc_File.h>
#include <nn/util/util_TFormatString.h>

#define NN_OLSC_DEFINE_DATA_ARRAY_METHOD(rtype, methodInfo) \
template <typename MounterType, typename EntryType, int MaxEntryCount, int CacheCount, int ReadBufferCount> \
inline rtype DataArray<MounterType, EntryType, MaxEntryCount, CacheCount, ReadBufferCount>::methodInfo NN_NOEXCEPT

namespace nn { namespace olsc { namespace srv { namespace database {

// ---------------------------------------------------------------------------------
// Private
// ---------------------------------------------------------------------------------

NN_OLSC_DEFINE_DATA_ARRAY_METHOD(Result, ReadMetadataFile())
{
    auto readMount = AcquireReadMount(m_MountManager);

    char path[MaxPathLen];
    NN_RESULT_TRY(util::ReadFile(MakeMetadataFilePath(path, sizeof(path), readMount.GetRootPath()), &m_Metadata, sizeof(m_Metadata), 0))
        NN_RESULT_CATCH(fs::ResultPathNotFound)
        {
            m_Metadata = {};
        }
    NN_RESULT_END_TRY;
    NN_RESULT_SUCCESS;
}

NN_OLSC_DEFINE_DATA_ARRAY_METHOD(Result, WriteMetadataFile())
{
    auto writeMount = AcquireWriteMount(m_MountManager);

    char path[MaxPathLen];
    NN_RESULT_DO(util::WriteFile(MakeMetadataFilePath(path, sizeof(path), writeMount.GetRootPath()), &m_Metadata, sizeof(m_Metadata)));
    NN_RESULT_SUCCESS;
}

NN_OLSC_DEFINE_DATA_ARRAY_METHOD(Result, FillCache(Cache& cache, int offsetIndex, int count) const)
{
    auto fillCount = std::min(m_Metadata.entryCount - offsetIndex, count);
    auto filler = [this](EntryType out[], int count) -> Result
    {
        auto readMount = AcquireReadMount(m_MountManager);
        char path[MaxPathLen];
        if (count > 0)
        {
            // m_ReaderWriter.Read を呼ぶと、リードキャッシュが破棄されて
            // 必ずファイルの先頭がリードキャッシュに載ってしまうので、
            // ここではリードキャッシュ util::ReadFile を直接呼びリードキャッシュをそのままに保つ。
            NN_RESULT_DO(util::ReadFile(MakeEntryFilePath(path, MaxPathLen, readMount.GetRootPath()), out, count * sizeof(EntryType), 0));
        }
        NN_RESULT_SUCCESS;
    };

    NN_RESULT_DO(cache.Fill(filler, fillCount));
    NN_RESULT_SUCCESS;
}

NN_OLSC_DEFINE_DATA_ARRAY_METHOD(Result, UpdateEntryFile(const EntryType entries[], int entryCount, int offsetIndex, const char* path))
{
    NN_RESULT_DO(m_ReaderWriter.Write(path, entries, offsetIndex, entryCount));
    NN_RESULT_SUCCESS;
}

NN_OLSC_DEFINE_DATA_ARRAY_METHOD(nn::util::optional<int>, FindIndex(const EntryType& key) const)
{
    return FindIndex([&key](const EntryType& entry) {
        return entry == key;
    });
}

NN_OLSC_DEFINE_DATA_ARRAY_METHOD(nn::util::optional<int>, FindIndex(const Predicate& pred) const)
{
    for (int i = 0; i < m_Metadata.entryCount && i < CacheCount; ++i)
    {
        if (pred(m_Cache.At(i)))
        {
            return i;
        }
    }

    auto readMount = AcquireReadMount(m_MountManager);
    char path[MaxPathLen];
    MakeEntryFilePath(path, MaxPathLen, readMount.GetRootPath());

    for (int i = CacheCount; i < m_Metadata.entryCount; ++i)
    {
        EntryType entry;
        int readCount;
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_ReaderWriter.Read(&readCount, &entry, path, i, 1));
        NN_ABORT_UNLESS(readCount == 1);
        if (pred(entry))
        {
            return i;
        }
    }

    return nn::util::nullopt;
}

NN_OLSC_DEFINE_DATA_ARRAY_METHOD(void, ClearMetaAndCache())
{
    m_Metadata = {};
    m_Cache.Clear();
    m_ReaderWriter.Clear();
}

NN_OLSC_DEFINE_DATA_ARRAY_METHOD(Result, Cache::Fill(Filler filler, int count))
{
    NN_SDK_ASSERT(count >= 0 && count <= m_CacheBufferCount);
    NN_RESULT_DO(filler(m_CacheBuffer, count));
    m_CachedCount = count;
    NN_RESULT_SUCCESS;
}

NN_OLSC_DEFINE_DATA_ARRAY_METHOD(const EntryType&, Cache::At(int index))
{
    NN_ABORT_UNLESS(index >= 0 && index < m_CachedCount);
    return m_CacheBuffer[index];
}

NN_OLSC_DEFINE_DATA_ARRAY_METHOD(Result, Cache::Read(Reader reader, int offset, int count))
{
    NN_ABORT_UNLESS(offset >= 0 && offset < m_CachedCount);
    NN_ABORT_UNLESS(count >= 0 && count <= m_CachedCount);
    NN_ABORT_UNLESS(offset + count <= m_CachedCount);

    NN_RESULT_DO(reader(&m_CacheBuffer[offset], count));
    NN_RESULT_SUCCESS;
}

NN_OLSC_DEFINE_DATA_ARRAY_METHOD(void, Cache::PushBack(const EntryType& entry))
{
    if (m_CacheBufferCount > 0)
    {
        NN_ABORT_UNLESS(m_CachedCount < m_CacheBufferCount);
        m_CacheBuffer[m_CachedCount++] = entry;
    }
}

NN_OLSC_DEFINE_DATA_ARRAY_METHOD(void, Cache::PushFront(const EntryType& entry))
{
    if (m_CacheBufferCount > 0)
    {
        std::memmove(&m_CacheBuffer[1], &m_CacheBuffer[0], sizeof(EntryType) * (m_CacheBufferCount - 1));
        m_CacheBuffer[0] = entry;
        if (m_CachedCount < m_CacheBufferCount)
        {
            m_CachedCount++;
        }
    }
}

NN_OLSC_DEFINE_DATA_ARRAY_METHOD(void, Cache::Update(int index, const EntryType& entry))
{
    if (m_CacheBufferCount > 0)
    {
        NN_ABORT_UNLESS(index >= 0 && index < m_CacheBufferCount);
        m_CacheBuffer[index] = entry;
    }
}

NN_OLSC_DEFINE_DATA_ARRAY_METHOD(int, Cache::GetCachedCount() const)
{
    return m_CachedCount;
}

NN_OLSC_DEFINE_DATA_ARRAY_METHOD(void, Cache::Clear())
{
    m_CachedCount = 0;
}

NN_OLSC_DEFINE_DATA_ARRAY_METHOD(const char*, MakeMetadataFilePath(char* out, size_t maxOutSize, const char* rootPath) const)
{
    NN_ABORT_UNLESS(static_cast<size_t>(nn::util::TSNPrintf(out, maxOutSize, "%s%s", rootPath, GetMetadataFileRelativePath())) < maxOutSize);
    return out;
}

NN_OLSC_DEFINE_DATA_ARRAY_METHOD(const char*, MakeEntryFilePath(char* out, size_t maxOutSize, const char* rootPath) const)
{
    NN_ABORT_UNLESS(static_cast<size_t>(nn::util::TSNPrintf(out, maxOutSize, "%s%s", rootPath, GetEntryFileRelativePath())) < maxOutSize);
    return out;
}


// ---------------------------------------------------------------------------------
// Protected
// ---------------------------------------------------------------------------------

NN_OLSC_DEFINE_DATA_ARRAY_METHOD(Result, InitializeUnsafe())
{
    NN_SDK_ASSERT(!m_IsOpened);

    NN_RESULT_DO(ReadMetadataFile());
    NN_RESULT_DO(FillCache(m_Cache, 0, CacheCount));

    m_IsOpened = true;

    NN_RESULT_SUCCESS;
}

// ---------------------------------------------------------------------------------
// Public
// ---------------------------------------------------------------------------------

NN_OLSC_DEFINE_DATA_ARRAY_METHOD(Result, PushFront(const EntryType& entry))
{
    NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());
    NN_SDK_ASSERT(m_IsOpened);
    NN_ABORT_UNLESS(m_Metadata.entryCount < MaxEntryCount);
    NN_ABORT_UNLESS(!HasEntry(entry));

    auto writeMount = AcquireWriteMount(m_MountManager);
    char path[MaxPathLen];
    MakeEntryFilePath(path, MaxPathLen, writeMount.GetRootPath());

    // 全体を一つ後ろにずらす
    NN_RESULT_DO(m_ReaderWriter.Move(path, 1, 0, m_Metadata.entryCount));

    // 先頭の空きに追加
    NN_RESULT_DO(UpdateEntryFile(&entry, 1, 0, path));

    m_Cache.PushFront(entry);

    m_Metadata.entryCount++;
    NN_RESULT_DO(WriteMetadataFile());

    NN_RESULT_SUCCESS;
}

NN_OLSC_DEFINE_DATA_ARRAY_METHOD(Result, PushBack(const EntryType& entry))
{
    NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());
    NN_SDK_ASSERT(m_IsOpened);
    NN_ABORT_UNLESS(m_Metadata.entryCount < MaxEntryCount);
    NN_ABORT_UNLESS(!HasEntry(entry));

    auto newIndex = m_Metadata.entryCount;

    auto writeMount = AcquireWriteMount(m_MountManager);
    char path[MaxPathLen];

    NN_RESULT_DO(UpdateEntryFile(&entry, 1, newIndex, MakeEntryFilePath(path, MaxPathLen, writeMount.GetRootPath())));

    if (newIndex < CacheCount)
    {
        m_Cache.PushBack(entry);
    }

    m_Metadata.entryCount++;
    NN_RESULT_DO(WriteMetadataFile());

    NN_RESULT_SUCCESS;
}

NN_OLSC_DEFINE_DATA_ARRAY_METHOD(Result, Remove(const EntryType& entry))
{
    NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());
    NN_SDK_ASSERT(m_IsOpened);

    auto index = FindIndex(entry);
    NN_RESULT_THROW_UNLESS(index, olsc::ResultDatabaseEntryNotFound());

    auto writeMount = AcquireWriteMount(m_MountManager);
    char path[MaxPathLen];
    MakeEntryFilePath(path, MaxPathLen, writeMount.GetRootPath());

    if (*index < m_Metadata.entryCount - 1)
    {
        auto fromIndex = *index + 1;
        NN_RESULT_DO(m_ReaderWriter.Move(path, *index, fromIndex, m_Metadata.entryCount - fromIndex));
    }

    m_Metadata.entryCount--;
    NN_RESULT_DO(WriteMetadataFile());

    if (*index < CacheCount)
    {
        NN_RESULT_DO(FillCache(m_Cache, 0, CacheCount));
    }

    NN_RESULT_SUCCESS;
}

NN_OLSC_DEFINE_DATA_ARRAY_METHOD(int, GetCount() const)
{
    NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());
    NN_SDK_ASSERT(m_IsOpened);

    return m_Metadata.entryCount;
}

NN_OLSC_DEFINE_DATA_ARRAY_METHOD(Result, At(EntryType* out, int index) const)
{
    NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());
    NN_SDK_ASSERT(m_IsOpened);
    NN_ABORT_UNLESS(index < m_Metadata.entryCount);

    if (index < CacheCount)
    {
        *out = m_Cache.At(index);
        NN_RESULT_SUCCESS;
    }

    auto readMount = AcquireReadMount(m_MountManager);
    char path[MaxPathLen];

    int readCount;
    NN_RESULT_DO(m_ReaderWriter.Read(&readCount, out, MakeEntryFilePath(path, MaxPathLen, readMount.GetRootPath()), index, 1));
    NN_ABORT_UNLESS(readCount == 1);
    NN_RESULT_SUCCESS;
}

NN_OLSC_DEFINE_DATA_ARRAY_METHOD(bool, HasEntry(const EntryType& key) const)
{
    NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());
    NN_SDK_ASSERT(m_IsOpened);

    auto index = FindIndex(key);
    return index;
}

NN_OLSC_DEFINE_DATA_ARRAY_METHOD(bool, HasSpace() const)
{
    NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());
    NN_SDK_ASSERT(m_IsOpened);

    return m_Metadata.entryCount < MaxEntryCount;
}

NN_OLSC_DEFINE_DATA_ARRAY_METHOD(Result, Cleanup())
{
    NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());
    NN_SDK_ASSERT(m_IsOpened);

    auto writeMount = AcquireWriteMount(m_MountManager);

    char path[MaxPathLen];

    NN_RESULT_DO(util::DeleteFile(MakeEntryFilePath(path, sizeof(path), writeMount.GetRootPath())));
    NN_RESULT_DO(util::DeleteFile(MakeMetadataFilePath(path, sizeof(path), writeMount.GetRootPath())));
    ClearMetaAndCache();

    NN_RESULT_SUCCESS;
}

NN_OLSC_DEFINE_DATA_ARRAY_METHOD(int, List(EntryType out[], int maxOutCount, int offset) const)
{
    NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());
    NN_SDK_ASSERT(m_IsOpened);
    NN_SDK_ASSERT(offset >= 0);

    int listed = 0;
    // キャッシュからコピー
    if (offset < CacheCount)
    {
        auto copyCount = m_Cache.GetCachedCount() - offset;
        m_Cache.Read([&out](const EntryType cached[], int count) -> Result {
            std::memcpy(out, cached, sizeof(EntryType) * count);
            NN_RESULT_SUCCESS;
        }, offset, copyCount);
        listed += copyCount;
    }

    auto readMount = AcquireReadMount(m_MountManager);

    // ファイルからコピー
    auto restCopyCount = std::min(maxOutCount, GetCount()) - listed;
    if (restCopyCount > 0)
    {
        auto offsetInFile = std::max(CacheCount, offset);
        char path[MaxPathLen];
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            util::ReadFile(
                MakeEntryFilePath(path, sizeof(path), readMount.GetRootPath()),
                &out[listed],
                sizeof(EntryType) * restCopyCount,
                sizeof(EntryType) * offsetInFile));
        listed += restCopyCount;
    }

    return listed;
}

NN_OLSC_DEFINE_DATA_ARRAY_METHOD(void, lock())
{
    m_Lock.Lock();
}

NN_OLSC_DEFINE_DATA_ARRAY_METHOD(void, unlock())
{
    m_Lock.Unlock();
}

NN_OLSC_DEFINE_DATA_ARRAY_METHOD(void, lock() const)
{
    m_Lock.Lock();
}

NN_OLSC_DEFINE_DATA_ARRAY_METHOD(void, unlock() const)
{
    m_Lock.Unlock();
}

NN_OLSC_DEFINE_DATA_ARRAY_METHOD(Result, RemoveIf(const Predicate& pred))
{
    NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());
    NN_SDK_ASSERT(m_IsOpened);

    Result result = nn::olsc::ResultDatabaseEntryNotFound();
    ForEach([&pred, &result, this](const EntryType& entry) {
        if (pred(entry))
        {
            result = Remove(entry);
            return false;
        }
        return true;
    });

    NN_RESULT_THROW(result);
}

NN_OLSC_DEFINE_DATA_ARRAY_METHOD(Result, RemoveAll(const Predicate& pred))
{
    NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());
    NN_SDK_ASSERT(m_IsOpened);

    auto count = m_Metadata.entryCount;
    for (int i = 0; i < count; ++i)
    {
        auto index = count - i - 1; // 破壊的に削除していくので逆順でアクセスする
        EntryType entry;
        NN_RESULT_DO(At(&entry, index));
        if (pred(entry))
        {
            NN_RESULT_DO(Remove(entry));
        }
    }
    NN_RESULT_SUCCESS;
}

NN_OLSC_DEFINE_DATA_ARRAY_METHOD(nn::util::optional<EntryType>, FindIf(const Predicate& pred) const)
{
    NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());
    NN_SDK_ASSERT(m_IsOpened);

    nn::util::optional<EntryType> retValue = nn::util::nullopt;
    ForEach([&pred, &retValue](const EntryType& entry) {
        if (pred(entry))
        {
            retValue.emplace(entry);
            return false;
        }
        return true;
    });

    return retValue;
}

NN_OLSC_DEFINE_DATA_ARRAY_METHOD(void, ForEach(const Predicate& pred) const)
{
    NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());
    NN_SDK_ASSERT(m_IsOpened);

    auto count = m_Metadata.entryCount;
    for (int i = 0; i < count; ++i)
    {
        EntryType entry;
        NN_ABORT_UNLESS_RESULT_SUCCESS(At(&entry, i));
        if (!pred(entry))
        {
            return;
        }
    }
}

NN_OLSC_DEFINE_DATA_ARRAY_METHOD(Result, Replace(int index, const EntryType& entry))
{
    NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());
    NN_SDK_ASSERT(m_IsOpened);
    NN_ABORT_UNLESS(index < m_Metadata.entryCount);

    auto writeMount = AcquireWriteMount(m_MountManager);
    char path[MaxPathLen];

    NN_RESULT_DO(UpdateEntryFile(&entry, 1, index, MakeEntryFilePath(path, MaxPathLen, writeMount.GetRootPath())));

    if (index < CacheCount)
    {
        m_Cache.Update(index, entry);
    }

    NN_RESULT_SUCCESS;
}

NN_OLSC_DEFINE_DATA_ARRAY_METHOD(Result, ReplaceIf(const Predicate& pred, const EntryType& entry))
{
    NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());
    NN_SDK_ASSERT(m_IsOpened);

    auto pIndex = FindIndex(pred);
    NN_RESULT_THROW_UNLESS(pIndex, olsc::ResultDatabaseEntryNotFound());
    NN_RESULT_DO(Replace(*pIndex, entry));

    NN_RESULT_SUCCESS;
}

NN_OLSC_DEFINE_DATA_ARRAY_METHOD(EntryType, operator[](int index) const)
{
    EntryType entry;
    NN_ABORT_UNLESS_RESULT_SUCCESS(At(&entry, index));
    return entry;
}

}}}} // namespace nn::olsc::srv::database

#undef NN_OLSC_DEFINE_DATA_ARRAY_METHOD
