﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief ミドルウェア、ブラウザ向けの nn::nsd API
 */

#pragma once

#include <nn/nsd/nsd_Common.h>

namespace nn { namespace nsd {

//! @name ミドルウェア、ブラウザが扱う API
//! @{

    /**
     * @brief FQDN を別の FQDN へ解決します。エラーハンドリングが必須な API です。
     * @param[out]  pOut    解決された FQDN
     * @param[in]   fqdn    解決したい FQDN
     *
     * @retresult
     *      @handleresult{ResultInvalidSettingVersion}
     *      @handleresult{ResultInvalidSettingData}
     *      @handleresult{ResultFqdnTooLong, 処理結果の FQDN が長すぎます. % を含む 254 文字以上の FQDN を入力した場合に返ります.}
     *      @handleresult{ResultFqdnIsNotNullTerminated, 入力の FQDN が NULL 終端していません}
     * @endretresult
     *
     * @pre
     *  - pOut != nullptr
     *
     * @post
     *  - pOut は NULL 終端している
     *
     * @details
     *  - この関数はスレッドセーフです。
     *  - 解決する必要がない場合は入力と同じ FQDN が pOut へ出力されます。
     *
     *  本 API の失敗は、実運用においても起こりえるので、
     *  必ずエラーハンドリングを行ってください。
     *
     *  FQDN 解決が失敗するケースにおいて、
     *  エラーハンドリングやシーケンス分岐を行いたくない場合は @ref ResolveEx 関数を利用してください。
     *
     * @see     ResolveEx()
     */
    Result Resolve(Fqdn *pOut, const Fqdn& fqdn) NN_NOEXCEPT;

    /**
     * @brief FQDN を別の FQDN へ解決します。エラーハンドリングが必須ではない API です。
     * @param[out]  pOut    解決された FQDN
     * @param[in]   fqdn    解決したい FQDN
     *
     * @retresult
     *      @handleresult{ResultFqdnTooLong, 処理結果の FQDN が長すぎます. % を含む 254 文字以上の FQDN を入力した場合に返ります.}
     *      @handleresult{ResultFqdnIsNotNullTerminated, 入力の FQDN が NULL 終端していません}
     * @endretresult
     *
     * @pre
     *  - pOut != nullptr
     *
     * @post
     *  - pOut は NULL 終端している
     *
     * @details
     *  - この関数はスレッドセーフです。
     *  - 解決する必要がない場合は入力と同じ FQDN が pOut へ出力されます。
     *  - 入力の FQDN が不正であるときにだけエラーを返します。
     *
     *  FQDN 解決が失敗した場合には、無効な FQDN を出力します
     *  (文字列としては正しい仕様の FQDN を出力します)。@n
     *  この pOut を使って処理を進めても構いません。その場合 socket での名前解決が必ず失敗します。
     *
     * @see     Resolve()
     */
    Result ResolveEx(Fqdn *pOut, const Fqdn& fqdn) NN_NOEXCEPT;

//! @}

}} // nn::nsd
