﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/kvdb/kvdb_BoundedString.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/fs/fs_IEventNotifier.h>
#include <nn/fs/fs_ContentStorage.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/ns/srv/ns_ApplicationRecordDatabase.h>
#include <nn/ns/srv/ns_ApplicationViewManager.h>
#include <nn/ns/srv/ns_IntegratedContentManager.h>
#include <nn/ns/srv/ns_MarkFile.h>
#include <nn/ns/srv/ns_RequestServer.h>
#include <nn/ns/srv/ns_SaveDataInfo.h>
#include <nn/util/util_BitPack.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_UuidTypes.h>
#include "ns_OsUtil.h"

namespace nn { namespace ns { namespace srv {
    class SdCardManager
    {
    public:
        SdCardManager(IntegratedContentManager* contentManager, RequestServer* requestServer, ApplicationRecordDatabase* recordDb, ApplicationViewManager* viewManager, uint8_t* stack, size_t stackSize) NN_NOEXCEPT :
            m_ContentManager(contentManager),
            m_RequestServer(requestServer),
            m_RecordDb(recordDb),
            m_ViewManager(viewManager),
            m_Stack(stack),
            m_StackSize(stackSize)
        {
            m_MarkFileDirPath.AssignFormat("%s:/%s/%s/", MountName, fs::SdCardNintendoRootDirectoryName, fs::ContentStorageDirectoryName);
            m_ContentsDirPath.AssignFormat("%s:/%s/%s/", MountName, fs::SdCardNintendoRootDirectoryName, fs::ContentStorageDirectoryName);
            m_SaveDirPath.AssignFormat("%s:/%s/save", MountName, fs::SdCardNintendoRootDirectoryName);
        }

        Result Initialize(const SaveDataInfo& info) NN_NOEXCEPT;

        void Finalize() NN_NOEXCEPT;

        void Run() NN_NOEXCEPT;

        Result CheckMountStatus() NN_NOEXCEPT;

        Result GetLastMountUnexpectedResult() NN_NOEXCEPT;

        Result GetMountStatusChangedEvent(sf::Out<sf::NativeHandle> outValue) NN_NOEXCEPT;

        Result GetRemovedEvent(sf::Out<sf::NativeHandle> outValue) NN_NOEXCEPT;

        Result Cleanup() NN_NOEXCEPT;

        bool IsAttached() const NN_NOEXCEPT;

        Result NeedsSystemUpdateToFormat(bool* outValue) NN_NOEXCEPT;
        Result Format() NN_NOEXCEPT;
        Result GetLastFormatUnexpectedResult() const NN_NOEXCEPT;

        bool NeedsSystemCleanup() const NN_NOEXCEPT;

        void Insert() NN_NOEXCEPT;
        void Remove() NN_NOEXCEPT;

    private:
        const char* MountName{"sdcard"};
        static const int MarkFileDirPathLength = 32;
        typedef kvdb::BoundedString<MarkFileDirPathLength> Path;

    private:
        Result InitializeSdCard() NN_NOEXCEPT;
        Result Verify() NN_NOEXCEPT;
        bool CanAccessSdCard() NN_NOEXCEPT;
        void CleanupThread() NN_NOEXCEPT;
        void DetachSdCard() NN_NOEXCEPT;
        void UpdateMountStateResult(Result result) NN_NOEXCEPT;
        void Unmount() NN_NOEXCEPT;
        Result DoFormatOperation(Result result) NN_NOEXCEPT;

    private:
        IntegratedContentManager* m_ContentManager;
        RequestServer* m_RequestServer;
        ApplicationRecordDatabase* m_RecordDb;
        ApplicationViewManager* m_ViewManager;

        std::unique_ptr<fs::IEventNotifier> m_Notifier;
        os::SystemEventType m_SdCardDetectEvent;
        ManualClearSystemEvent m_ChangedEvent;
        ManualClearSystemEvent m_RemovedEvent;
        bool m_IsSdCardAttached{true};

        util::optional<os::ThreadType> m_Thread;
        uint8_t* m_Stack;
        size_t m_StackSize;

        os::EventType m_EndEvent;
        os::EventType m_EmulationEvent;

        mutable os::Mutex m_StateLock{true};
        mutable os::Mutex m_ThreadLock{false};

        MarkData m_SystemMarkData{};

        Result m_CurrentResult{ResultSuccess()};
        Result m_LastMountUnexpectedResult{ResultSuccess()};
        Result m_LastFormatUnexpectedResult{ResultSuccess()};

        Path m_MarkFileDirPath;
        Path m_ContentsDirPath;
        Path m_SaveDirPath;

        RequestServer::ManagedStop m_RequestStopper{};

        bool m_NeedsSystemCleanup{};
        bool m_IsInsertingEmulation{};
        bool m_IsRemovingEmulation{};
    };
}}}
