﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/arp/arp_Api.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_ContentStorage.h>
#include <nn/lr/lr_Path.h>
#include <nn/ns/ns_ApplicationLaunchInfo.h>
#include <nn/ns/ns_ApplicationRightsTypes.h>
#include <nn/ns/ns_ApplicationRightsSystemApi.h>
#include <nn/ns/srv/ns_ApplicationControlDataManager.h>
#include <nn/ns/srv/ns_ApplicationLaunchPropertyManager.h>
#include <nn/ns/srv/ns_ApplicationRecordDatabase.h>
#include <nn/ns/srv/ns_ApplicationVersionManager.h>
#include <nn/ns/srv/ns_CommitManager.h>
#include <nn/ns/srv/ns_IntegratedContentManager.h>
#include <nn/ns/srv/ns_GameCardManager.h>
#include <nn/ns/srv/ns_TicketManager.h>

namespace nn { namespace ns { namespace srv {

    class ApplicationLaunchManager
    {
    public:
        // 外部公開用
        Result LaunchApplication(os::ProcessId* outValue, const ApplicationLaunchInfo& info, uint8_t programIndex) NN_NOEXCEPT;

        Result CheckApplicationLaunchVersion(ncm::ApplicationId id) NN_NOEXCEPT;
        Result CheckApplicationLaunchRights(ncm::ApplicationId id) NN_NOEXCEPT;
        Result GetApplicationRightsOnClient(int* outCount, ApplicationRightsOnClient* outList, int listCount, ncm::ApplicationId appId, const account::Uid& uid, ApplicationRightsOnClientQueryFlag flags) NN_NOEXCEPT;

        Result CheckApplicationResumeRights(ncm::ApplicationId id) NN_NOEXCEPT;
        Result CheckContentsRights(const ncm::ContentMetaKey& key, ncm::StorageId storageId = ncm::StorageId::Any) const NN_NOEXCEPT;
        Result RegisterContentsExternalKey(const ncm::ContentMetaKey& key, ncm::StorageId storageId = ncm::StorageId::Any) const NN_NOEXCEPT;
        Result BoostSystemMemoryResourceLimit(int64_t boostSize) NN_NOEXCEPT;

        Result GetApplicationLaunchInfo(ApplicationLaunchInfo* outValue, ncm::ApplicationId appId) NN_NOEXCEPT;
        Result AcquireApplicationLaunchInfo(ApplicationLaunchInfo* outValue, os::ProcessId processId) NN_NOEXCEPT;

    public:
        // 内部用
        void Initialize(ApplicationRecordDatabase* recordDb, IntegratedContentManager* integrated, ApplicationVersionManager* version, GameCardManager* gameCard, ApplicationControlDataManager* control, TicketManager* ticket, CommitManager* commit) NN_NOEXCEPT;

        Result LaunchApplicationForDevelop(os::ProcessId* outValue, const ApplicationLaunchInfo& info) NN_NOEXCEPT;
        Result PrepareLaunchApplication(ApplicationLaunchInfo* outValue, const ApplicationLaunchInfo& info, uint8_t programIndex) NN_NOEXCEPT;
        Result AfterLaunchProgram(os::ProcessId processId, const ApplicationLaunchInfo& info, uint8_t programIndex) NN_NOEXCEPT;
        Result LaunchApplicationForAutoBoot(
            os::ProcessId* processId,
            ncm::ApplicationId id) NN_NOEXCEPT;

        Result CallLaunchProgramForApplication(os::ProcessId* outValue, const ApplicationLaunchInfo& info, uint8_t programIndex) NN_NOEXCEPT;

        Result RegisterLogoDataExternalKey(const ncm::ContentMetaKey& key, uint8_t programIndex) const NN_NOEXCEPT;

        Result GetProgramIndexForDevelop(uint8_t* outValue, const ApplicationLaunchInfo& info) const NN_NOEXCEPT;

        Result RedirectApplicationProgramForDevelop(ncm::ApplicationId id, uint8_t programIndex) NN_NOEXCEPT;

        Result GetApplicationRightsOnClientImpl(int* outCount, ApplicationRightsOnClient* outList, int listCount, ncm::ApplicationId appId, const account::Uid& uid, ApplicationRightsOnClientQueryFlag flags) NN_NOEXCEPT;

    private:
        IntegratedContentManager* m_pIntegrated;
        ApplicationRecordDatabase* m_pRecordDb;
        ApplicationVersionManager* m_pVersionManager;
        GameCardManager*           m_pGameCardManager;
        ApplicationControlDataManager* m_pControlManager;
        TicketManager*             m_pTicketManager;
        CommitManager*             m_pCommitManager;

        Result PrepareApplicationControlProperty(const ApplicationLaunchInfo& info, uint8_t programIndex) NN_NOEXCEPT;
        Result GetControlPath(ncm::Path* outValue, const ncm::ContentMetaKey& key, uint8_t programIndex) NN_NOEXCEPT;
        Result RedirectApplicationProgram(const ncm::ContentMetaKey& key, ncm::ApplicationId id, uint8_t programIndex, ncm::StorageId storageId) NN_NOEXCEPT;
        Result RegisterProgramPatch(const ncm::ContentMetaKey& key, ncm::ApplicationId id, uint8_t programIndex, ncm::StorageId storageId) NN_NOEXCEPT;
        Result GetContentPathByStorageId(lr::Path* outValue, const ncm::ContentMetaKey& key, ncm::ContentType type, uint8_t programIndex, ncm::StorageId targetStorage) NN_NOEXCEPT;

        Result LaunchApplicationImpl(os::ProcessId* outValue, const ApplicationLaunchInfo& info, uint8_t programIndex) NN_NOEXCEPT;
        Result RegisterContentExternalKeyImpl(const ncm::ContentMetaKey& key, uint8_t programIndex, ncm::ContentType contentType, ncm::StorageId storageId) const NN_NOEXCEPT;
        Result RegisterApplicationExternalKey(ncm::ApplicationId id) const NN_NOEXCEPT;
        Result CheckProgramContentRights(const ncm::ContentMetaKey& key) NN_NOEXCEPT;

        Result GetRequiredApplicationVersion(uint32_t* outValue, ncm::ApplicationId id) const NN_NOEXCEPT;

        Result RegisterAllAddOnContentLocationAndExternalKey(ncm::ApplicationId id) NN_NOEXCEPT;
        Result UnregisterAllAddOnContentLocation() NN_NOEXCEPT;

        enum class CheckLaunchVersionResultHandlingPolicy
        {
            ThrowAll,
            IgnoreResultApplicationUpdateRecommended,
        };
        Result CheckLaunchVersionImpl(ncm::ApplicationId id, CheckLaunchVersionResultHandlingPolicy policy) NN_NOEXCEPT;
        Result CheckApplicationLaunchRightsImpl(const util::optional<ncm::ContentMetaKey>& appMetaKey, const util::optional<ncm::ContentMetaKey>& patchMetaKey) NN_NOEXCEPT;

        struct ApplicationRightsOnClientWorkingInfo
        {
            ncm::ApplicationId appId;
            account::Uid uid;
            ApplicationContentType contentType;
            es::RightsId* pWorkingList;
            int listCount;
            ApplicationRightsOnClientFlag flags;
            int currentIndex;
        };
        Result UpdateApplicationRightsOnClientWorkingInfo(ApplicationRightsOnClientWorkingInfo* pRefValue, const ncm::ContentMetaKey& key, const account::Uid& uid) NN_NOEXCEPT;
        Result MakeApplicationRightsOnClient(ApplicationRightsOnClient* outValue, const ApplicationRightsOnClientWorkingInfo& info) NN_NOEXCEPT;

        ApplicationLaunchPropertyManager m_LaunchPropertyManager;
    };

}}}
