﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_StaticAssert.h>
#include <nn/fs/fs_File.h>
#include <rapidjson/encodedstream.h>
#include <nn/nim/nim_NetworkInstallManagerApi.h>

namespace nn { namespace ns { namespace srv { namespace detail { namespace json {

/** @brief 固定長の文字列バッファに文字列を格納するための InputStreamType の部分実装です。
    @details
        ImportJsonByRapidJson() の InputStreamType を部分的に実装しています。
        このクラスを継承したクラスで InputStreamType を完全に実装してください。

        このクラスは StringBufferForRapidJson::SetStringBuffer() で与えられるバッファの範囲に切り詰めて文字列を取り扱います。
        ここで切り詰められるのはJSONドキュメント中のペアの name と、文字列値の両方です。
        このバッファの範囲に収まらない文字列は、切り詰められた形で取り扱われます。
 */
class StringBufferForRapidJson
{
    NN_DISALLOW_COPY(StringBufferForRapidJson);

public:
    typedef char Ch;

private:
    Ch* m_Buffer;
    size_t m_BufferSize;
    size_t m_FilledSize;

protected:
    StringBufferForRapidJson() NN_NOEXCEPT
        : m_Buffer(nullptr)
        , m_BufferSize(0u)
    {
        // In-situ でなければ、StringBuffer は nullptr でも構わない。
    }
    ~StringBufferForRapidJson() NN_NOEXCEPT {} // default

public:
    void SetStringBuffer(Ch* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(buffer == nullptr || bufferSize > 0u);
        NN_SDK_ASSERT(reinterpret_cast<uintptr_t>(buffer) % NN_ALIGNOF(Ch) == 0);
        NN_SDK_ASSERT(bufferSize % sizeof(Ch) == 0);
        m_Buffer = buffer;
        m_BufferSize = bufferSize;
    }

    Ch* PutBegin() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(
            m_Buffer,
            "[nn::ns] -----------------------------------------------\n"
            "  ABORT: Streaming string output to nullptr. (internal)\n");
        m_FilledSize = 0u;
        return m_Buffer;
    }
    void Put(Ch c) NN_NOEXCEPT
    {
        if (m_FilledSize >= m_BufferSize)
        {
            NN_SDK_ASSERT(m_FilledSize == m_BufferSize);
            return;
        }
        m_Buffer[m_FilledSize ++] = c;
    }
    void Flush() NN_NOEXCEPT  { NN_ABORT("not implemented"); }
    size_t PutEnd(Ch* ptr) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(ptr == m_Buffer);
        NN_UNUSED(ptr);
        NN_ABORT_UNLESS(m_FilledSize > 0 && m_FilledSize <= m_BufferSize);
        if (m_Buffer[m_FilledSize - 1] != '\0')
        {
            m_Buffer[m_FilledSize - 1] = '\0';
            NN_SDK_LOG(
                "[nn::ns] INFO: Input string is truncated to length %u. (input data)\n",
                m_FilledSize - 1);
        }
        return m_FilledSize;
    }
};

/** @brief ストリームからの入力を一時的にメモリにプールして IO 効率を向上した InputStreamType の実装です。
    @details
        ImportJsonByRapidJson() の InputStreamType を実装しています。

        このクラスは抽象クラスです。
        BufferedStreamForRapidJson::FillBufferImpl() 仮想関数を実装してください。
        この関数はバッファに任意のデータを読み込むためのものです。
        読み込まれた内容は ImportJsonByRapidJson() から参照されます。
 */
class BufferedStreamForRapidJson
    : public StringBufferForRapidJson
{
    NN_DISALLOW_COPY(BufferedStreamForRapidJson);

private:
    typedef StringBufferForRapidJson Base;

public:
    typedef Base::Ch Ch;
    NN_STATIC_ASSERT(sizeof(Ch) == 1);

private:
    Ch* m_Buffer;
    size_t m_BufferSize;
    size_t m_TotalBufferedBytes;
    size_t m_BufferedBytes;
    size_t m_PositionInBuffer;

    bool FillBuffer() NN_NOEXCEPT
    {
        m_PositionInBuffer = 0u;
        m_BufferedBytes = FillBufferImpl(m_Buffer, m_BufferSize);
        m_TotalBufferedBytes += m_BufferedBytes;
        return m_BufferedBytes > 0;
    }

protected:
    BufferedStreamForRapidJson() NN_NOEXCEPT
        : m_Buffer(nullptr)
        , m_BufferSize(0u)
        , m_TotalBufferedBytes(0u)
        , m_BufferedBytes(0u)
        , m_PositionInBuffer(0u)
    {
    }

    // これを実装せよ
    virtual size_t FillBufferImpl(void* buffer, size_t bufferSize) NN_NOEXCEPT = 0;

public:
    void SetInputBuffer(Ch* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(buffer && bufferSize > 0u);
        NN_SDK_ASSERT(reinterpret_cast<uintptr_t>(buffer) % NN_ALIGNOF(Ch) == 0);
        NN_SDK_ASSERT(bufferSize % sizeof(Ch) == 0);
        m_Buffer = buffer;
        m_BufferSize = bufferSize;
    }

    inline Ch Peek() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Buffer != nullptr);
        NN_SDK_ASSERT(m_BufferSize > 0);

        if (m_PositionInBuffer >= m_BufferedBytes)
        {
            NN_SDK_ASSERT(m_PositionInBuffer == m_BufferedBytes);
            if (!FillBuffer())
            {
                return '\0';
            }
        }
        return m_Buffer[m_PositionInBuffer];
    }
    inline Ch Take() NN_NOEXCEPT
    {
        auto c = Peek();
        ++ m_PositionInBuffer;
        return c;
    }
    inline size_t Tell() const NN_NOEXCEPT
    {
        return (m_TotalBufferedBytes - m_BufferedBytes) + m_PositionInBuffer;
    }
};



class MemoryInputStreamBaseForRapidJson
    : public nn::ns::srv::detail::json::StringBufferForRapidJson
{
public:
    typedef char Ch;
private:
    typedef nn::ns::srv::detail::json::StringBufferForRapidJson StringBuffer;
    Ch* m_Buffer;
protected:
    Ch* GetBufferAddress() NN_NOEXCEPT
    {
        return m_Buffer;
    }
    MemoryInputStreamBaseForRapidJson(Ch* buffer, size_t bufferSize) NN_NOEXCEPT
        : m_Buffer(buffer)
    {
        NN_ABORT_UNLESS(buffer != nullptr);
        SetStringBuffer(buffer, bufferSize);
    }
};

class MemoryInputStreamForRapidJson
    : public MemoryInputStreamBaseForRapidJson
{
public:
    typedef char Ch;

private:
    const char* m_Ptr;
    size_t m_Bytes;
    size_t m_Pos;

public:
    MemoryInputStreamForRapidJson(void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
        : MemoryInputStreamBaseForRapidJson(reinterpret_cast<Ch*>(workBuffer), workBufferSize)
        , m_Ptr(nullptr)
        , m_Bytes(0u)
        , m_Pos(0u)
    {
    }
    ~MemoryInputStreamForRapidJson() NN_NOEXCEPT
    {
    }
    void Set(const char* ptr, size_t bytes) NN_NOEXCEPT
    {
        m_Ptr = ptr;
        m_Bytes = bytes;
    }
    Ch Peek() const NN_NOEXCEPT
    {
        return m_Pos == m_Bytes? '\0': m_Ptr[m_Pos];
    }
    Ch Take() NN_NOEXCEPT
    {
        auto c = Peek();
        if (c != '\0')
        {
            ++ m_Pos;
        }
        return c;
    }
    size_t Tell() const NN_NOEXCEPT
    {
        return m_Pos;
    }
    nn::Result GetResult() const NN_NOEXCEPT
    {
        NN_RESULT_SUCCESS;
    }
};

typedef nne::rapidjson::EncodedInputStream<nne::rapidjson::UTF8<>, detail::json::MemoryInputStreamForRapidJson> Utf8InputStream;

class EncodedMemoryInputStreamForRapidJson
    : public Utf8InputStream
{
public:
    explicit EncodedMemoryInputStreamForRapidJson(MemoryInputStreamForRapidJson& inputStream) NN_NOEXCEPT
        : Utf8InputStream(inputStream), m_InputStream(inputStream)
    {
    }

    Ch* PutBegin() NN_NOEXCEPT
    {
        return m_InputStream.PutBegin();
    }
    void Put(Ch c) NN_NOEXCEPT
    {
        m_InputStream.Put(c);
    }
    void Flush() NN_NOEXCEPT { NN_ABORT("not implemented"); }

    size_t PutEnd(Ch* ptr) NN_NOEXCEPT
    {
        return m_InputStream.PutEnd(ptr);
    }

    nn::Result GetResult() const NN_NOEXCEPT
    {
        return m_InputStream.GetResult();
    }

private:
    MemoryInputStreamForRapidJson& m_InputStream;
};

template<int StringBufferSize = 256>
class AsyncDataInputStreamForRapidJson : public detail::json::BufferedStreamForRapidJson
{
public:
    AsyncDataInputStreamForRapidJson(nim::AsyncData* asyncData, char* readBuffer, size_t readBufferSize) NN_NOEXCEPT
        : m_AsyncData(asyncData), m_BufferedOffset(0), m_LastResult(nn::ResultSuccess())
    {
        SetStringBuffer(m_StringBuffer, sizeof(m_StringBuffer));
        SetInputBuffer(readBuffer, readBufferSize);
    }

    Result GetResult() NN_NOEXCEPT
    {
        return m_LastResult;
    }
protected:
    virtual size_t FillBufferImpl(void* buffer, size_t bufferSize) NN_NOEXCEPT NN_OVERRIDE
    {
        size_t readSize;
        m_LastResult = m_AsyncData->Read(&readSize, m_BufferedOffset, buffer, bufferSize);
        if (m_LastResult.IsFailure())
        {
            return 0;
        }
        m_BufferedOffset += readSize;
        return readSize;
    }
private:
    char m_StringBuffer[StringBufferSize];
    nim::AsyncData* m_AsyncData;
    int64_t m_BufferedOffset;
    Result m_LastResult;
};

}}}}} // ~namespace nn::ns::srv::detail::json
