﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_ApplicationId.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/account/account_Types.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/os/os_SystemEvent.h>

#include <nn/ns/ns_UserResourceManagementTypes.h>
#include <nn/ns/detail/ns_UserResourceManagementInternalTypes.h>


namespace nn { namespace ns {

namespace detail {
class IProgressMonitorForDeleteUserSaveDataAll;
}

class ProgressMonitorForDeleteUserSaveDataAll;

}} // namespace nn::ns::detail


namespace nn { namespace ns {

/** @brief 指定したユーザーアカウントに関して、セーブデータの統計情報を取得します。

    @return   処理の結果が返ります。
    @retval   nn::ResultSuccess 成功
    @retval   上記以外 失敗しました。エラービューアでエラー内容を表示してください。

    @details
        取得した統計情報は、 nn::ns::DeleteUserSaveDataAll() の進捗に対する母数として使用できます。
*/
Result CalculateUserSaveDataStatistics(ns::UserSaveDataStatistics* pOut, const account::Uid& uid) NN_NOEXCEPT;

/** @brief 指定したユーザーアカウントに関して、そのセーブデータをすべて削除します。

    @return   処理の結果が返ります。
    @retval   nn::ResultSuccess 成功
    @retval   上記以外 失敗しました。エラービューアでエラー内容を表示してください。

    @details
        返却される ProgressMonitorForDeleteUserSaveDataAll オブジェクトを使用してその進捗を取得できます。
*/
Result DeleteUserSaveDataAll(ProgressMonitorForDeleteUserSaveDataAll* pOut, const account::Uid& uid) NN_NOEXCEPT;

/** @brief 指定したユーザーアカウントに関して、指定した nn::fs::SystemSaveDataId のセーブデータを削除します。

    @return   処理の結果が返ります。
    @retval   nn::ResultSuccess 成功
    @retval   上記以外 失敗しました。エラービューアでエラー内容を表示してください。
*/
Result DeleteUserSystemSaveData(const account::Uid& uid, fs::SystemSaveDataId sysSaveId) NN_NOEXCEPT;

/** @brief 指定したユーザーアカウントに関して、ネットワークサービスアカウントの登録を解除し、関連する情報を削除します。

    @return   処理の結果が返ります。
    @retval   nn::ResultSuccess 成功
    @retval   上記以外 失敗しました。エラービューアでエラー内容を表示してください。

    @pre
        - インターネットに接続されている
        - ユーザーアカウントにネットワークサービスアカウントが登録されている
    @post
        - ユーザーアカウントにネットワークサービスアカウントが登録されていない
        - ユーザーアカウントの「いっしょに遊んだユーザー情報」が削除されている
        - ニンテンドーアカウントがネットワークサービスアカウントに連携している場合、その権利情報が本体システムから削除されている

    @details
        指定したユーザーアカウントに関して、ネットワークサービスアカウントの登録を解除し、関連する情報を削除します。
        関連する情報は次の通りです。

        - ユーザーアカウントの「いっしょに遊んだユーザー情報」
        - ニンテンドーアカウントがネットワークサービスアカウントに連携している場合、その権利情報
*/
Result UnregisterNetworkServiceAccount(const account::Uid& uid) NN_NOEXCEPT;

/** @brief 指定したユーザーアカウントに関して、セーブデータを削除したうえで、ネットワークサービスアカウントの登録を解除し、関連する情報を削除します。

    @return   処理の結果が返ります。
    @retval   nn::ResultSuccess 成功
    @retval   上記以外 失敗しました。エラービューアでエラー内容を表示してください。

    @pre
        - インターネットに接続されている
        - ユーザーアカウントにネットワークサービスアカウントが登録されている
    @post
        - ユーザーアカウントにネットワークサービスアカウントが登録されていない
        - ユーザーアカウントのセーブデータが、アプリケーションやシステムに関わらず、すべて削除されている
        - ニンテンドーアカウントがネットワークサービスアカウントに連携している場合、その権利情報が本体システムから削除されている

    @details
        指定したユーザーアカウントに関して、セーブデータを削除したうえで、ネットワークサービスアカウントの登録を解除し、関連する情報を削除します。
        関連する情報は次の通りです。

        - ニンテンドーアカウントがネットワークサービスアカウントに連携している場合、その権利情報
*/
Result UnregisterNetworkServiceAccountWithUserSaveDataDeletion(const account::Uid& uid) NN_NOEXCEPT;


/** @brief nn::ns::DeleteUserSaveDataAll() 関数の進捗を取得するためのクラス
    @details
        nn::ns::DeleteUserSaveDataAll() 関数の進捗を取得するためのクラスです。

        このクラスをデフォルトコンストラクタで構築すると、オブジェクトは未初期化状態で作成されます。
        未初期化状態のオブジェクトに対するデストラクタ以外の呼び出しは失敗します。

        未初期化状態のオブジェクトを nn::ns::DeleteUserSaveDataAll() に与えこれが成功した場合、オブジェクトは初期化済み状態となります。
        初期化済み状態のオブジェクトからは進捗を取得できます。
*/
class ProgressMonitorForDeleteUserSaveDataAll
{
    NN_DISALLOW_COPY(ProgressMonitorForDeleteUserSaveDataAll);

private:
    detail::IProgressMonitorForDeleteUserSaveDataAll* m_Ptr;
    detail::ProgressForDeleteUserSaveDataAll m_Data;

public:
    /** @brief デフォルトコンストラクタ
        @details
            オブジェクトを未初期化状態で構築します。
            未初期化状態のオブジェクトを nn::ns::DeleteUserSaveDataAll() に与えこれが成功した場合、オブジェクトは初期化済み状態となります。
    */
    ProgressMonitorForDeleteUserSaveDataAll() NN_NOEXCEPT;

    /** @brief デストラクタ
        @pre
            次のいずれか
            - *this が未初期化状態
            - *this が初期化済み状態で、 this->IsFinished() == true
        @details
            もしオブジェクトが初期化済み状態で this->IsFinished() == false の場合、デストラクタの呼び出しは失敗します。
    */
    virtual ~ProgressMonitorForDeleteUserSaveDataAll() NN_NOEXCEPT;

    /** @brief 処理の終了の通知を受けるためのシステムイベントを取得します。
        @details
            this->GetSystemEvent() は this につきひとつのシステムイベントのみを発行します。
    */
    void GetSystemEvent(os::SystemEvent* pOut) NN_NOEXCEPT;

    /** @brief 処理が終了している場合に true を返します。
    */
    bool IsFinished() NN_NOEXCEPT;

    /** @brief 処理が終了している場合にその結果を返します。

        @return   処理の結果が返ります。
        @retval   nn::ResultSuccess 成功
        @retval   上記以外 失敗しました。エラービューアでエラー内容を表示してください。
    */
    Result GetResult() NN_NOEXCEPT;

    /** @brief 処理開始時点から現在までの経過時間を取得します。
    */
    TimeSpan GetElapsedTime() const NN_NOEXCEPT;

    /** @brief 進捗状況を更新します。
    */
    void Update() NN_NOEXCEPT;

    /** @brief Update() 呼び出し時点での進捗状況を取得します。
    */
    UserSaveDataStatistics GetStatistics() const NN_NOEXCEPT;

    /** @brief Update() 呼び出し時点での処理中のセーブデータが SystemSaveData であるかを返します。
    */
    bool IsSystemSaveData() const NN_NOEXCEPT;

    /** @brief Update() 呼び出し時点での処理中のセーブデータがアプリケーションのものであれば、その ID を返します。
        @details
            false が返る場合、処理中のセーブデータはアプリケーションのものではありません。
            false が返る場合、処理中のセーブデータが SystemSaveData であることは保証されません。
            処理中のセーブデータが SystemSaveData かは IsSystemSaveData() を使用して判定してください。
    */
    bool TryGetCurrentApplicationId(ApplicationId* pOut) const NN_NOEXCEPT;

    // 内部使用
    explicit ProgressMonitorForDeleteUserSaveDataAll(detail::IProgressMonitorForDeleteUserSaveDataAll* ptr) NN_NOEXCEPT;
    ProgressMonitorForDeleteUserSaveDataAll(ProgressMonitorForDeleteUserSaveDataAll&& rhs) NN_NOEXCEPT;
    ProgressMonitorForDeleteUserSaveDataAll& operator=(ProgressMonitorForDeleteUserSaveDataAll&& rhs) NN_NOEXCEPT;
};

}}  // namespace ns
