﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/ns/ns_Async.h>

namespace nn { namespace ns {
     /**
     * @brief    予約購入状態のアプリケーションの権利の取得を非同期処理で要求します。
     *
     * @details  非同期処理をリクエストして AsyncResult を返します。
     *           処理の結果は AsyncResult から取得してください。
     *           処理が成功すると、バックグラウンドで予約購入状態のアプリケーションの権利の取得が始まります。
     *
     *           処理が成功するためには、インフラ通信が確立している必要があります。
     *           この関数で同時にリクエストできる非同期処理は最大１つです。
     *
     * @return   処理の結果が返ります。通信エラーなどは下位レイヤの Result が返されます。必要があればエラーコードを表示してください。
     * @retval   ResultApplicationRecordNotFound                    指定されたアプリケーション記録が見つかりません。
     * @retval   ResultPrepurchasedContentNotFound                  予約購入状態のコンテンツが見つかりません。
     * @retval   ResultOutOfMaxRunningTask                          同時に発行できるリクエスト数が最大に達しています。
     * @retval   ResultPrepurchasedContentStillUnavailable          （AsyncResult から返される Result）予約購入状態のコンテンツがまだ利用可能ではありません。
     * @retval   ResultPrepurchasedContentPartiallyStillUnavailable （AsyncResult から返される Result）予約購入状態のコンテンツの一部がまだ利用可能ではありません。
     * @retval   ResultPrepurchasedRightsLost                       （AsyncResult から返される Result）予約購入状態のコンテンツの権利が失われました。
     * @retval   ResultTryLater                                     （AsyncResult から返される Result）サーバの負荷を軽減するため後でリトライしてください。
     * @retval   ResultCanceled                                     （AsyncResult から返される Result）処理がキャンセルされました。
     * @retval   ec::ResultSystemUpdateRequired                     （AsyncResult から返される Result）処理を行うためには本体更新が必要です。
     */
     Result RequestDownloadApplicationPrepurchasedRights(AsyncResult* outValue, ncm::ApplicationId id) NN_NOEXCEPT;

     struct ApplicationTicketInfo
     {
         enum class Status : Bit8
         {
             Owned,                             // 権利を保有している
             PrepurchasedAndMaybeReleased,      // 予約していて発売済みである
             PrepurchasedButMaybeNotReleased,   // 予約していて未発売である
             NotOwned,                          // 権利を保有していない
         };

         Status status;
         Bit8 reserved[7];
     };

     /**
     * @brief    アプリケーションの権利情報を取得します。
     *
     * @details  チケットを保有しているかや予約購入をしているかのみで権利の状態を判定し、それらが有効であるかの判定は行いません。
     *           そのためこの API で権利を保有している状態である、または予約購入をしている状態と判定された場合でも実際には権利を利用できない場合があります。
     *
     * @param[out]  outValue    取得したアプリケーションの権利情報
     * @param[in]   id          対象のアプリケーション
     *
     * @return   処理の結果が返ります。
     */
     Result GetApplicationTicketInfo(ApplicationTicketInfo* outValue, ncm::ApplicationId id) NN_NOEXCEPT;

     /**
     * @brief    全てのアカウントの権利の同期を非同期処理で要求します。
     *
     * @details  この関数の呼出し後は、機器認証している全てのアカウントの機器認証権利が利用可能になります。
     *           内部的には、機器認証の権利の発行をサーバに依頼し、全アカウントの eLicense 同期処理とチケット同期処理を行います。
     *           非同期処理をリクエストして AsyncResult を返します。
     *           処理の結果は AsyncResult から取得してください。
     *
     *           処理が成功するためには、インフラ通信が確立している必要があります。
     *           この関数で同時にリクエストできる非同期処理は最大１つです。
     *
     * @return   処理の結果が返ります。通信エラーなどは下位レイヤの Result が返されます。必要があればエラーコードを表示してください。
     * @retval   ResultOutOfMaxRunningTask                       同時に発行できるリクエスト数が最大に達しています。
     * @retval   ResultCanceled                                  （AsyncResult から返される Result）処理がキャンセルされました。
     * @retval   ec::ResultDeviceAccountNotRegistered            （AsyncResult から返される Result）デバイスアカウント情報が保存されていません。
     * @retval   ec::ResultSystemUpdateRequired                  （AsyncResult から返される Result）処理を行うためには本体更新が必要です。
     */
     Result RequestSyncRights(AsyncResult* outValue) NN_NOEXCEPT;

}}  // namespace nn::ns
