﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/ncm/ncm_ContentMetaKey.h>
#include <nn/ns/ns_ApplicationRecordApi.h>

namespace nn { namespace ns {
    enum class ApplicationRecordAttribute : Bit8
    {
        AutoDeleteDisabled = 1u << 0,   // デフォルトが Enable
        AutoUpdateEnabled = 1u << 1,
    };

    inline void SetApplicationRecordAttribute(ApplicationRecord* record, ApplicationRecordAttribute attribute, bool isOn) NN_NOEXCEPT
    {
        if (isOn)
        {
            record->attributes |= static_cast<Bit8>(attribute);
        }
        else
        {
            record->attributes &= ~static_cast<Bit8>(attribute);
        }
    }

    inline bool GetApplicationRecordAttribute(const ApplicationRecord& record, ApplicationRecordAttribute attribute) NN_NOEXCEPT
    {
        return (record.attributes & static_cast<Bit8>(attribute)) == static_cast<Bit8>(attribute);
    }

    Result PushApplicationRecord(ncm::ApplicationId id, ApplicationEvent appEvent, const ncm::StorageContentMetaKey keyList[], int count) NN_NOEXCEPT;

    Result ListApplicationRecordContentMeta(int* outCount, ncm::StorageContentMetaKey outList[], int count, ncm::ApplicationId id, int offset) NN_NOEXCEPT;

    Result ListApplicationRecordInstalledContentMeta(int* outCount, ncm::StorageContentMetaKey outList[], int count, ncm::ApplicationId id, int offset) NN_NOEXCEPT;

    Result DeleteApplicationRecord(ncm::ApplicationId id) NN_NOEXCEPT;

    struct AutoUpdateInfo
    {
        ncm::PatchId patchId;
        uint32_t currentVersion;
        Bit8 reserved[4];
    };

    struct RequestUpdateInfo
    {
        Bit32 reasonResultInnerValue;
        bool needsUpdate;
        Bit8 reserved[3];

        void SetReasonResult(Result result) NN_NOEXCEPT
        {
            reasonResultInnerValue = result.GetInnerValueForDebug();
        }

        Result GetReasonResult() const NN_NOEXCEPT
        {
            return result::detail::ConstructResult(reasonResultInnerValue);
        }
    };

    struct ApplicationRecordProperty
    {
        enum class PreInstallFlag : Bit8
        {
            IsPreInstalledApplication   = 1 << 0,
        };

        Bit32 terminateResult;
        bool needsCleanupAoc;
        Bit8 preInstallFlag;
        Bit8 reserved[2];
        AutoUpdateInfo autoUpdateInfo;
        RequestUpdateInfo requestUpdateInfo;
        Bit8 reserved2[480];

        void SetTerminateResult(Result result) NN_NOEXCEPT
        {
            terminateResult = result.GetInnerValueForDebug();
        }

        Result GetTerminateResult() const NN_NOEXCEPT
        {
            return result::detail::ConstructResult(terminateResult);
        }

        bool IsPreInstalledApplication() const NN_NOEXCEPT
        {
            constexpr auto flag = static_cast<Bit8>(PreInstallFlag::IsPreInstalledApplication);
            return (preInstallFlag & flag) == flag;
        }
    };
    NN_STATIC_ASSERT(sizeof(ApplicationRecordProperty) == 512);

    Result GetApplicationRecord(ApplicationRecord* outValue, ncm::ApplicationId id) NN_NOEXCEPT;

    Result GetApplicationRecordProperty(ApplicationRecordProperty* outValue, ncm::ApplicationId id) NN_NOEXCEPT;

    Result EnableApplicationAutoUpdate(ncm::ApplicationId id) NN_NOEXCEPT;

    Result DisableApplicationAutoUpdate(ncm::ApplicationId id) NN_NOEXCEPT;

    Result RequestApplicationUpdate(ncm::ApplicationId id, Result reason) NN_NOEXCEPT;

    Result RequestCleanupAddOnContents(ncm::ApplicationId id) NN_NOEXCEPT;

    Result ClearPreInstalledApplicationFlag(ncm::ApplicationId id) NN_NOEXCEPT;
}}  // namespace nn::ns
