﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/npns/npns_Types.h>
#include <nn/npns/detail/npns_InternalTypes.h>
#include <nn/os.h>
#include <nn/account.h>

namespace nn
{
namespace npns
{
    ///////////////////////////////////////////////////////////////////////////
    //! @name プッシュ通知ライブラリの初期化・終了 API
    //! @{

    /**
     * @brief   ライブラリの初期化をおこない、システム用の機能を使用可能な状態にします。
     *
     * @return  処理の結果が返ります。
     *
     * @details システムが使用するための特別な API を呼び出すための初期化処理を行います。@n
     *          一般向けの機能を利用するために別途 @ref Initialize を呼ぶ必要はありません。@n
     *          ただし、@ref Initialize より先に呼び出す必要があります。
     */
    nn::Result InitializeForSystem() NN_NOEXCEPT;

    /**
     * @brief   ライブラリの利用を終了します。
     *
     * @return  処理の結果が返ります。
     *
     * @details @ref nn::npns::InitializeForSystem 関数を複数回呼んでいる場合、ライブラリの利用を完全に終了するには同じ回数だけこの関数を呼ぶ必要があります。@n
     *
     *          ライブラリの利用が完全に終了されると、システムで確保していたリソースが解放されます。@n
     *
     *          ライブラリが未初期化の状態でコールしても副作用はありません。
     */
    void FinalizeForSystem() NN_NOEXCEPT;

    //! @}

    ///////////////////////////////////////////////////////////////////////////
    //! @name プッシュ通知における受信 API
    //! @{

    /**
     * @brief   どのプロセスも受信の意思を表明しなかった宛先を持つ通知を受信します。本体機能向けです。
     *
     * @return  処理の結果が返ります。
     *
     * @details
     *
     */
    nn::Result ListenUndelivered() NN_NOEXCEPT;

    //! @}

    ///////////////////////////////////////////////////////////////////////////
    //! @name プッシュ通知におけるトピック管理 API
    //! @{

    /**
     * @brief   指定したトピックを購読します
     *
     * @pre
     *  - 通知サーバに接続済み: GetState() == State_Connected
     *
     * @param[in] pTopic                        対象のトピック名
     *
     * @return  処理の結果が返ります。
     * @retval  ResultSuccess                   トピックの購読に成功、もしくは既に購読していた
     * @retval  ResultNotConnected              通知サーバに接続していない
     * @retval  ResultTopicNotFound             指定されたトピックが存在しない
     * @retval  ResultCommandTimeOut            コマンドの応答を一定期間内に得られなかった
     *
     * @details 指定したトピックを購読するようにサーバに要求します。オフラインでは使用できません。
     *
     */
    nn::Result SubscribeTopic(const char* pTopic) NN_NOEXCEPT;

    /**
     * @brief   指定したトピックの購読を解除します
     *
     * @pre
     *  - 通知サーバに接続済み: GetState() == State_Connected
     *
     * @param[in] pTopic                        対象のトピック名
     *
     * @return  処理の結果が返ります。
     * @retval  ResultSuccess                   トピックの購読解除に成功
     * @retval  ResultNotConnected              通知サーバに接続していない
     * @retval  ResultTopicNotFound             指定されたトピックが存在しない
     * @retval  ResultTopicNotSubscribed        指定されたトピックは購読していない
     * @retval  ResultCommandTimeOut            コマンドの応答を一定期間内に得られなかった
     *
     * @details 指定したトピックの購読を解除するようにサーバに要求します。オフラインでは使用できません。
     *
     */
    nn::Result UnsubscribeTopic(const char* pTopic) NN_NOEXCEPT;

    /**
     * @brief   指定したトピックが存在するかを調べます
     *
     * @pre
     *  - 通知サーバに接続済み: GetState() == State_Connected
     *
     * @param[out] pIsSubscribing               存在の有無
     * @param[in] pTopic                        対象のトピック名
     *
     * @return  処理の結果が返ります。
     * @retval  ResultSuccess                   成功
     * @retval  ResultNotConnected              通知サーバに接続していない
     * @retval  ResultCommandTimeOut            コマンドの応答を一定期間内に得られなかった
     *
     * @details 指定したトピックが存在するかをサーバに問い合わせます。オフラインでは使用できません。
     *
     */
    nn::Result QueryIsTopicExist(bool* pIsSubscribing, const char* pTopic) NN_NOEXCEPT;
    //! @}

    ///////////////////////////////////////////////////////////////////////////
    //! @name プッシュ通知におけるトークン管理 API
    //! @{

    /**
     * @brief   通知トークンを作成します
     *
     * @pre
     *  - JID が作成済み: GetJid() == ResultSuccess
     *
     * @param[out] pTokenOut            作成された通知トークンの格納先
     * @param[in] uid                   通知の宛先となるユーザ識別子（ユーザを区別しない場合は 0 を指定します）
     * @param[in] applicationId         通知の宛先となアプリケーション ID
     *
     * @return  処理の結果が返ります。@ref nn::http のエラーがそのまま返る場合があります。
     * @retval  ResultSuccess                   成功
     * @retval  ResultAccountNotAvailable       ID が未作成
     *
     * @details 通知トークンの作成をサーバに要求します。オフラインでは使用できません。
     */
    nn::Result CreateToken(NotificationToken* pTokenOut, const nn::account::Uid& uid, ApplicationId applicationId) NN_NOEXCEPT;

    /**
     * @brief   通知トークンを削除します
     *
     * @return  処理の結果が返ります。
     *
     * @details 通知トークンの削除をサーバに要求します。オフラインでは使用できません。
     *
     */
    nn::Result DestroyToken(const nn::account::Uid& uid, ApplicationId applicationId) NN_NOEXCEPT;

    /**
     * @brief   BaaS 向け通知トークンを作成し、指定した Uid に紐づく DA に関連付けます
     *
     * @param[in] uid                   紐づける DA を持つユーザの Uid
     *
     * @return  処理の結果が返ります。@ref nn::account のエラーがそのまま返る場合があります。
     *
     * @details  BaaS 向け通知トークンを作成し、指定した Uid に紐づく DA に関連付けます。オフラインでは使用できません。
     *
     */
    nn::Result UploadTokenToBaaS(const nn::account::Uid& uid) NN_NOEXCEPT;

    /**
     * @brief   BaaS 向け通知トークンを削除します
     *
     * @param[in] uid                   紐づけた DA を持つユーザの Uid
     *
     * @return  処理の結果が返ります。
     *
     * @details BaaS 向け通知トークンの削除をサーバに要求します。オフラインでは使用できません。
     *
     */
    nn::Result DestroyTokenForBaaS(const nn::account::Uid& uid) NN_NOEXCEPT;

    //! @}

    ///////////////////////////////////////////////////////////////////////////
    //! @name プッシュ通知のアカウント管理 API
    //! @{

    /**
     * @brief   NPNS サーバのログインに使用する ID を取得します。
     *
     * @param[out] pJidOut              ID の格納先バッファ
     * @param[in]  length               格納先バッファの長さ
     *
     * @return  処理の結果が返ります。
     * @retval  ResultSuccess                   成功
     * @retval  ResultAccountNotAvailable       ID が未作成
     *
     * @details NPNS サーバのログインに使用する ID を取得します。この ID の文字列を表示することは避けてください。@n
     *          引っ越しや修理移行時には ID が変わります。得られる文字列は NULL 終端されます。
     *
     */
    nn::Result GetJid(char* pJidOut, size_t length) NN_NOEXCEPT;

    /**
     * @brief   NPNS サーバのログインに使用する ID を作成します。
     *
     * @return  処理の結果が返ります。
     *
     * @details NPNS サーバのログインに使用する ID を作成します。サーバ都合でアカウントが無効になっていた場合は、有効にします。
     *
     */
    nn::Result CreateJid() NN_NOEXCEPT;

    /**
     * @brief   NPNS サーバのログインに使用する ID を削除します。
     *
     * @return  処理の結果が返ります。
     *
     * @details NPNS サーバのログインに使用する ID を削除します。通知は一切受信できなくなり、通知トークンは全て無効になります。@n
     *          @ref Suspend を呼んでいない場合は自律動作により新しい ID が自動的に作成されることに注意してください。
     *
     */
    nn::Result DestroyJid() NN_NOEXCEPT;

    /**
     * @brief   NPNS サーバのログインに使用する ID とパスワードを設定します。
     *
     * @pre
     *  - JID が未作成: GetJid() == ResultTBD
     *
     * @return  処理の結果が返ります。
     *
     * @details NPNS サーバのログインに使用する ID を設定します。
     *
     */
    nn::Result AttachJid(const char* pJid, const char* pPassword) NN_NOEXCEPT;

    /**
     * @brief   （未実装）NPNS サーバのログインに使用する ID とパスワードを取得すると同時にデーモン内の設定から消去します。
     *
     * @param[out] pJidOut              ID の格納先バッファ
     * @param[in]  jidLength            ID の格納先バッファの長さ
     * @param[out] pPasswordOut         パスワードの格納先バッファ
     * @param[in]  passwordLength       パスワードの格納先バッファの長さ
     *
     * @pre
     *  - JID が作成済み: GetJid() == ResultSuccess
     *
     * @return  処理の結果が返ります。
     *
     * @details 他のデバイスに認証情報を移動するために、ID とパスワードを取得すると同時にデーモン内の設定から消去します。@n
     *          得られる文字列は NULL 終端されます。
     *
     */
    nn::Result DetachJid(char* pJidOut, size_t jidLength, char* pPasswordOut, size_t passwordLength) NN_NOEXCEPT;

    //! @}

    ///////////////////////////////////////////////////////////////////////////
    //! @name プッシュ通知の状態制御 API
    //! @{

    nn::Result Suspend() NN_NOEXCEPT;
    nn::Result Resume() NN_NOEXCEPT;

    //! @}

    ///////////////////////////////////////////////////////////////////////////
    //! @name プッシュ通知のプレイレポート制御 API
    //! @{


    nn::Result GetStatistics(nn::npns::Statistics* pOutStatistics) NN_NOEXCEPT;

    /**
     * @brief プレイレポートの送信要求イベントをバインドします。
     *
     * @param[out] pEventType システムイベント
     * @param[in]  clearMode  イベントのクリアモード
     *
     * @pre
     *   - ライブラリが初期化済み
     *   - pOutEventType != nullptr
     *   - pOutEventType は未初期状態
     *
     * @post
     *   - pOutEventType は初期化状態
     *
     * @details
     *  未初期化の SystemEventType のポインタとクリアモードを渡すことで、
     *  プレイレポートの送信要求イベントを受け取るための ReadableHandle がアタッチされて初期化されます。
     */
    nn::Result BindPlayReportRequestEvent(nn::os::SystemEventType* pEventType, nn::os::EventClearMode clearMode) NN_NOEXCEPT;

    //! @}

    ///////////////////////////////////////////////////////////////////////////
    //! @name 通知リソースのリフレッシュ API
    //! @{

    /**
     * @brief   通知のリソースを更新します。(トピック購読情報は対象外)
     *
     * @param[in] uids  DAを持つユーザーの Uid リスト
     * @param[in] count uidsの要素数
     *
     * @pre
     *  - 0 <= count <= nn::account::UserCountMax
     *
     * @details
     *  JID の再発行や、BaaS 向け通知トークンの再登録など、通知に必要なリソース更新を一括して行います。
     *  オフラインでは使用できません。
     *
     *  トピックの購読情報はリフレッシュされません。
     */
    nn::Result RefreshResources(const nn::account::Uid uids[], int count) NN_NOEXCEPT;

    //! @}
}
}
